import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReferenceArray;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;

public class BoundedQueue<T> {
  ReentrantLock enqLock, deqLock;
  Condition notEmptyCondition, notFullCondition;
  final AtomicReferenceArray<T> elems;
  AtomicInteger size;
  volatile int head, tail;
  final int capacity;
  
  public BoundedQueue(int capacity) {
    this.capacity = capacity;
    elems = new AtomicReferenceArray<T>(capacity);
    size = new AtomicInteger(0);
    enqLock = new ReentrantLock();
    notFullCondition = enqLock.newCondition();
    deqLock = new ReentrantLock();
    notEmptyCondition = deqLock.newCondition();
  }

  public void enq(T x) throws InterruptedException {
    boolean mustWakeDequeuers = false;
    enqLock.lock();
    try {
      while (size.get() == capacity) {
        notFullCondition.await();
      }
      elems.set(tail, x);
      tail = (tail + 1) % capacity;
      if (size.getAndIncrement() == 0)
        mustWakeDequeuers = true;
    } finally { enqLock.unlock(); }
    if (mustWakeDequeuers) {
      deqLock.lock();
      try {
        notEmptyCondition.signalAll();
      } finally { deqLock.unlock(); }
    }
  }

  public T deq() throws InterruptedException {
    T result;
    boolean mustWakeEnqueuers = false;
    deqLock.lock();
    try {
      while (size.get() == 0) {
        notFullCondition.await();
      }
      result = elems.get(head);
      head = (head + 1) % capacity;
      if (size.getAndDecrement() == capacity)
        mustWakeEnqueuers = true;
    } finally { deqLock.unlock(); }
    if (mustWakeEnqueuers) {
      enqLock.lock();
      try {
        notFullCondition.signalAll();
      } finally { enqLock.unlock(); }
    }
    return result;
  }
}