# -*- coding: utf-8 -*-
import subprocess # To call NUPACK programs
import re # To parse results from NUPACK programs

class NupackOps(object):

  def __init__(self):
    self.nupack_bin = 'nupack3.0/bin'
    self.nupack_design_bin = 'design'
    self.nupack_pairs_bin = 'pairs'
    self.nupack_mfe_bin = 'mfe'
    self.nupack_energy_bin = 'energy'
    self.nupack_prob_bin = 'prob'


  def generate_pairs_data(self, temperature, design_filename):
    """ Runs the nupack program to generate the file with pairing probabilties
        for the different bases """
    output = subprocess.check_output([self.nupack_bin + '/' + self.nupack_pairs_bin,
                                    '-T', str(temperature),
                                    '-multi',
                                    '-material', 'dna',
                                    design_filename])


  def design_sequences(self, temperature, fstop, design_filename):
    """ Runs the nupack program to design the abstract sequence described in the
        design_file, returns the specific nucleotide sequence that should result
        with the appropriate binding """

    output = subprocess.check_output([self.nupack_bin + '/' + self.nupack_design_bin,
                                      '-T', str(temperature),
                                      '-pairs',
                                      '-fstop', str(fstop),
                                      '-material', 'dna',
                                      design_filename])

    lines = output.split("\n")
    seq_pattern = re.compile("^\s*Sequence\s*:\s*([ATGCU+]*)\s*$")
    for line in lines:
      m = seq_pattern.match(line)
      if not m is None:
        return m.group(1)

    return ""


  def mfe_for_temperature(self, temperature, design_filename, mfe_energies):
    """ Runs the nupack program to calculate the MFE for our design. Returns the
        resulting MFE structure, and updates the supplied mfe_energies hash with
        the free energy computed for that structure """

    mfe_structure = None
    output = subprocess.check_output([self.nupack_bin + '/' + self.nupack_mfe_bin,
                                      '-T', str(temperature),
                                      '-multi',
                                      '-material', 'dna',
                                      design_filename])

    mfe_file = open(design_filename + '.mfe', 'r')
    lines_after_comments = 0
    for line in mfe_file:
      stripped_line = line.strip()
      if not (stripped_line.startswith("%") or (len(stripped_line) == 0)):
        # Not a comment/empty line, we parse it
        lines_after_comments += 1
        if lines_after_comments == 2:
          # This is the energy
          mfe_energies[temperature] = float(stripped_line)
        if lines_after_comments == 3:
          # This is the structure of the MFE
          mfe_structure = stripped_line

    return mfe_structure


  def run_complex_prob_analysis(self, temperature, filename):
    """ Runs the nupack program to calculate the probability of a certain complex
        forming """
    output = subprocess.check_output([self.nupack_bin + '/' + self.nupack_prob_bin,
                                      '-T', str(temperature),
                                      '-multi',
                                      '-material', 'dna',
                                      filename])
    lines = output.split("\n")
    for line in lines:
      stripped_line = line.strip()
      if not (stripped_line.startswith("%") or (len(stripped_line) == 0)):
        return float(stripped_line)


  def run_energy_analysis(self, temperature, desired_design_filename):
    """ Runs the nupack program to calculate the free energy of a structure. """

    output = subprocess.check_output([self.nupack_bin + '/' + self.nupack_energy_bin,
                                        '-T', str(temperature),
                                        '-multi',
                                        '-material', 'dna',
                                        desired_design_filename])
    lines = output.split("\n")
    for line in lines:
      stripped_line = line.strip()
      if stripped_line.startswith("-"):
        return float(stripped_line)

# End of NupackOps class

