#
# Recognition problem simulation program 1
#
# Nimrod Priell
#

import numpy as np
import scipy as sp
import matplotlib as mpl
import matplotlib.pyplot as plt

#
# PARAMETERS
# ----------
#
# Target strand concentration, related to code concentration. 1.0 means there's
# 1 molecule of the clocked DNA program for every 1 molecule of target strand
target_c = 5.0
# A normally distributed error around target_c
target_c_err_stdev = 1e-1 * target_c

# Whether or not the target strand contains the 'A' strand we look for.
# TODO : make this an integer, how many non-overlapping A's match target
target_has_a = True

# How many tick strands are poured as a fraction of the concentration of code
# complexes
tick_strand_pour_amount = 1.0

# A normally distributed error around tick_strand_pour_amount
tick_strand_pour_amount_err_stdev = 1e-2

# How many tick strands actually react with the hanging anti-ticks for each
# instruction
tick_strand_hybridize_yield = 0.9

# A normally distributed error around how much of the released A strand binds to
# the target and how much to the gate
a_stoichiometry_err_stdev = 1e-5

# How many a strands when released, actually react with the target or gate
# strands
a_hybridization_yield = 0.9

# Whether the target also matches the tick strands
# TODO : make this an integer, how many non-overlapping ticks match target
target_has_tick = True

# What is the maximal concentration of a_bound_to_gate detectible by our
# flourescence measuring device
max_flourescence = 0.25

#
# INITIAL STATE VARIABLES
# -----------------------
#
# How many 'a' instructions were already released, relative to the amount of
# code concentration
a_free_in_solution = 0.0

# How many 'a' strands bound to the target, as a fraction of the concentration
# of code strands
a_bound_to_target = 0.0

# How many 'a' strands bound to the gate, as a fraction of the concentration of
# program complex
a_bound_to_gate = 0.0

# Flourescent signal as a fraction of the maximal flourescence detectable by our
# instrument
flourescence = 0.0

#
# SIMULATION PROGRESSION
#

def experiment(output = True):
  # Realization of the RV target_c
  real_target_c = target_c + (np.random.randn() * target_c_err_stdev)

  # Once we pour in the tick strand
  real_tick_strand_pour_amount = (tick_strand_pour_amount + (np.random.randn() * tick_strand_pour_amount_err_stdev))
  a_free_in_solution = real_tick_strand_pour_amount * tick_strand_hybridize_yield
  if target_has_a:
    x_label = "[y a z]"
  else: 
    x_label = "[x]"

  if output:
    print "Pouring {:.4} tick strand\t\t => [t]: {:.4} [A]: {:.4} {}: {:.4} [T | a]: {:.4}".format(tick_strand_pour_amount, real_tick_strand_pour_amount, a_free_in_solution, x_label, real_target_c, 1.0)

  # A strands bind according to stoichiometry
  if target_has_a:
    bind_to_target_stoich_factor = (real_target_c / (real_target_c + 1.0)) + (np.random.randn() * a_stoichiometry_err_stdev)
    a_bound_to_target = bind_to_target_stoich_factor * a_hybridization_yield * a_free_in_solution
    a_bound_to_gate = (1.0 - bind_to_target_stoich_factor) * a_hybridization_yield * a_free_in_solution
    a_free_in_solution = a_free_in_solution * (1.0 - a_hybridization_yield)
    if output:
      print "Cooling for hybridization\t => [t]: {:.4} [A]: {:.4} {}: {:.4} [y a^* z | A^*]: {:.4} \033[92m[T | a^* | A^*]\033[0m: {:.4} [T | a]: {:.4}".format(real_tick_strand_pour_amount, a_free_in_solution, x_label, real_target_c - a_bound_to_target, a_bound_to_target, a_bound_to_gate, 1.0 - a_bound_to_gate)

    flourescence = a_bound_to_gate / max_flourescence
  else:
    a_bound_to_gate = a_free_in_solution * a_hybridization_yield
    a_free_in_solution = a_free_in_solution * (1.0 - a_hybridization_yield)
    if output:
      print "Cooling for hybridization\t => [t]: {:.4} [A]: {:.4} {}: {:.4} \033[92m[T | a^* | A^*]\033[0m: {:.4} [T | a]: {:.4}".format(real_tick_strand_pour_amount, a_free_in_solution, x_label, real_target_c, a_bound_to_gate, 1.0 - a_bound_to_gate)

    flourescence = a_bound_to_gate / max_flourescence

  if output:
    print "measured flourescence ", flourescence

  return flourescence

experiment()

