package superply.math;

/*
 * Created on Nov 21, 2004
 */

import java.awt.Point;
import java.util.Vector;

import heurgame.PlayerToken;

/**
 * @author Chris Quackenbush
 *
 */
public abstract class HintChecker {
	private PlayerToken[] boardModel;
	HintChooser hintChooser;
	String hintStyle = "random";

	public Vector playOrder;
	public int winner = 2;
	
	/**
	 * @param hintType "random" or "deterministic"
	 */
	public HintChecker(String hintType) {
		if(hintType.equals("deterministic"))
			hintStyle = hintType;
	}

	/**
	 * @param who player who wants to move into this spot
	 * @param p p.x is the row we are moving to p.y is the column
	 * @return
	 */
	public boolean add(PlayerToken who, Point p) {
		int index = p.x*getBoardSize()+p.y;
		if(boardModel[index] != null){
			hintChooser.next();
			return false; //occupied
		}else if( isValidHint(p.x,p.y,hintChooser.counter) ){
			boardModel[index] = who;
			hintChooser.next();
			return true;
		}
		hintChooser.next();
		return false; //invalid for hint
	}

	/**
	 * resets the HintChecker
	 */
	public void clear() {
		winner = 2;
		hintChooser = new HintChooser(this);
		boardModel = new PlayerToken[getBoardSize()*getBoardSize()];
		for(int i=0; i <getBoardSize()*getBoardSize();i++)
			boardModel[i] = null;
	}

	/**
	 * @return the dimention of the board (it is square)
	 */
	public abstract int getBoardSize();

	/**
	 * @return the size of the cells on the board
	 */
	public abstract int getCellSize();

	/**
	 * @return the space seperated values along the top edge
	 */
	public abstract String getColumnLabels();
	
	/**
	 * @return the current hint
	 */
	public String getHint() {
		return hintChooser.get();
	}

	/**
	 * @return random or deterministic
	 */
	public String getHintMethod() {
		return hintStyle;
	}

	/**
	 * @return the hints for this game
	 */
	public abstract String[] getHints();

	/**
	 * @param r row
	 * @param c column
	 * @return the label for the cell (r,c)
	 */
	public abstract String getLabelAt(int r, int c);

	/**
	 * @param pt the PlayerToken of the player whose moves you want
	 * @return a vector of points corresponding to that players moves
	 */
	public Vector getMoves(PlayerToken pt) {
		Vector answer = new Vector();
		for(int i=0; i < getBoardSize()*getBoardSize();i++){
			if( boardModel[i] != null && boardModel[i].equals(pt) )
				answer.add(new Point(i/getBoardSize(),i%getBoardSize()));
		}
		return answer;
	}

	/**
	 * @return the name of this SuperPly Game
	 */
	public abstract String getOperation();

	/**
	 * @param r row
	 * @param c column
	 * @return the PlayerToken of the player who controls (r,c)
	 */
	public PlayerToken getPlayerAt(int r, int c) {
		return boardModel[r*getBoardSize()+c];
	}

	/**
	 * @return the space seperated values along the left edge
	 */
	public abstract String getRowLabels();

	/**
	 * @return true if there is a path belonging to p from LEFT to RIGHT starting at
	 * (row,col)
	 */
	private boolean searchLR(PlayerToken p, int row, int col, boolean[] marks) {
		marks[row*getBoardSize()+col] = true;
		if(!p.equals(boardModel[row*getBoardSize()+col]))
			return false;
		if(col == getBoardSize()-1)
			return true;
			
		int nr,nc;
		for(int dRow = -1; dRow < 2; dRow++)
		 for(int dCol = -1; dCol < 2; dCol++){
			nr = row+dRow; nc = col+dCol;
			if(nr >= 0 && nc >= 0 && nr < getBoardSize() && nc < getBoardSize() &&
			   !marks[nr*getBoardSize()+nc] && (nr != row || nc != col)){
				if(searchLR(p,nr,nc,marks))
					return true;	
			}
		 }	
		return false;
	}

	/**
	 * @return true if there is a path belonging to p from TOP to BOTTOM starting at
	 * (row,col)
	 */
	private boolean searchTB(PlayerToken p, int row, int col, boolean[] marks) {
		marks[row*getBoardSize()+col] = true;
		if(!p.equals(boardModel[row*getBoardSize()+col]))
			return false;
		if(row == getBoardSize()-1)
			return true;
			
		int nr,nc;
		for(int dRow = -1; dRow < 2; dRow++)
		 for(int dCol = -1; dCol < 2; dCol++){
		 	nr = row+dRow; nc = col+dCol;
			if(nr >= 0 && nc >= 0 && nr < getBoardSize() && nc < getBoardSize() &&
			   !marks[nr*getBoardSize()+nc] && (nr != row || nc != col)){
				if(searchTB(p,nr,nc,marks))
					return true;	
		 	}
		 }
		return false;
	}

	/**
	 * @param row the row of a cell
	 * @param col the column of a cell
	 * @param hint the number of the hint
	 * @return true if this cell is valid for hint constraint h
	 */
	public abstract boolean isValidHint(int row, int col, int hint);

	public PlayerToken winner(){
		if(playOrder.size() > 1){
			boolean[] tried = new boolean[getBoardSize()*getBoardSize()];
			PlayerToken p = (PlayerToken)playOrder.get(0);
			//try to get from left to right
			for(int i=0; i <getBoardSize()*getBoardSize();i++)
				tried[i] = false;
			for(int r = 0; r < getBoardSize(); r++)
				if(searchLR(p,r,0,tried)){
					winner = 0;
					return p;
				}
				
			p = (PlayerToken)playOrder.get(1);
			//try to get from top to bottom
			for(int i=0; i <getBoardSize()*getBoardSize();i++)
				tried[i] = false;
			for(int c = 0; c < getBoardSize(); c++)
				if(searchTB(p,0,c,tried)){
					winner = 1;
					return p;
				}
		}
		return null;
	}

}
