package nanomunchers.ui;

import java.awt.Canvas;
import java.awt.Color;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.Vector;

import nanomunchers.graph.Edge;
import nanomunchers.graph.Graph;
import nanomunchers.graph.GraphFactory;
import nanomunchers.graph.Node;

/*
 * Created on Oct 26, 2004
 */

/**
 * This is the panel the user interacts with when editing a graph.
 * 
 * @author David Kaplin
 */
public class GraphEditPanel extends Canvas{
    private Graph graph;
    private Graphics bufferGraphics;
	private Image offscreen;
	private boolean[] edgesShown;
	private Point[] midpoints;
	private int hoffset,woffset;
	private int lastWidth=-1;
	private int lastHeight = -1;
	private Vector chosenEdges = new Vector();
	private Point mLocation = null;
	private Drawing graphDrawing = new Drawing();
    /**
     * GraphEditPanels automatically repaint themselves.
     */
    public GraphEditPanel(){
        Thread paintThread = new Thread(){
            public void run(){
                while(true){
                try {
                    Thread.sleep(100);
                }catch (InterruptedException e){
                    
                }
                repaint();
                }
            }
        };
        paintThread.start();
        GridDragWatcher dragWatcher = new GridDragWatcher();
        addMouseListener(dragWatcher);
        addMouseMotionListener(dragWatcher);
    }
    /**
     * Instead of just loading the graph, this method
     * actually creates a new square graph of the same size
     * so a larger range of editing decisions can be made by the user.
     * 
     * @param inGraph of the edges to be turn on
     */
    public void loadGraph(Graph inGraph){
        graph = GraphFactory.buildGridGraph(inGraph.getWidth(),inGraph.getHeight());
        
        edgesShown = new boolean[graph.getUniqueEdges().length];
        midpoints = new Point[graph.getUniqueEdges().length];
        Node[] originalNodes = inGraph.getNodes();
        Edge[] originalEdges = inGraph.getUniqueEdges();
        Edge[] currentEdges = graph.getUniqueEdges();
        for(int i=0;i<edgesShown.length;i++){
            boolean showIt = false;
            for(int j=0;j<originalEdges.length;j++){
                if (originalEdges[j].equalsSpacially(currentEdges[i])){
                    showIt = true;
                    break;
                }
            }
            edgesShown[i]=showIt;
        }
        //now let
        graphDrawing.setGraph(graph);
    }
    
    /**
     * Used to get the product graph.
     * Only the edges that are listed as shown will be sent to
     * the destination graph.
     * 
     * @return MAY BE NULL
     */
    public Graph getGraph(){
        if (graph==null){
            return null;
        }
        Node[] nodes = graph.getNodes();
        Edge[] originalEdges = graph.getUniqueEdges();
        Vector edgeVector = new Vector();
        for(int i=0;i<edgesShown.length;i++){
            if (edgesShown[i]){
                edgeVector.add(originalEdges[i]);
            }
        }
        Edge[] finalEdges = new Edge[edgeVector.size()];
        for(int i=0;i<edgeVector.size();i++){
            finalEdges[i] = (Edge) edgeVector.get(i);
        }
        return GraphFactory.buildFromParts(nodes,finalEdges);
    }
    /**
     * Draws the graph and selected edges on the screen 
     * @see java.awt.Component#paint(java.awt.Graphics)
     */
    public void paint(Graphics g){
        update(g);
    }
    /**
     * Actually does the drawing 
     * @see java.awt.Component#update(java.awt.Graphics)
     */
    public void update(Graphics g){
        if (graph==null){
            g.drawString("No Graph Loaded",10,10);
            return;
        }
        int width = getWidth();
        int height = getHeight();
        if (offscreen == null || (width != lastWidth) || (height != lastHeight)) {
			offscreen = createImage(width, height);
			bufferGraphics = offscreen.getGraphics();
		}
        lastWidth = width;
        lastHeight = height;
        bufferGraphics.setColor(Color.BLACK); 
		bufferGraphics.fillRect(0, 0, width, height);
        
        //g.translate(0,-1);
        woffset = width/(graph.getWidth()+1);
        hoffset = height/(graph.getHeight()+1);
        bufferGraphics.setColor(Color.GRAY);
        graphDrawing.drawGraph(bufferGraphics,Color.GRAY,width,height);
        if (((width+1) > graph.getWidth()) && ((height+1) > graph.getHeight())){
            Edge[] edges = graph.getUniqueEdges();
            int sx,sy,dx,dy;
            for(int i=0;i<edges.length;i++){
                g.setColor(Color.LIGHT_GRAY);
                if (edgesShown[i]){
                    graphDrawing.highlightEdge(bufferGraphics,Color.LIGHT_GRAY,edges[i],4);
                 }
                sx = woffset + woffset*edges[i].getNodes()[0].getX();
                dx = woffset + woffset*edges[i].getNodes()[1].getX();
                sy = hoffset + hoffset*edges[i].getNodes()[0].getY();
                dy = hoffset + hoffset*edges[i].getNodes()[1].getY();
                if (chosenEdges.indexOf(new Integer(i))!= -1){
                    graphDrawing.drawEdge(bufferGraphics,Color.CYAN,edges[i],3);
                }
                    midpoints[i] = new Point((sx+dx)/2,(sy+dy)/2);
                
            }
        }
        if (mLocation != null){
            bufferGraphics.setColor(Color.WHITE);
            bufferGraphics.drawString("("+mLocation.x+","+mLocation.y+")",10,10);
        }
        //g.translate(0,1);
        g.drawImage(offscreen, 0, 0, this);
    }
    /**
     * Debugging method
     * @param args
     */
    public static void main(String args[]){
        final Frame f = new Frame("Graph Editor");
        f.setLayout(new java.awt.BorderLayout());
        GraphEditPanel gep = new GraphEditPanel();
        
        gep.setSize(500,350);
        f.add(gep);
        
        f.addWindowListener(new WindowAdapter(){
            public void windowClosing(WindowEvent e){
                f.dispose();
                System.exit(1);
            }
        });
        f.pack();
        f.show();
    }
    /**
     * Watches Mouse Movement to determine where drags are occuring. 
     *
     * @author David Kaplin
     *
     */
    class GridDragWatcher extends MouseAdapter implements MouseMotionListener{
        private boolean dragStarted = false;
        private int sx = -1;
        private int sy = -1;
        /**
         * Finishes any Drag 
         * @see java.awt.event.MouseListener#mouseReleased(java.awt.event.MouseEvent)
         */
        public void mouseReleased(MouseEvent e){
            
            for(int i=0;i<chosenEdges.size();i++){
                Integer place = (Integer)chosenEdges.get(i);
                edgesShown[place.intValue()] = !edgesShown[place.intValue()];
            }
            chosenEdges.clear();
        }

        /** 
         * Selects edges near the mouse and adds them to 
         * a set of internal edges.
         * @see java.awt.event.MouseMotionListener#mouseDragged(java.awt.event.MouseEvent)
         */
        public void mouseDragged(MouseEvent e) {
            int ex = e.getX();
            int ey = e.getY();
            sx = ex;
            sy = ey;
            int closest = -1;
            double bestdist = 10000;
            for(int i=0;i<midpoints.length;i++){
                Point p = midpoints[i];
                int dx = (ex> p.x) ? ex - p.x : p.x - ex;
                int dy = (ey> p.y) ? ey - p.y : p.y - ey;
                double dist = Math.sqrt(dx*dx +dy*dy);
                if (dist < bestdist){
                    closest = i;
                    bestdist = dist;
                }
            }
            Integer possible = new Integer(closest);
            if (bestdist < 10 && chosenEdges.indexOf(possible)==-1){
            //edgesShown[closest] = !edgesShown[closest];
                chosenEdges.add(possible);
            }
            //repaint();
            //edgesShown[x + y*graph.getWidth()] = !edgesShown[x + y*graph.getWidth()]; 
        }

        /** 
         * updates the mouse position
         * in the visualization
         * @see java.awt.event.MouseMotionListener#mouseMoved(java.awt.event.MouseEvent)
         */
        public void mouseMoved(MouseEvent e) {
            if (woffset==0 || hoffset==0) return;
            int ex = e.getX();
            int ey = e.getY();
            ex = (ex-woffset/2)/woffset;
            ey = (ey-hoffset/2)/hoffset;
            if (((ex >= 0) && (ex < getWidth())) &&
                    ((ey >= 0) && (ey < getHeight()))){
                mLocation = new Point(ex,ey);
            }
        }
    }
    
}
