import java.util.*;

/**
 * Jump Snatch Board
 */
public class Board {

	public int			   mnSize;           // 3, 4, or 5
   public boolean	[][]  mbArrCellEmpty;   // boolean representation of the board: true  <-- cell is empty
                                          //                                      false <-- cell is occupied by a stone
                                                   
   /**
    * constructor
    */	
	public Board(int pnSize) {
		
		mnSize         = pnSize;
		mbArrCellEmpty = new boolean[mnSize][mnSize];
      
		for (int i=0; i < mnSize; i++) {
         for (int j=0; j < mnSize; j++) {
			   mbArrCellEmpty[i][j] = false;
         }
		}            
	}

   /**
    * copy constructor
    */	
	public Board(Board board) {
		
		this.mnSize         = board.mnSize;
		this.mbArrCellEmpty = new boolean[this.mnSize][this.mnSize];
      
		for (int i=0; i < mnSize; i++) {
         for (int j=0; j < mnSize; j++) {
			   this.mbArrCellEmpty[i][j] = board.mbArrCellEmpty[i][j];
         }
		}            
	}

   /**
    * create a new copy of the board
    */	
   public Board copy() {

      Board newBoard = new Board(this);
      
      return newBoard;
   }

   /**
    * return this
    */	
   public Board getBoard() {
            
      return this;
   }

   /**
    * return this
    */	
   public boolean [][] getRawBoard() {
            
      return mbArrCellEmpty;
   }

   /**
    * check if a cell is empty
    */	
   public boolean isEmpty(int pnX, int pnY) {
      return(mbArrCellEmpty[pnX][pnY]);
   }

   public boolean isValidIndex(int pnX, int pnY) {
      if((pnX >= 0 && pnX < mnSize) &&
         (pnY >= 0 && pnY < mnSize)) {
         return(true);
      }
      else {
         return(false);
      }
   }

   public boolean isValidMoveIndex(Move move) {
      if(move != null) {
         return(isValidIndex(move.x, move.y));
      }
      else {
         return(false);
      }
   }

   public boolean isValidMoveTo(Move move) {
      if(isValidIndex(move.x, move.y) &&
         mbArrCellEmpty[move.x][move.y]) {
         return(true);
      }
      else {
         return(false);
      }
   }

   public boolean isBoardFull() {
      if(getNumOfStonesOnBoard() == mnSize * mnSize) {
         return(true);
      }
      else {
         return(false);
      }
   }

   /**
    * getNumOfStonesOnBoard
    */	
   public int getNumOfStonesOnBoard() {
      int lnStoneCtr = 0;

		for (int i=0; i < mnSize; i++) {
         for (int j=0; j < mnSize; j++) {
			   if(!mbArrCellEmpty[i][j]) {
               lnStoneCtr++;
            }
         }
		}            

      return(lnStoneCtr);
   }

   /**
    * edit the moves, return true if valid
    */	
   public String editMoves(MoveList moveList) {
      boolean        lbValid  = true;
      MoveIterator   iterator = moveList.getIterator();
      Move           move = null;
      Move           moveFrom = null;
      Move           moveTo = null;      
      MoveList       moveListFromDummy = null;
      MoveList       moveListToDummy = null;
      String         lcMsg = null;
      int            lnTotNumOfStones = mnSize * mnSize;

      //snatch
      if(getNumOfStonesOnBoard() == lnTotNumOfStones) {
         if(moveList.size() == 1) {
            move = iterator.next();
            if(!isValidIndex(move.x, move.y)) {
               lbValid = false;
               lcMsg = "Invalid index to snatch: " + move.toString();
            }
            else {
               move.type = Const.TYPE_SNATCH;
            }
         }
         else {
            lbValid = false;
            lcMsg = "Only one piece is allowed to snatch";
         }
      }
      else {
         if(moveList.size() == 1) {
            move = iterator.next();
            lbValid = false;
            lcMsg = "Ending position is required for move starting at: " + move.toString();
         }
         else if(moveList.size() == 2) {
            //jump or slide
            moveFrom = iterator.next();
            moveTo   = iterator.next();

            if(!isValidIndex(moveFrom.x, moveFrom.y)) {
               lbValid = false;
               lcMsg = "Invalid from-index to move: " + moveFrom.toString();
            }
            else if(!isValidIndex(moveTo.x, moveTo.y)) {
               lbValid = false;
               lcMsg = "Invalid to-index to move: " + moveTo.toString();
            }                  

            if(lbValid) {
               if(hasValidJumps()) {
                  //jump
                  if(isValidJump(moveFrom, moveTo)) {
                     moveFrom.type = Const.TYPE_JUMP;
                     moveTo.type   = Const.TYPE_JUMP;
                  }
                  else {
                     lbValid = false;
                     lcMsg = "Invalid jump starting at: " + moveFrom.toString() +
                                           " ending at: " + moveTo.toString();
                  }               
               }
               else {
                  //slide
                  boolean lbPossible = IsSlidingToCenterPossible(moveFrom);
                  if(isValidSlide(moveFrom, moveTo, lbPossible)) {
                     moveFrom.type = Const.TYPE_SLIDE;
                     moveTo.type   = Const.TYPE_SLIDE;
                  }
                  else {
                     lbValid = false;
                     lcMsg = "Invalid slide starting at: " + moveFrom.toString() +
                                            " ending at: " + moveTo.toString();
                  }
               }            
            }
         }
         else {
            //multiple jumps
            moveFrom = iterator.next();
            if(!isValidIndex(moveFrom.x, moveFrom.y)) {
               lbValid = false;
               lcMsg = "Invalid from-index to jump: " + moveFrom.toString();
            }

            if(lbValid) {
               while (iterator.hasNext()) {
                  moveTo = iterator.next();

                  if(!isValidIndex(moveTo.x, moveTo.y)) {
                     lbValid = false;
                     lcMsg = "Invalid to-index to jump: " + moveTo.toString();
                     break;
                  }                  

                  //jump
                  if(isValidJump(moveFrom, moveTo)) {
                     moveFrom.type = Const.TYPE_JUMP;
                     moveTo.type   = Const.TYPE_JUMP;
                  }
                  else {
                     lbValid = false;
                     lcMsg = "Invalid jump starting at: " + moveFrom.toString() +
                                           " ending at: " + moveTo.toString();
                     break;
                  }               
               
                  moveFrom = moveTo;
               }            
            }
         }
      }

      //debug if(!lbValid) {
      //debug    System.out.println(lcMsg);
      //debug }
      
      return(lcMsg);
   }

   /**
    * play the moves
    * moves should be edited before invoking this method.
    */	
   public void executeMoves(MoveList moveList) {
      MoveIterator   iterator = moveList.getIterator();
      Move           move = null;
      Move           moveFrom = null;
      Move           moveTo = null;      
      int            lnTotNumOfStones = mnSize * mnSize;

      //snatch
      if(getNumOfStonesOnBoard() == lnTotNumOfStones) {
         if(moveList.size() == 1) {
            move = iterator.next();
            mbArrCellEmpty[move.x][move.y] = true;
         }
      }
      else {
         if(moveList.size() == 2) {
            //jump or slide
            moveFrom = iterator.next();
            moveTo   = iterator.next();
            
            if(isValidJump(moveFrom, moveTo)) {
               //jump
               jump(moveFrom, moveTo);
            }
            else {
               //slide
               slide(moveFrom, moveTo);
            }               
         }
         else {
            //multiple jumps
            moveFrom = iterator.next();
            while (iterator.hasNext()) {
               moveTo = iterator.next();

               //jump
               jump(moveFrom, moveTo);
                              
               moveFrom = moveTo;
            }            
         }
      }
      
      return;
   }

   /**
    * jump
    */	
   public void jump(Move moveFrom, Move moveTo) {
      int lnX = (moveFrom.x + moveTo.x)/2; 
      int lnY = (moveFrom.y + moveTo.y)/2;

      mbArrCellEmpty[moveFrom.x][moveFrom.y] = true;
      mbArrCellEmpty[lnX]       [lnY]        = true;
      mbArrCellEmpty[moveTo.x][moveTo.y]     = false;

      return;
   }

   /**
    * slide
    */	
   public void slide(Move moveFrom, Move moveTo) {

      mbArrCellEmpty[moveFrom.x][moveFrom.y] = true;
      mbArrCellEmpty[moveTo.x]  [moveTo.y]   = false;

      return;
   }

   /**
    * return true if the board has any valid jumps
    */	
   public boolean hasValidJumps() {
      boolean lbJump = false;
      int i = 0;
      int j = 0;
      int m = 0;
      int n = 0;      

      //loop thru each non-empty cell
		for (i=0; i < mnSize; i++) {
         for (j=0; j < mnSize; j++) {         
			   if(!mbArrCellEmpty[i][j]) {
               //beg inner loop
		         for (m=0; m < mnSize; m++) {
                  for (n=0; n < mnSize; n++) {
			            if(mbArrCellEmpty[m][n]) {
                        if(isValidJump(new Move(i, j), new Move(m, n))) {
                           lbJump = true;
                           break;
                        }
                     }
                  }
                  if(lbJump) {
                     break;
                  }
		         }
               //end inner loop
            }
            if(lbJump) {
               break;
            }
         }
         if(lbJump) {
            break;
         }
		}

      return(lbJump);
   }


   /**
    * find valid slides
    */	
   public boolean findValidSlides(MoveList moveListSlideFrom, MoveList moveListSlideTo) {
      boolean lbSlide = false;
      boolean lbSlideToCenterPossible = false;

      int i = 0;
      int j = 0;

      Move  moveFrom  = new Move(-1, -1);
      Move  moveL  = new Move(-1, -1);
      Move  moveR  = new Move(-1, -1);
      Move  moveTL = new Move(-1, -1);
      Move  moveTM = new Move(-1, -1);
      Move  moveTR = new Move(-1, -1);
      Move  moveBL = new Move(-1, -1);
      Move  moveBM = new Move(-1, -1);
      Move  moveBR = new Move(-1, -1);

		for (i=0; i < mnSize; i++) {
         for (j=0; j < mnSize; j++) {
			   if(!mbArrCellEmpty[i][j]) {

               moveFrom.set(i, j);
               moveL.set (i,   j-1);
               moveR.set (i,   j+1);
               moveTL.set(i-1, j-1);
               moveTM.set(i-1, j  );
               moveTR.set(i-1, j+1);
               moveBL.set(i+1, j-1);
               moveBM.set(i+1, j  );
               moveBR.set(i+1, j+1);

               lbSlideToCenterPossible = IsSlidingToCenterPossible(moveFrom);

               if(isValidSlide(moveFrom, moveL, lbSlideToCenterPossible)) {
                  moveListSlideFrom.add(moveFrom.copy());
                  moveListSlideTo.add(moveL.copy());                  
               }
               if(isValidSlide(moveFrom, moveR, lbSlideToCenterPossible)) {
                  moveListSlideFrom.add(moveFrom.copy());
                  moveListSlideTo.add(moveR.copy());                  
               }
               if(isValidSlide(moveFrom, moveTL, lbSlideToCenterPossible)) {
                  moveListSlideFrom.add(moveFrom.copy());
                  moveListSlideTo.add(moveTL.copy());                  
               }
               if(isValidSlide(moveFrom, moveTM, lbSlideToCenterPossible)) {
                  moveListSlideFrom.add(moveFrom.copy());
                  moveListSlideTo.add(moveTM.copy());                  
               }
               if(isValidSlide(moveFrom, moveTR, lbSlideToCenterPossible)) {
                  moveListSlideFrom.add(moveFrom.copy());
                  moveListSlideTo.add(moveTR.copy());                  
               }
               if(isValidSlide(moveFrom, moveBL, lbSlideToCenterPossible)) {
                  moveListSlideFrom.add(moveFrom.copy());
                  moveListSlideTo.add(moveBL.copy());                  
               }
               if(isValidSlide(moveFrom, moveBM, lbSlideToCenterPossible)) {
                  moveListSlideFrom.add(moveFrom.copy());
                  moveListSlideTo.add(moveBM.copy());                  
               }
               if(isValidSlide(moveFrom, moveBR, lbSlideToCenterPossible)) {
                  moveListSlideFrom.add(moveFrom.copy());
                  moveListSlideTo.add(moveBR.copy());                  
               }                           
            }
         }
		}            

      if(moveListSlideFrom.size() > 0) {
         lbSlide = true;
      }

      return(lbSlide);
   }

   /**
    * return true if sliding toward the center is possible
    */	
   public boolean IsSlidingToCenterPossible(Move moveFrom) {
      boolean lbPossible = false;
      int i = moveFrom.x;
      int j = moveFrom.y;
      int lnCtr = 0;

      Move  moveL  = new Move(i,   j-1);
      Move  moveR  = new Move(i,   j+1);
      Move  moveTL = new Move(i-1, j-1);
      Move  moveTM = new Move(i-1, j  );
      Move  moveTR = new Move(i-1, j+1);
      Move  moveBL = new Move(i+1, j-1);
      Move  moveBM = new Move(i+1, j  );
      Move  moveBR = new Move(i+1, j+1);

      double ldDist = distToCenter(moveFrom);

      if(isValidMoveTo(moveL)) {
         if(distToCenter(moveL) < ldDist) {
             lnCtr++;
         }             
      }
      if(isValidMoveTo(moveR)) {
         if(distToCenter(moveR) < ldDist) {
             lnCtr++;
         }             
      }
      if(isValidMoveTo(moveTL)) {
         if(distToCenter(moveTL) < ldDist) {
             lnCtr++;
         }             
      }
      if(isValidMoveTo(moveTM)) {
         if(distToCenter(moveTM) < ldDist) {
             lnCtr++;
         }             
      }
      if(isValidMoveTo(moveTR)) {
         if(distToCenter(moveTR) < ldDist) {
             lnCtr++;
         }             
      }
      if(isValidMoveTo(moveBL)) {
         if(distToCenter(moveBL) < ldDist) {
             lnCtr++;
         }             
      }
      if(isValidMoveTo(moveBM)) {
         if(distToCenter(moveBM) < ldDist) {
             lnCtr++;
         }             
      }
      if(isValidMoveTo(moveBR)) {
         if(distToCenter(moveBR) < ldDist) {
             lnCtr++;
         }             
      }

      if(lnCtr > 0) {
         lbPossible = true;
      }

      return(lbPossible);
   }

   /**
    * check for next valid jump
    */	
   public boolean isValidJump(Move moveFrom, Move moveTo) {
      boolean lbValid = false;
      int diffX = Math.abs(moveFrom.x - moveTo.x);
      int diffY = Math.abs(moveFrom.y - moveTo.y);
      int lnX = (moveFrom.x + moveTo.x)/2; 
      int lnY = (moveFrom.y + moveTo.y)/2;

      if( (diffX == 0 && diffY ==2) ||
          (diffX == 2 && diffY ==0) ||
          (diffX == 2 && diffY ==2) ) {
          lbValid = true;
      }
      else {
         lbValid = false;
      }

      if(lbValid) {
         if(!mbArrCellEmpty[moveFrom.x][moveFrom.y] &&   //not empty
            !mbArrCellEmpty[lnX]       [lnY]        &&   //not empty
             mbArrCellEmpty[moveTo.x][moveTo.y]) {       //empty
            lbValid = true;
         }             
         else {
            lbValid = false;
         }
      }

      return(lbValid);
   }

   /**
    * check for next valid slide
    */	
   public boolean isValidSlide(Move moveFrom, Move moveTo, boolean pbSlideToCenterPossible) {
      boolean lbValid = false;
      int diffX = Math.abs(moveFrom.x - moveTo.x);
      int diffY = Math.abs(moveFrom.y - moveTo.y);      

      if(isValidMoveIndex(moveFrom) &&
         isValidMoveIndex(moveTo)) {
         lbValid = true;
      }
      else {
         lbValid = false;
      }

      if(lbValid) {
         if( (diffX == 0 && diffY ==1) ||
             (diffX == 1 && diffY ==0) ||
             (diffX == 1 && diffY ==1) ) {
             lbValid = true;
         }
         else {
            lbValid = false;
         }
      }

      if(lbValid) {
         if(!mbArrCellEmpty[moveFrom.x][moveFrom.y] &&   //not empty
             mbArrCellEmpty[moveTo.x][moveTo.y]) {       //empty
             if(isCenterPiece(moveFrom)) {
               lbValid = false;
             }
             else {
               lbValid = true;
             }
         }             
         else {
            lbValid = false;
         }
      }

      //check for closer to center
      if(lbValid) {
         if(distToCenter(moveTo) < distToCenter(moveFrom)) {
             lbValid = true;
         }             
         else {
            if(!pbSlideToCenterPossible) {
               lbValid = true;
            }
            else {
               lbValid = false;
            }
         }
      }
      
      return(lbValid);
   }

   /**
    * distToCenter
    */	
   public double distToCenter(Move move) {
      double dist = 0.0;
      double ldX = 1.0 * (mnSize - 1) / 2;
      double ldY = ldX;

      dist = Math.sqrt(Math.pow((move.x -ldX), 2.0) + Math.pow((move.y -ldY), 2.0));      
      
      return(dist);
   }

   /**
    * isCenterPiece
    */	
   public boolean isCenterPiece(Move move) {
      boolean lbCenter = false;
      
      if((mnSize % 2) == 1) {
         if( (move.x == (mnSize - 1) / 2) &&
             (move.y == (mnSize - 1) / 2) ) {
             
             if(!mbArrCellEmpty[move.x][move.y]) {
               lbCenter = true;
             }
         }
      }
      
      return(lbCenter);
   }
   
}
