import { rgbToHex } from '@material-ui/core';
import solver from 'javascript-lp-solver';

let foodData = require("./foods.json");



function getRandom(arr, n) {
    var result = new Array(n),
        len = arr.length,
        taken = new Array(len);
    if (n > len)
        throw new RangeError("more elements taken than available");
    while (n--) {
        var x = Math.floor(Math.random() * len);
        result[n] = arr[x in taken ? taken[x] : x];
        taken[x] = --len in taken ? taken[len] : len;
    }
    return result;
}

function generateSample(recipes) {
    //Our total dataset is too large to solve on. We have ~26k recipes. Picking 100 makes the problem much quicker to solve.
    return new Promise((resolve, reject) => {
        //console.log("Generating sample from", recipes)
        let foods = {};
        let intVars = {};
        let servings = {};

        let shuffled = getRandom(recipes, 100)

        //Generate neccessary components for the model.
        for (const item of shuffled) {
            foods[item] = foodData[item];
            intVars[item] = 1;
            servings[item] = { "max": 1.05, }; //no  "priority": 0 
        }

        //Async
        resolve({ foods, intVars, servings })
    });

}

export function calculate(recipes, goals, numMeals, index, halfServings = false) {
    numMeals = numMeals ?? 3;
    return new Promise((resolve, reject) => {
        //Each worker generates a sample of size 100 and solves on that sample. Recursively call until we get valid results (the solver is not the best).
        generateSample(recipes).then((data) => {
            let ans = solver.Solve({

                "opType": "min",
                "optimize": "quantity",
                "constraints": {
                    ...goals,
                    //"quantity": { "max": 5, "min": 3 }, -> We tried this, it does not work. You need to check afterwards.
                    ...data.servings
                },
                "variables": data.foods,
                "ints": data.intVars,
                "options": {
                    "tolerance": 0.05,
                    "keep_solutions": true
                }

            });
            //console.log(model)
            const { isIntegral, bounded, result, feasible, ...solution } = ans;
            console.log("Solver:", {

                "opType": "min",
                "optimize": "quantity",
                "constraints": {
                    ...goals,
                    //"quantity": { "max": 5, "min": 3 }, -> We tried this, it does not work. You need to check afterwards.
                    ...data.servings
                },
                "variables": data.foods,
                "ints": data.intVars,
                "options": {
                    "tolerance": 0.05,
                    "keep_solutions": true
                }

            }, ans) //.model.solutions
            let totalCals = 0;

            //This ensures we get a minumum of three meals per day (the solver is unreliable when you imput this constraint)
            if (Object.keys(solution).length < numMeals) {
                console.log("Rejecting solution because there was less than ", numMeals, " meals");
                resolve(calculate(recipes, goals, numMeals, index));
            }
            else {
                for (const key in solution) {
                    //If we get more than one or less than one solution
                    if (solution[key] > 1 || solution[key] < 1) {
                        console.log("The solution isn't valid. Need to regenerate after")
                        resolve(calculate(recipes, goals, numMeals, index));
                    }
                    else {
                        solution[key] = data.foods[key];
                        totalCals += solution[key].calories;
                        //In general, calories is a good measure for whether a solution is valid. Resolve if overboard.
                        if (totalCals > goals.calories[1]) {
                            console.log("The solution over did the calories")
                            resolve(calculate(recipes, goals, numMeals, index));
                        }
                        /* 
                        Fix this on the db side...
    
                        fetch("http://www.whateverorigin.org/get?url=" + solution[key].src, ).then(function(response) {
                            if (response.status != 200 || response.status != 301) {
                                // make the promise be rejected if we didn't get a 2xx response
                                console.log(response)
                                throw new Error("Not 2xx response")
                            } else {
                                 console.log("url valid")
                            }
                        }).catch(function(err) {
    
                           console.log(err);
                        }); 
                        
                        */

                    }
                }

                console.log("Worker solved:", solution)
                resolve(solution);
            }
        })
    })

}

export function groupCalculator(recipes, outlier, numMeals, index) {
    console.log(outlier)
    return new Promise((resolve, reject) => {
        generateSample(recipes).then((data) => {
            let ans = solver.Solve({

                "opType": "min",
                "optimize": "quantity",
                "constraints": {

                    calories: {
                        min: outlier.goals.calories[0],
                        max: outlier.goals.calories[1],
                    }, //"priority": 0
                    protein: {
                        min: outlier.goals.protein[0],
                        max: outlier.goals.protein[1],
                    }, //, "priority": 1
                    carbohydrates: {
                        min: outlier.goals.carbohydrates[0],
                        max: outlier.goals.carbohydrates[1],
                    }, //, "priority": 2
                    fat: {
                        min: outlier.goals.fat[0],
                        max: outlier.goals.fat[1],
                    }, //, "priority": 1
                    seconds: {
                        min: outlier.time[0],
                        max: outlier.time[1],
                    },

                    //"quantity": { "max": 5, "min": 3 }, -> We tried this, it does not work. You need to check afterwards.
                    ...data.servings
                },
                "variables": data.foods,
                "ints": data.intVars,
                "options": {
                    "tolerance": 0.05,
                    "keep_solutions": true
                }

            });
            //console.log(model)
            const { isIntegral, bounded, result, feasible, ...solution } = ans;
            console.log("Solver:", {

                "opType": "min",
                "optimize": "quantity",
                "constraints": {


                    calories: {
                        min: outlier.goals.calories[0],
                        max: outlier.goals.calories[1],
                    }, //"priority": 0
                    protein: {
                        min: outlier.goals.protein[0],
                        max: outlier.goals.protein[1],
                    }, //, "priority": 1
                    carbohydrates: {
                        min: outlier.goals.carbohydrates[0],
                        max: outlier.goals.carbohydrates[1],
                    }, //, "priority": 2
                    fat: {
                        min: outlier.goals.fat[0],
                        max: outlier.goals.fat[1],
                    }, //, "priority": 1
                    seconds: {
                        min: outlier.time[0],
                        max: outlier.time[1],
                    },
                    //"quantity": { "max": 5, "min": 3 }, -> We tried this, it does not work. You need to check afterwards.
                    ...data.servings
                },
                "variables": data.foods,
                "ints": data.intVars,
                "options": {
                    "tolerance": 0.05,
                    "keep_solutions": true
                }

            }, ans) //.model.solutions
            let totalCals = 0;
            console.log("Solution found:", solution)
            //This ensures we get a minumum of two meals per day (the solver is unreliable when you imput this constraint)
            if (Object.keys(solution).length < numMeals || Object.keys(solution).length > numMeals) {
                console.log("Rejecting solution because it did not equal ", numMeals, " meals");
                resolve(groupCalculator(recipes, outlier, numMeals, index));
            }
            else {
                for (const key in solution) {
                    //If we get more than one or less than one solution
                    if (solution[key] > 1 || solution[key] < 1) {
                        console.log("The solution isn't valid. Need to regenerate after")
                        resolve(groupCalculator(recipes, outlier, numMeals, index));
                    }
                    else {
                        solution[key] = data.foods[key];
                        totalCals += solution[key].calories;
                        //In general, calories is a good measure for whether a solution is valid. Resolve if overboard.
                        if (totalCals > outlier.goals.calories[1]) {
                            console.log("The solution over did the calories")
                            resolve(groupCalculator(recipes, outlier, numMeals, index));
                        }

                    }
                }

                console.log("Worker solved:", solution)
                resolve(solution);
            }
        })

    })
}

export function calculatePortions(recipes, outlier, index) {
    console.log(recipes, outlier)
    return new Promise((resolve, reject) => {
        let foods = {}
        let intVars = {}
        let servings = {}

        for (const item of recipes) {
            let key = item.toString() // "food-" +
            foods[key] = foodData[item];
            intVars[key] = 1;
            servings[key] = { "min": 1, "max": 2.05, }; //no  "priority": 0 
        }

        let ans = solver.Solve({

            "opType": "min",
            "optimize": "quantity",
            "options": {
                "timeout": 10000,
                "keep_solutions": true,
                "tolerance": 0.2
            },
            "constraints": {

                calories: {
                    min: 1 * outlier.goals.calories[0],
                    max: 1 * outlier.goals.calories[1],
                }, //"priority": 0
                protein: {
                    min: 1 * outlier.goals.protein[0],
                    max: 1 * outlier.goals.protein[1],
                }, //, "priority": 1
                carbohydrates: {
                    min: 1 * outlier.goals.carbohydrates[0],
                    max: 1 * outlier.goals.carbohydrates[1],
                }, //, "priority": 2
                fat: {
                    min: 1 * outlier.goals.fat[0],
                    max: 1 * outlier.goals.fat[1],
                }, //, "priority": 1
                seconds: {
                    min: 1 * outlier.time[0],
                    max: 1 * outlier.time[1],
                },

                //"quantity": { "max": 5, "min": 3 }, -> We tried this, it does not work. You need to check afterwards.
                ...servings
            },
            "variables": foods,
            "ints": intVars,

        });
        //console.log(model)
        const { isIntegral, bounded, result, feasible, ...solution } = ans;
        console.log("Solver:", {

            "opType": "min",
            "optimize": "quantity",
            "constraints": {


                calories: {
                    min: outlier.goals.calories[0],
                    max: outlier.goals.calories[1],
                }, //"priority": 0
                protein: {
                    min: outlier.goals.protein[0],
                    max: outlier.goals.protein[1],
                }, //, "priority": 1
                carbohydrates: {
                    min: outlier.goals.carbohydrates[0],
                    max: outlier.goals.carbohydrates[1],
                }, //, "priority": 2
                fat: {
                    min: outlier.goals.fat[0],
                    max: outlier.goals.fat[1],
                }, //, "priority": 1
                seconds: {
                    min: outlier.time[0],
                    max: outlier.time[1],
                },
                //"quantity": { "max": 5, "min": 3 }, -> We tried this, it does not work. You need to check afterwards.
                ...servings
            },
            "variables": foods,
            "ints": intVars,

        }, ans) //.model.solutions
        let totalCals = 0;
        console.log("Solution found:", solution)
        //This ensures we get a minumum of two meals per day (the solver is unreliable when you imput this constraint)
        if (Object.keys(solution).length < 2 || Object.keys(solution).length > 4) {
            console.log("Rejecting solution because it did not fit between 2 and 4 meals");
            resolve(null);
        }
        else {
            for (const key in solution) {
                //If we get more than one or less than one solution
                solution[key] = (Math.round(solution[key] * 4) / 4).toFixed(2);
                if (solution[key] < 0 || solution[key] > 2) {
                    resolve(null);
                }
                else {
                    totalCals += solution[key].calories;
                    //In general, calories is a good measure for whether a solution is valid. Resolve if overboard.
                    if (totalCals > outlier.goals.calories[1]) {
                        console.log("The solution over did the calories")
                        resolve(calculatePortions(recipes, outlier, index));
                    }
                }

            }
            console.log("Worker solved:", solution)
            resolve({ [outlier.user]: solution });
        }
    })

}

export function calculateGroup(recipes, outliers, time, index) {
    let numMeals = 3
    return new Promise((resolve, reject) => {
        //Each worker generates a sample of size 100 and solves on that sample. Recursively call until we get valid results (the solver is not the best).
        Promise.all(outliers.map((outlier) => groupCalculator(recipes, { ...outlier, time: [time.min, time.max] }, numMeals, index))).then((solutions) => {
            let meals = solutions.map((element) => Object.keys(element)).flat(1)

            const first = Math.floor(Math.random() * meals.length);
            meals.splice(first, 1);
            const second = Math.floor(Math.random() * meals.length);
            meals.splice(second, 1);

            resolve([...new Set(meals)]);
        })
        //            solutions.push(Object.keys(solution))

    })
}

