
#include "SAT.h"
#include "SATNode.h"
#include <stdio.h>
#include <stdlib.h>
#include <float.h>
#include <math.h>
#include <fstream>
#include <time.h>
#include <assert.h>

#define BUF_LEN 4096
//#define TIMING


SAT::SAT() : _topWinSize(1), _maxWinSize(1), _dataLen(-1), _bOutput(true), 
			_threshSize(0), _aggBuf(NULL)
{
}

SAT::~SAT()
{
	delete [] _aggBuf;
	for (int i = 0; i < _layerBank.size(); ++ i)
		delete _layerBank[i];
}


int SAT::loadThresholds(char* thresFile)
{
	ifstream ifs(thresFile);
	if (!ifs)
	{
		printf("can't open %s\n", thresFile);
		assert( false );
		return -1;
	}

//	printf("read %s\n", thresFile);

	char buf[BUF_LEN];
	int winSize, lastWinSize=0;
	double thresh, lastThresh=0.0;
	while (!ifs.eof())
	{
		ifs.getline(buf, 512);
		if (strlen(buf) == 0)
			break;
		char* space = strchr(buf, '\t');
		thresh = atof(space+1);
		*space = 0;
		winSize = atoi(buf);
		if (winSize <= lastWinSize || thresh <= lastThresh)
		{
			printf("need increasing size and thresholds\n");
			return -1;
		}
		lastWinSize = winSize;
		lastThresh = thresh;

		_thresholds.push_back( pair<int,double>(winSize, thresh) );
		if (winSize > _maxWinSize)
			_maxWinSize = winSize;
	}
	
	ifs.close();

	_threshSize = _thresholds.size();

	if (_maxWinSize > _sat[_numLayer-1]->maxCoveredSize())
	{
		printf("SAT tree can't cover the max window size\n");
		assert( false );
		return -1;
	}

	int l, h;
	for (int i = 0; i <= MAX_SHIFTMODE2; ++ i)
	{
		l = 0;
		h = 0;
		while (h < _threshSize && l < _sat.size())
		{
			while (_thresholds[h].first <= _sat[l]->_h-_sat[l]->_s*(1<<i)/(1<<MAX_SHIFTMODE)+1 && h < _threshSize)
				++ h;
			if (h < _threshSize)
				_sat[l++]->_threshHigh[i] = h-1;
		}
		if (l < _sat.size())
			_sat[l]->_threshHigh[i] = _threshSize-1;
		for (; l < _numLayer; ++ l)
			_sat[l]->_threshHigh[i] = _threshSize-1;
	}

	for (l = 0; l < _sat.size(); ++ l)
		_sat[l]->_threshHighIdx = _sat[l]->_threshHigh[MAX_SHIFTMODE];

	return 0;
}

void SAT::readStruct(char* satFname, char* thresFname)
{
//	printf("read %s\n", satFname);

	ifstream ifs(satFname);
	if (!ifs)
	{
		printf("can't open %s\n", satFname);
		assert( false );
		return;
	}

	char head[256], *buf;
	int bufLen;

	ifs.getline(head, 256);
	sscanf(head, "%d %d", &_numLayer, &_topWinSize);
	bufLen = _topWinSize * 4;
	buf = new char[bufLen];

	int h, s, i, j;
	float ap = (float)0.1;
	for (i = 0; i < _numLayer; ++ i)
	{
		SATLayer* layer = new SATLayer();
		
		ifs.getline(buf, bufLen);
		sscanf(buf, "%d %d", &s, &h);
		layer->_h = h;
		layer->_s = s;
		if (s == 1)
		{
			layer->_shiftMode = 0;
			layer->_shiftStep = 1;
		}
		layer->_active = 1;

		layer->_aggLen = ((_topWinSize+s-1)/s) << MAX_SHIFTMODE2;
		layer->_aggregate = new double[layer->_aggLen];
		layer->_curIdx = 0;
		for (j = 0; j < layer->_aggLen; ++ j)
			layer->_aggregate[j] = 0.0;

		_sat.push_back(layer);
		_layerBank.push_back(layer);

		// hstruct
		char* space = strchr(buf, ' ');
		space = strchr(space+1, ' ');
		++ space;

		int *hstruct = new int[_topWinSize+1];
		int sum = 0;
		int count = 0;
		while (sscanf(space, "%d", &hstruct[count]) > 0)
		{
			assert( hstruct[count] < h || h == 1 );
			sum += hstruct[count];
			++ count;
			space = strchr(space, ' ');
			if (NULL == space)
				break;
			++ space;
		}
		hstruct[count] = 0;
//		assert( sum == layer->_h );
		if (sum != layer->_h)
			printf("incorrect sum %d != %d\n", sum, layer->_h);

		for (j = 0; j < count; ++ j)
		{
			layer->_childLayers.push_back(_sat[0]);
			layer->_sIdx.push_back(0);
		}

		for (j = count-1; j >= 0; -- j)
		{
			// lIdx
			int h = hstruct[j];
			int shiftMode = 0;
			int l;
			for (l = i-1; l >= 0; -- l)
				if (_sat[l]->_h == h)
				{
					layer->_childLayers[j] = _sat[l];
					layer->_unqChildLayers.insert(_sat[l]);
					_sat[l]->_dependLayers[layer] = -1;
					shiftMode = _sat[l]->_shiftMode;
					break;
				}
			if (l < 0 && h != 1)
				printf("error: no %d in the lower layer\n", h);

			// sIdx
			layer->_sIdx[j] = (l>=0) ? hstruct[j+1]/_sat[l]->_s : 0;
			if (l >= 0)
				layer->_sIdx[j] = layer->_sIdx[j] << _sat[l]->_shiftMode;

			hstruct[j] += hstruct[j+1];
		}

		delete [] hstruct;

		layer->_numActiveChildLayers = layer->_unqChildLayers.size();
		set<SATLayer*>::iterator sit;
		for (sit = layer->_unqChildLayers.begin(); sit != layer->_unqChildLayers.end(); ++ sit)
			++ ((*sit)->_numActiveDependLayers);
	}

	ifs.close();

	// read threshold
	int ret = loadThresholds(thresFname);

	// alloc extra layers
	addExtraLayers();

	// alloc buffer
	_aggBuf = new double[_topWinSize+1];
	for (i = 0; i < _topWinSize+1; ++ i)
		_aggBuf[i] = 0.0;

/*	printf("\nread SAT: %d\n", _numLayer);
	for (i = 0; i < _numLayer; ++ i)
	{
		SATLayer* layer = _sat[i];
		printf("%d %d %d %d ", 
					layer->_h, layer->_s, layer->_threshHighIdx, layer->_childLayers.size());
		for (int j = 0; j < layer->_childLayers.size(); ++ j)
			printf("%d/%d ", layer->_childLayers[j]->_h, layer->_sIdx[j]);
		printf("\n");
	}
	printf("\n");

	printf("density = %f\n", density());
*/
}

string getNextToken(const char **cur, char sep)
{
	char buf[BUF_LEN];
	char* next = strchr(*cur, sep);
	if (next == NULL)
		return "";
	strncpy(buf, *cur, next-(*cur));
	buf[next-(*cur)] = 0;
	*cur = next + 1;
	return string(buf);
}


double SAT::density() const
{
	int numNode = 0;
	int i;
	for (i = 0; i < _numLayer; ++ i)
		numNode += _sat[_numLayer-1]->_s/_sat[i]->_s;
	return numNode/(double)(_maxWinSize*_sat[_numLayer-1]->_s);
}

double SAT::alarmProb() const
{
	double ap = 0.0;
	int i;
	ap += _sat[0]->_alarmCount;	
	for (i = 1; i < _numLayer-1; ++ i)
	{
		ap += _sat[i]->_alarmCount * (_sat[i]->maxCoveredSize()-_sat[i-1]->maxCoveredSize());
	}
	ap += _sat[i]->_alarmCount * (_maxWinSize-_sat[i-1]->maxCoveredSize()+1);
	return ap/_maxWinSize;
}


int SAT::hasBurst(double agg, int lowIdx, int highIdx)
{
	if (lowIdx > highIdx)
		return -1;
	if (agg - _thresholds[lowIdx].second < -DBL_EPSILON)
		return -1;
	if (agg - _thresholds[highIdx].second > -DBL_EPSILON)
		return highIdx;

	while (lowIdx <= highIdx)
	{
		int midIdx = (lowIdx+highIdx)>>1;
		double midAgg = _thresholds[midIdx].second;
		if (fabs(agg-midAgg) < DBL_EPSILON)
			return midIdx;
		if (agg-midAgg > DBL_EPSILON)
			lowIdx = midIdx + 1;
		else
			highIdx = midIdx - 1;
	}
	return highIdx;
}

void SAT::reportBurst(long t, int h, double agg)
{
	if (_dataLen == -1 || t <= _dataLen)
	{
		if (_bOutput)
			printf("%ld, %d, %f\n", t, h, agg);
	}
}

int SAT::detailSearch(long t1, int l, double agg, int hidx)
{
	// DSR range
	int threshLowIdx = (l>0)?_sat[l-1]->_threshHighIdx+1:0;
	int threshHighIdx = hidx;
	int h0 = _thresholds[threshLowIdx].first;
	int h1 = _thresholds[threshHighIdx].first;
	long t0 = t1-_sat[l]->_s+1;
	long t;

	// find a seed
	int sh = _sat[l-1]->_h;
	long st = t1;
	_aggBuf[sh] = _sat[l-1]->aggVal();

	// compute a line of cells along time t1
	int h;
	for (h = sh-1; h >= h0; -- h)
	{
		_aggBuf[h] = _aggBuf[h+1] - _sat[0]->aggVal(h);
	}
	for (h = sh+1; h <= h1; ++ h)
	{
		_aggBuf[h] = _aggBuf[h-1] + _sat[0]->aggVal(h-1);
	}

	// for each window size
	int hi, bh = 0, lh = h1, nd = 0, bt0 = 0, bt1 = 0;
	for (hi = threshLowIdx; hi <= threshHighIdx; ++ hi)
	{
		h = _thresholds[hi].first;
		double agg = _aggBuf[h];
		if (agg - _thresholds[hi].second > -DBL_EPSILON)
		{
			reportBurst(t1, h, agg);
		}
		for (t = t1-1; t >= t0; -- t)
		{
			agg += _sat[0]->aggVal(h-1+t1-t) - _sat[0]->aggVal(t1-t-1);
			if (agg - _thresholds[hi].second > -DBL_EPSILON)
			{
				reportBurst(t, h, agg);
			}
		}
	}

	return (threshHighIdx-threshLowIdx+1);
}

void SAT::detect(long t, double d)
{
	// layer 0
	_sat[0]->incIdx();
	_sat[0]->setAggVal(d);
	if (_thresholds[0].first == 1 && (d-_thresholds[0].second)>-DBL_EPSILON)
	{
		reportBurst(t, 1, d);
	}

	// higher layers
	int l = 1;
	SATLayer *curLayer = _sat[l], *lowerLayer = _sat[l-1];
	while (l < _numLayer && curLayer->needUpdate(t))
	{
		// update structure
		curLayer->incIdx();
		double agg = 0;
		int i;
		for (i = 0; i < curLayer->_childLayers.size(); ++ i)
		{
			int sIdx = curLayer->_sIdx[i];
			agg += curLayer->_childLayers[i]->aggVal(sIdx);
		}
		curLayer->setAggVal(agg);

		int hidx = hasBurst(agg, lowerLayer->_threshHighIdx+1, curLayer->_threshHighIdx);
		if (hidx >= 0)
		{
			detailSearch(t, l, agg, hidx);
		}

		// next layer
		++ l;
		lowerLayer = curLayer;
		curLayer = _sat[l];
	}
}


int SAT::detailSearch2(long t1, long offset, SATLayer* curLayer, SATLayer* lowerLayer, double agg, double lowAgg, int hidx)
{
	// DSR range
	int threshLowIdx = lowerLayer->_threshHighIdx+1;
	int threshHighIdx = hidx;
	int h0 = _thresholds[threshLowIdx].first;
	int h1 = _thresholds[threshHighIdx].first;
	long t0 = t1-curLayer->_s+1;
	long t;

	// find a seed
	int sh = lowerLayer->_h;
	long st = t1;
	_aggBuf[sh] = lowAgg;

	// compute a line of cells along time t1
	int h;
	for (h = sh-1; h >= h0; -- h)
	{
		_aggBuf[h] = _aggBuf[h+1] - _sat[0]->aggVal(h+offset);
	}
	for (h = sh+1; h <= h1; ++ h)
	{
		_aggBuf[h] = _aggBuf[h-1] + _sat[0]->aggVal(h-1+offset);
	}

	// for each window size
	int hi, bh = 0, lh = h1, nd = 0, bt0 = 0, bt1 = 0;
	for (hi = threshLowIdx; hi <= threshHighIdx; ++ hi)
	{
		h = _thresholds[hi].first;
		double agg = _aggBuf[h];
		if (agg - _thresholds[hi].second > -DBL_EPSILON)
		{
			reportBurst(t1, h, agg);
		}
		for (t = t1-1; t >= t0; -- t)
		{
			agg += _sat[0]->aggVal(h-1+t1-t+offset) - _sat[0]->aggVal(t1-t-1+offset);
			if (agg - _thresholds[hi].second > -DBL_EPSILON)
			{
				reportBurst(t, h, agg);
			}
		}
	}

	return (threshHighIdx-threshLowIdx+1);
}


void SAT::dynamicDetect(long t, double d)
{
	// layer 0
	_sat[0]->incIdx();
	_sat[0]->setAggVal(d);
	if (_thresholds[0].first == 1 && (d-_thresholds[0].second)>-DBL_EPSILON)
		reportBurst(t, 1, d);

	// higher layers
	int dl = _numLayer+1, al = -1, sl = -1, als = _topWinSize;
	int l = 1;
	SATLayer *curLayer = _sat[l], *lowerLayer = _sat[l-1], *lowerLayer2;
	while (l < _numLayer && curLayer->needUpdate(t))
	{
		assert( curLayer->_s >= lowerLayer->_s );

		// update structure
		curLayer->incIdx();
		double agg = 0;
		int i;
		for (i = 0; i < curLayer->_childLayers.size(); ++ i)
		{
			int sIdx = curLayer->_sIdx[i];
			agg += curLayer->_childLayers[i]->aggVal(sIdx);
			assert( curLayer->_childLayers[i]->isActive() );
		}
		curLayer->setAggVal(agg);

		// check threshold
		if (l < 2 || lowerLayer2->_threshHighIdx < 0 || agg > _thresholds[lowerLayer2->_threshHighIdx].second)
		{
			int hidx = hasBurst(agg, lowerLayer->_threshHighIdx+1, curLayer->_threshHighIdx);

			// do detailSearch if needed
			if (hidx >= 0)
			{
				sl = l;
				int olds = curLayer->_s;
				int pt = t-olds;

				// check if add a layer
				int mh = _thresholds[hidx].first;
				int newcs;

				SATLayer* candLayer = NULL;
				SATLayer* layer = _layerBank[lowerLayer->_selfIdx+1];
				while (layer != curLayer)
				{
					if (!layer->isActive() && layer->childrenActive() && layer->_threshHighIdx > lowerLayer->_threshHighIdx)
					{
						if (curLayer->maxCoveredSize() >= layer->_h || l == _numLayer-1)
						{
							// compute news for candLayer
							int ns, kk;
							for (kk = 0, ns = layer->_mins; 
								 kk <= MAX_SHIFTMODE2; ++ kk, ns = (ns<<1))
							{
								if (ns >= lowerLayer->_s && ns <= curLayer->_s && 
									layer->narrowReady(kk, pt+ns) && pt+ns >= layer->_nextAddT)
									break;
							}
							if (kk <= MAX_SHIFTMODE2)
							{
								assert( ns >= layer->_s>>(layer->_shiftMode) );
								assert( ns <= layer->_s<<(MAX_SHIFTMODE2-layer->_shiftMode) );
								if (layer->_h-ns+1 >= lowerLayer->_h)
								{
									candLayer = layer;
									newcs = ns;
									if (candLayer->maxCoveredSize() >= mh)
										break;
								}
							}
						}
					}
					layer = _layerBank[layer->_selfIdx+1];
				}
				if (candLayer)
				{
						while (candLayer->_s > newcs)
							candLayer->narrow(pt);
						while (candLayer->_s < newcs)
							candLayer->widen();

						assert( candLayer->_s <= curLayer->_s && candLayer->_s >= lowerLayer->_s );
						assert( candLayer->_threshHighIdx >= lowerLayer->_threshHighIdx );

						int d = 0;
						int chidx = (candLayer->_threshHighIdx > hidx) ? hidx : candLayer->_threshHighIdx;
						int dsidx[64];
						for (int st = olds-candLayer->_s, cc = 0; st >= 0; st -= candLayer->_s, ++ cc)
						{
							// agg at candLayer
							candLayer->incIdx();
							double agg1 = 0.0;
							for (int i = 0; i < candLayer->_childLayers.size(); ++ i)
							{
								int sIdx = candLayer->_sIdx[i];
								SATLayer* clayer = candLayer->_childLayers[i];
								agg1 += clayer->aggVal(sIdx+((st/clayer->_s)<<clayer->_shiftMode));
								assert( candLayer->_childLayers[i]->isActive() );
							}
							candLayer->setAggVal(agg1);

							int cidx = hasBurst(agg1, lowerLayer->_threshHighIdx+1, chidx);
							if (cidx < chidx)
								d = 1;

							dsidx[cc] = cidx;
						}

						if (d > 0)
						{
							// add this layer
							candLayer->_active = 1;
							_sat.insert(_sat.begin()+l, candLayer);
							++ _numLayer;

							// update the depending/child layers
							map<SATLayer*, int>::iterator sit;
							for (sit = candLayer->_dependLayers.begin(); sit != candLayer->_dependLayers.end(); ++ sit)
							{
								SATLayer* layer = (*sit).first;
								++ (layer->_numActiveChildLayers);
								int tmp = t + (*sit).second;
								if (layer->_nextAddT < tmp)
									layer->_nextAddT = tmp;
							}
							set<SATLayer*>::iterator ucit;
							for (ucit = candLayer->_unqChildLayers.begin(); ucit != candLayer->_unqChildLayers.end(); ++ ucit)
								++ ((*ucit)->_numActiveDependLayers);

							// detail search
							for (int st = olds-candLayer->_s, cc = 0; st >= 0; st -= candLayer->_s, ++ cc)
								if (dsidx[cc] >= 0)
							{
								detailSearch2(t-st, st, candLayer, lowerLayer, candLayer->aggVal(cc<<candLayer->_shiftMode), 
											lowerLayer->aggVal((st/lowerLayer->_s)<<lowerLayer->_shiftMode), dsidx[cc]);
							}

							addBurstRange(t, _sat[l+1]->_h);

							if (hidx <= candLayer->_threshHighIdx)
								hidx = -1;

							++ l;
						}
				}
				else
				{
					al = l;
					if (als > lowerLayer->_s && lowerLayer->_s > 1)
						als = lowerLayer->_s;
				}

				// detail search between curLayer and candLayer
				if (hidx >= 0)
				{
					detailSearch(t, l, agg, hidx);

					if (l < _numLayer-1)
						addBurstRange(t, _sat[l+1]->_h);
					else
						addBurstRange(t, curLayer->_h);
				}
			}	// if (hidx >= 0)
			else
			{
				if (l > 2)
				{
					int newls = lowerLayer->_s << 1;
					if (lowerLayer->_s < curLayer->_s && 
						lowerLayer->_shiftMode < MAX_SHIFTMODE2 &&
						lowerLayer->_h-newls+1 >= lowerLayer2->_h &&
						lowerLayer->_h > newls)
					{
						if (agg < _thresholds[lowerLayer->_threshHigh[lowerLayer->_shiftMode+1]].second 
							&& !inBurstRange(t, lowerLayer->_h))
						{
							lowerLayer->widen();
						}
					}
				}
			}
		}
		else
		{
			if (l >= 2 && !inBurstRange(t, lowerLayer->_h))
			{
				// del a layer
 				if (lowerLayer->_numActiveDependLayers == 0)	// del at the end of the upper cycle
				{
					lowerLayer->_active = 0;
					_sat.erase(_sat.begin()+l-1);
					-- _numLayer;

					// update the depending/child layers
					set<SATLayer*>::iterator sit;
					for (sit = lowerLayer->_unqChildLayers.begin(); sit != lowerLayer->_unqChildLayers.end(); ++ sit)
					{
						-- ((*sit)->_numActiveDependLayers);
						assert( (*sit)->_numActiveDependLayers >= 0 );
					}
					map<SATLayer*, int>::iterator dit;
					for (dit = lowerLayer->_dependLayers.begin(); dit != lowerLayer->_dependLayers.end(); ++ dit)
					{
						-- (((*dit).first)->_numActiveChildLayers);
						assert( ((*dit).first)->_numActiveChildLayers >= 0 );
					}
			
					-- l;
				}
				else
				{
					int newls = lowerLayer->_s << 1;
					if (lowerLayer->_s < curLayer->_s && 
						lowerLayer->_shiftMode < MAX_SHIFTMODE2 &&
						lowerLayer->_h-newls+1 >= lowerLayer2->_h &&
						lowerLayer->_h > newls)
					{
						lowerLayer->widen();
					}

					if (l == _numLayer-1)
					{
						int news = curLayer->_s << 1;
						if (curLayer->_shiftMode < MAX_SHIFTMODE2 &&
							curLayer->_h-news+1 >= _maxWinSize &&
							curLayer->_h-news+1 >= lowerLayer->_h &&
							curLayer->_h > news)
						{
							curLayer->widen();
						}
					}

				}
			}
		}


		// next layer
		++ l;
		lowerLayer2 = lowerLayer;
		lowerLayer = curLayer;
		curLayer = _sat[l];
	}

	// narrow
	int nl = al;
	for (l = 1; l <= nl; ++ l)
	{
		SATLayer* curLayer = _sat[l];
		int olds = curLayer->_s;
		if (olds >= als && (olds%2 == 0))
		{
			int news = (olds>>1);
			if (olds > _sat[l-1]->_s && olds > curLayer->_minds &&
				curLayer->narrowReady(curLayer->_shiftMode-1, t+news) &&
				(l < al || (l == _numLayer-1 && t%news==0) || _sat[l+1]->needUpdate(t)))		// only do it at the begining of the next layer's cycle
			{
				curLayer->narrow(t);
			}
		}
	}
}


void SAT::addExtraLayers()
{
	vector<int> hstruct;
	vector<SATLayer*> child;
	int i;

	if (_layerBank[1]->_h >= 16)		// to deal with small sizes
	{
			i = 0;
			int newh = _layerBank[1]->_h/2;
			int news = _layerBank[1]->_s;
			if (newh <= news)
				news /= 2;

			if (newh > news)
			{
				hstruct.clear();
				child.clear();
				for (int k = 0; k < newh; ++ k)
				{
					hstruct.push_back(1);
					child.push_back(_sat[0]);
				}

				SATLayer* add = new SATLayer();
				add->_h = newh;
				add->_s = news;
				if (add->_s == 1)
				{
					add->_shiftMode = 0;
					add->_shiftStep = 1;
				}

				add->_aggLen = ((_topWinSize+add->_s-1) / add->_s) << MAX_SHIFTMODE2;
				add->_aggregate = new double[add->_aggLen];
				add->_childLayers = child;
				vector<SATLayer*>::iterator vit;
				for (vit = child.begin(); vit != child.end(); ++ vit)
					add->_unqChildLayers.insert(*vit);
				set<SATLayer*>::iterator sit;
				for (sit = add->_unqChildLayers.begin(); sit != add->_unqChildLayers.end(); ++ sit)
				{
					if ((*sit)->_active)
						++ (add->_numActiveChildLayers);
					(*sit)->_dependLayers[add] = -1;
				}

				int count = hstruct.size();
				int j;
				for (j = 0; j < count; ++ j)
				{
					add->_sIdx.push_back(0);
				}
				hstruct.push_back(0);
				for (j = count-1; j >= 0; -- j)
				{
					// sIdx
					add->_sIdx[j] = hstruct[j+1]/add->_childLayers[j]->_s;
					add->_sIdx[j] = add->_sIdx[j] << add->_childLayers[j]->_shiftMode;

					hstruct[j] += hstruct[j+1];
				}

				for (int sk = 0; sk <= MAX_SHIFTMODE2; ++ sk)
				{
					for (j = 0; j < _thresholds.size(); ++ j)
						if ((_thresholds[j]).first > add->_h-add->_s*(1<<sk)/(1<<MAX_SHIFTMODE)+1)
							break;
					add->_threshHigh[sk] = j-1;
					if (sk == MAX_SHIFTMODE)
						add->_threshHighIdx = j-1;
				}
				add->_curIdx = 0;
				add->_active = 0;

				// add to layerBank
				_layerBank.insert(_layerBank.begin()+i+1, add);
			}
	}

	i = 1;
	while (i < _layerBank.size()-1)
	{
		int lh = _layerBank[i]->_h;
		int ls = _layerBank[i]->_s;
		int h = _layerBank[i+1]->_h;
		int s = _layerBank[i+1]->_s;

		int nl;
		for (nl = i; nl >= 1; -- nl)
		{
			SATLayer* tmp = _layerBank[nl];
			if (lh+_layerBank[nl]->_h < h-s/2+1 && _layerBank[nl]->isActive())
		{
			int newh = lh+_layerBank[nl]->_h;
			int news = (newh-lh > h-newh) ? s : ls;
			if (news == s && lh+_layerBank[nl]->_h - news +1 < lh)
				news = ls;

			if (lh+_layerBank[nl]->_h - (news>>MAX_SHIFTMODE) +1 >= lh)
			{
				hstruct.clear();
				hstruct.push_back(_layerBank[nl]->_h);
				hstruct.push_back(lh);
				child.clear();
				child.push_back(_layerBank[nl]);
				child.push_back(_layerBank[i]);

				SATLayer* add = new SATLayer();
				add->_h = newh;
				add->_s = news;
				if (add->_s == 1)
				{
					add->_shiftMode = 0;
					add->_shiftStep = 1;
				}

				add->_aggLen = ((_topWinSize+add->_s-1) / add->_s) << MAX_SHIFTMODE2;
				add->_aggregate = new double[add->_aggLen];
				add->_childLayers = child;
				vector<SATLayer*>::iterator vit;
				for (vit = child.begin(); vit != child.end(); ++ vit)
					add->_unqChildLayers.insert(*vit);
				set<SATLayer*>::iterator sit;
				for (sit = add->_unqChildLayers.begin(); sit != add->_unqChildLayers.end(); ++ sit)
				{
					if ((*sit)->_active)
						++ (add->_numActiveChildLayers);
					(*sit)->_dependLayers[add] = -1;
				}

				int count = hstruct.size();
				int j;
				for (j = 0; j < count; ++ j)
				{
					add->_sIdx.push_back(0);
				}
				hstruct.push_back(0);
				for (j = count-1; j >= 0; -- j)
				{
					// sIdx
					add->_sIdx[j] = hstruct[j+1]/add->_childLayers[j]->_s;
					add->_sIdx[j] = add->_sIdx[j] << add->_childLayers[j]->_shiftMode;

					hstruct[j] += hstruct[j+1];
				}

				for (int sk = 0; sk <= MAX_SHIFTMODE2; ++ sk)
				{
					for (j = 0; j < _thresholds.size(); ++ j)
						if ((_thresholds[j]).first > add->_h-add->_s*(1<<sk)/(1<<MAX_SHIFTMODE)+1)
							break;
					add->_threshHigh[sk] = j-1;
					if (sk == MAX_SHIFTMODE)
						add->_threshHighIdx = j-1;
				}
				add->_curIdx = 0;
				add->_active = 0;

				// add to layerBank
				_layerBank.insert(_layerBank.begin()+i+1, add);

				h = _layerBank[i+1]->_h;
				s = _layerBank[i+1]->_s;
			}
		}
		}

		++ i;
	}

	i = 1;
	while (i < _layerBank.size()-1)
	{
		int lh = _layerBank[i]->_h;
		int ls = _layerBank[i]->_s;
		int h = _layerBank[i+1]->_h;
		int s = _layerBank[i+1]->_s;

		int nl;
		for (nl = i; nl >= 1; -- nl)
			if (lh+_layerBank[nl]->_h < h-s/2+1)
		{
			int newh = lh+_layerBank[nl]->_h;
			int news = (newh-lh > h-newh) ? s : ls;
			if (news == s && lh+_layerBank[nl]->_h - news +1 < lh)
				news = ls;

			if (lh+_layerBank[nl]->_h - (news>>MAX_SHIFTMODE) +1 >= lh)
			{
				hstruct.clear();
				hstruct.push_back(_layerBank[nl]->_h);
				hstruct.push_back(lh);
				child.clear();
				child.push_back(_layerBank[nl]);
				child.push_back(_layerBank[i]);

				SATLayer* add = new SATLayer();
				add->_h = newh;
				add->_s = news;
				if (add->_s == 1)
				{
					add->_shiftMode = 0;
					add->_shiftStep = 1;
				}

				add->_aggLen = ((_topWinSize+add->_s-1) / add->_s) << MAX_SHIFTMODE2;
				add->_aggregate = new double[add->_aggLen];
				add->_childLayers = child;
				vector<SATLayer*>::iterator vit;
				for (vit = child.begin(); vit != child.end(); ++ vit)
					add->_unqChildLayers.insert(*vit);
				set<SATLayer*>::iterator sit;
				for (sit = add->_unqChildLayers.begin(); sit != add->_unqChildLayers.end(); ++ sit)
				{
					if ((*sit)->_active)
						++ (add->_numActiveChildLayers);
					(*sit)->_dependLayers[add] = -1;
				}

				int count = hstruct.size();
				int j;
				for (j = 0; j < count; ++ j)
				{
					add->_sIdx.push_back(0);
				}
				hstruct.push_back(0);
				for (j = count-1; j >= 0; -- j)
				{
					// sIdx
					add->_sIdx[j] = hstruct[j+1]/add->_childLayers[j]->_s;
					add->_sIdx[j] = add->_sIdx[j] << add->_childLayers[j]->_shiftMode;

					hstruct[j] += hstruct[j+1];
				}

				for (int sk = 0; sk <= MAX_SHIFTMODE2; ++ sk)
				{
					for (j = 0; j < _thresholds.size(); ++ j)
						if ((_thresholds[j]).first > add->_h-add->_s*(1<<sk)/(1<<MAX_SHIFTMODE)+1)
							break;
					add->_threshHigh[sk] = j-1;
					if (sk == MAX_SHIFTMODE)
						add->_threshHighIdx = j-1;
				}
				add->_curIdx = 0;
				add->_active = 0;

				// add to layerBank
				_layerBank.insert(_layerBank.begin()+i+1, add);

				h = _layerBank[i+1]->_h;
				s = _layerBank[i+1]->_s;
			}
		}

		++ i;
	}

	for (i = 0; i < _layerBank.size(); ++ i)
	{
		SATLayer* layer = _layerBank[i];
		layer->_selfIdx = i;
		layer->_mins = layer->_s;
		int d = layer->_shiftMode;
		while (d > 0 && (layer->_mins % 2 == 0))
		{
			layer->_mins = layer->_mins >> 1;
			-- d;
		}

		// children
		set<SATLayer*>::iterator sit;
		for (sit = layer->_unqChildLayers.begin(); sit != layer->_unqChildLayers.end(); ++ sit)		
		{
			SATLayer* clayer = (*sit);
			int ms = layer->_mins;
			int ii = 0;
			if (clayer->_s >= ms)
			{
				while (ms < clayer->_s)
				{
					ms = ms << 1;
					++ ii;
				}
			}
			for (; ii <= MAX_SHIFTMODE2; ++ ii)
				++ (layer->_nextShiftReady[ii]);
		}

		// dependLayers
		layer->_minds = -1;
		map<SATLayer*, int>::iterator dit;
		for (dit = layer->_dependLayers.begin(); dit != layer->_dependLayers.end(); ++ dit)		
		{
			SATLayer* dlayer = (*dit).first;
			int dt = -1;
			for (int k = 0; k < dlayer->_childLayers.size(); ++ k)
			{
				if (dlayer->_childLayers[k] == layer && dt == -1)
					dt = 0;
				else
				if (dt >= 0)
					dt += dlayer->_childLayers[k]->_h;
			}
			(*dit).second = dt;
			if (layer->_minds == -1 || layer->_minds > (dlayer->_s >> dlayer->_shiftMode))
				layer->_minds = (dlayer->_s >> dlayer->_shiftMode);
		}

	}
}


bool SAT::getNextData(FILE* fp, double& d)
{
	static int bufptr = 0;
	static int readSize = 0;
	static double buffer[BUF_LEN];

	if (bufptr == readSize)
	{
		if (feof(fp))
			return false;
		readSize = fread(buffer, sizeof(double), BUF_LEN, fp);
		bufptr = 0;
	}

	d = buffer[bufptr++];
	return true;
}


void SAT::incrementalDetect(char* dataFile)
{
	FILE* fp = fopen(dataFile, "rb");
	if (!fp)
	{
		printf("can't open file %s\n", dataFile);
		assert( false );
		return;
	}

//	printf("read %s\n", dataFile);

	long t = 0;
	double d;

#ifdef TIMING
	clock_t totalStartT, totalStopT;
	totalStartT = clock();
#endif

	while (getNextData(fp, d))
	{
		++ t;
		detect(t, d);
	}
	flush(t);

#ifdef TIMING
	totalStopT = clock();
	printf("totalT=%ld\n", totalStopT-totalStartT);
#endif

	fclose(fp);
}

void SAT::incrementalDynamicDetect(char* dataFile)
{
	FILE* fp = fopen(dataFile, "rb");
	if (!fp)
	{
		printf("can't open file %s\n", dataFile);
		assert( false );
		return;
	}

//	printf("read %s\n", dataFile);

	long t = 0;
	double d;

#ifdef TIMING
	clock_t totalStartT, totalStopT;
	totalStartT = clock();
#endif

	while (getNextData(fp, d))
	{
		++ t;
		dynamicDetect(t, d);
	}
	flush(t);

#ifdef TIMING
	totalStopT = clock();
	printf("totalT=%ld\n", totalStopT-totalStartT);
#endif

	fclose(fp);
}

void SAT::batchDetect(double* buffer, long len)
{
	long t;
	for (t = 0; t < len; ++ t)
		detect(t+1, buffer[t]);
	flush(t);
}

void SAT::flush(long t)
{
	_dataLen = t;
	while (!_sat[_numLayer-1]->needUpdate(t))
	{
		++ t;
		detect(t, 0.0);
	}
}

void SAT::addBurstRange(long t, int h)
{
	if (t > _burstRangeT)
	{
		_burstRangeT = t;
		_burstRangeH = h;
	}
	else
	if (t == _burstRangeT)
	{
		if (h < _burstRangeH)
			_burstRangeH = h;
	}
}

bool SAT::inBurstRange(long t, int h)
{
	return (t <= _burstRangeT && h <= _burstRangeH);
}

