
/**
 * Idea Conceived by Dennis Shasha, Implemented by Stacey Kuznetsov
 * @author Dennis Shasha, Stacey Kuznetsov
 */


import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Rectangle;
import java.awt.FlowLayout;
import java.awt.TextField;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.ListIterator;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JApplet;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import StratPal.EmptyDepositionException;
import StratPal.InputParser;
import StratPal.QNode;
import StratPal.QuestionSet;
import StratPal.ScrollThread;

/** 
* Applet class that calls parser on an input document and renders the questions.
* After deposition ends, applet displays deposition results.
*
*/



public class StratPalApplet extends JApplet
	implements ActionListener, MouseListener{

    /**
	 * Id required by Java.
	 */
	private static final long serialVersionUID = 1L;

	
	/**
	 * Internal variables to keep track of parsed questions.
	 */
    InputParser parser;  
    QuestionSet qset;
    
    /**
     * GUI components of applet when it displays deposition questions.
     */
    private JScrollPane pane;  	 //displays questions, answers and graphics
    private JPanel panel;		 //contains pane
    private JPanel bpanel;		 //contains possible answers to questions
    private JButton [] buttons;	 //array of possible answers to question
    private JPanel buttonP;		 //contains deposition option buttons
    private JButton quit;
    private JButton seeDoc;
    private JButton back;
    private JButton forward;
    private JButton togle;
    private JTextArea doc;
    private JButton about;		//contains information about StratPal

  	/** GUI components of applet when it displays original question document.	 */
	private JPanel docOptions;
	private JButton save;
	private JButton exit;

	  /** Icons displayed next to questions depending on their status.   */
    private ImageIcon yellowIcon, redIcon, greenIcon, gryellowIcon, line;
	private ImageIcon checkmark;
	private Icon oldIcon;  //last icon before user selected questions in an order-override
	
	    
    /** URL of question document to be parsed.     */
    private String urlstring;
    
    /** Used when parsing a document in case document contains errors and user wants to resubmit.*/
    boolean continues;
    
    /** True if deposition is over.     */
    boolean done = false;
    
    /** True if user wants to hide impossible questions.    */
    boolean toggle = false;

    /** Y Value of the question that is currently being drawn.    */
    int currentY =0;
    
    /** Node at which the user chose to override default question ordering. */
	private QNode overideNode = null;

	/** Question that is currently being asked. */
	private QNode current = null;
	
	/** Whether or not the user has clicked forward and the previous steps should be remembered. */
	private boolean forwarding = false;
   
	/** Whether or not user is back tracking. */
	private boolean backtracking = false;


	/** Contents of the deposition. */
	private String text = null;
	
	/** Labels to contain image icons.*/
	private LinkedList labels;
	private ListIterator labelIter;

    /** List of all questions. */
    LinkedList questions;
    private ListIterator questionIter;
    
	/** Linked list of lines to separate questions. */
	private LinkedList lines;
	private ListIterator linesIter;
	
	/** Hashmap of questions and their labels. */
	private HashMap qLabelMap;
	
	/** Background color. */
	private int bgColor = 0xEFEFEF;


	/** Default input file.	 */
	private String defaultInput="input.txt";
	

	/**
	 * Initializes icons, and all applet containers and GUI components.
	 * Sets internal variables.
	 * Checks if an input file was specified, if not, tries to read a default file called "input.txt". 
	 */
    public void init() {
   
    	//initialize icons
		try {
			yellowIcon = new ImageIcon(new URL("http://newssohot.com/StratPal/yellow.gif"));
			redIcon = new ImageIcon(new URL("http://newssohot.com/StratPal/red.gif"));
			greenIcon = new ImageIcon(new URL("http://newssohot.com/StratPal/green.gif"));
			checkmark = new ImageIcon(new URL("http://newssohot.com/StratPal/check.gif"));
			line = new ImageIcon(new URL("http://newssohot.com/StratPal/hr.gif"));
		} catch (MalformedURLException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	
		/** Initialize panels and button panel */
    	panel = new JPanel();
    	panel.setLayout(null);
    	panel.setBackground(new Color(0xEFEFEF));
    	
    	bpanel = new JPanel();
    	bpanel.setBackground(new Color(0xEFEFEF));
    	panel.add(bpanel);
    	
    	 
        pane= new JScrollPane(panel);
               
    	
    	/** Initialize buttons */
        quit= new JButton("quit");
        seeDoc = new JButton("edit");
        back = new JButton("undo");
        forward = new JButton("redo");
        about= new JButton("about");
        togle = new JButton("toggle");
        
        /** Add action listeners */
        about.addActionListener(this);
        forward.addActionListener(this);
        back.addActionListener(this);
        quit.addActionListener(this);
        seeDoc.addActionListener(this);
        togle.addActionListener(this);
        
        buttonP = new JPanel();
        buttonP.setLayout(new FlowLayout(FlowLayout.CENTER));

        buttonP.add(back);
        buttonP.add(forward);
        buttonP.add(seeDoc);
        buttonP.add(quit);
        buttonP.add(togle);
        buttonP.add(about);

        getContentPane().add(pane, BorderLayout.CENTER);
        getContentPane().add(buttonP, BorderLayout.SOUTH);
        
       
        parser = new InputParser();
		
		continues = true;
		
		docOptions = new JPanel();
	    docOptions.setLayout(new FlowLayout(FlowLayout.CENTER));
	    save = new JButton("save changes and reparse");
	    save.addActionListener(this);
	   
	    exit = new JButton("back to deposition");
	    exit.addActionListener(this);
	    docOptions.add(save);
	    docOptions.add(exit);
	
		if(getParameter("file")== null)
			urlstring = null;
		else 
			urlstring = getParameter("file");
	    
	}

/**
 * Calls parse to parse document.
 */
    public void start(){
    	
    	parse();
    	
    }
    	
    /**
     * Uses InputParser class to parse deposition document.
     * If document is empty or non-existent, error message is displayed.
     *
     */
    private void parse() {
		URL url;
		
		try {
			if(urlstring!=null){
				url = new URL("http://newssohot.com/StratPal/input/"+urlstring);
		
				while(continues){
	   				qset = parser.parse(url);
					continues = false;
				}
			}
			else{
				while(continues){
					BufferedReader in =new BufferedReader(new FileReader(defaultInput));
					if(in==null)
						throw new Exception("Unable to open input file.");
	   				qset = parser.parse(in);
	   				continues = false;
				}
			}
   		}catch (MalformedURLException e1) {
   			JOptionPane.showMessageDialog(null, "Could not open file "+urlstring+".");
			e1.printStackTrace();
			
		} catch(Exception e) {
			handleException(e);
			}
		createQuestions();
    }
		
	private void handleException(Exception e) {
		e.printStackTrace();	
		System.out.println("printed error");
		Object []options = {"edit file", "quit"};
		int selectedValue = JOptionPane.showOptionDialog(null, e.getMessage(), "OPSE!!!",
		            JOptionPane.CLOSED_OPTION, JOptionPane.ERROR_MESSAGE, null, options, options[0]);
		if (selectedValue ==0){
			showDoc();
		}
		else{	
			System.out.println((String)options[selectedValue]);
			continues = false;
			finish();
		 }
		
	}


	/**
	 * Initialize question GUI.
	 *
	 */
	private void createQuestions() {
		questions = new LinkedList();
		labels = new LinkedList();
		lines = new LinkedList();
		
		qLabelMap = new HashMap();
		ListIterator iter = qset.getAllIter();

		while(iter.hasNext()){
			QNode node = (QNode)iter.next();
			JTextArea q = new JTextArea(node.getText());
			JLabel label = new JLabel(node.getName(), yellowIcon, JLabel.LEFT);
			labels.addLast(label);
			label.setName("JLabel");
			q.addMouseListener(this);
			label.addMouseListener(this);
			q.setBackground(new Color(bgColor));
			questions.addLast(q);
			q.setName(node.getName());
			panel.add(q);
			panel.add(label);
			qLabelMap.put(q, label);
			
			JLabel l = new JLabel(line);
			lines.addLast(l);
			panel.add(l);
			
		}
		
		setNext("top");
	
		
	}

				

	/**
	 * Calls on QuestionSet methods to determine which question should be asked next.
	 * Calls on RepaintAll to display the deposition.
	 *
	 * @param answer to previous question
	 */
    private void setNext(String answer) {
  
    	if(!done){

	    	
	    	QNode nd = null;
	    	try{
	    	 nd = qset.setNext(answer);
	    	 if(nd!=null)
		    		current = nd; 
	    	 
	    	}catch(EmptyDepositionException e){
	    		handleEmptyDeposition(e);
	    	}catch(Exception e){
	    		handleException(e);
	    	}
	    	if(nd == null){
	    		finish();
	    	}
	    	
    	}
    	
    	repaintAll();
    }
    
    
    /**
     * Iterates through questions and calls on RenderQuestion to display each question one at a time.
     *
     */
    public void repaintAll(){
      	questionIter = questions.listIterator();
    	labelIter = labels.listIterator();
    	linesIter = lines.listIterator();
    	
    	ListIterator allIter = qset.getAllIter();
		int y =10;
		
		while (allIter.hasNext()){
			QNode node = (QNode)allIter.next();
			y=renderQuestion(node, y);
	
		
		}
		panel.setPreferredSize(new Dimension(460, y));
		getContentPane().validate();
		getContentPane().repaint();
		panel.validate();

	
	}


    /**
     * Java requirement- called by Applet when it is closed.
     */
    public void destroy() {
        pane.add(new TextField("preparing for unloading..."));
    }

    /**
     * Deals with end of deposition.
     */
    public void finish(){
		JOptionPane.showMessageDialog(null, "End of deposition. Thanks for using StratPal.");
		continues = false;
		done = true;
        setNext(null);
      	bpanel.removeAll();
      	buttonP.remove(togle);
      	buttonP.remove(quit);
    }
    
    /**
     * Renders the given question, displaying its color and layout depending on whether
     * it was asked, answered, can't be answered, etc.
     * 
     * @param q questin to be displayed
     * @param y y coordinate at which it should be drawn
     * @return the y coordinate at which the next question should be drawn
     */
    public int renderQuestion(QNode q, int y){
 
    	JTextArea question = (JTextArea)questionIter.next();
       	int numlines = 	q.getNewLines();
    	question.setFont(new Font("arial", Font.PLAIN, 12));
    	question.setEditable(false);
    	question.setLineWrap(true);
    	question.setWrapStyleWord(true);
    	question.setForeground(Color.GRAY);
    	JLabel label = (JLabel)labelIter.next();
    	
    	if((q.isSkipped() && toggle)){
    		question.setVisible(false);
    		label.setVisible(false);
    		((JLabel)linesIter.next()).setVisible(false);
    		return y;
    		
    	}
    	label.setVisible(true);
    	label.setBounds(5, y, 400, 25);
		y+=25;
		
    	if(done == true){
    		bpanel.removeAll();
    		panel.remove(bpanel);
    		System.out.println("done");
    		if(q.isAnswered()){
    			label.setIcon(checkmark);
    			question.setFont(new Font("arial", Font.PLAIN, 12));
    	   		numlines *=25;
        		question.setBounds(10+q.getIndent(), y, (460-q.getIndent()), numlines);
        		y+=numlines;
    			String answer;
    		
    			if(q.numAnswers()>0)
    				answer = q.getAnswer();
    			else
    				answer = "default answer";
    			
    			JTextArea ans = new JTextArea("answer: "+answer);
    			ans.setBounds(20+q.getIndent(), y+5, 450-q.getIndent(), 20);
    			ans.setFont(new Font("arial", Font.PLAIN, 12 ));
    			ans.setBackground(new Color(bgColor));
    			
    			panel.add(ans);
    			y+=30;
    		}
    		else{
    			label.setIcon(null);
    			question.setFont(new Font("arial", Font.PLAIN, 12));
    			question.setForeground(Color.gray);
    			
    			numlines *= 22;
    			question.setBounds(10+q.getIndent(), y, 460-q.getIndent(), numlines);
    			y+=numlines;
    		}
    		
    	}
    	
    	else{
	    	if(q.isCurrent()){
	    		panel.scrollRectToVisible(new Rectangle(0, y-10, 460, 500));
	        	bpanel.removeAll();
	    		currentY = y;
	       		label.setIcon(greenIcon);
	    		label.setFont(new Font("arial", Font.BOLD, 14));
	    		
	    		question.setFont(new Font("arial", Font.PLAIN, 12));
	    		
	    		question.setForeground(Color.BLACK);
	    		numlines *=22;
	    		question.setBounds(10+q.getIndent(), y, 460-q.getIndent(), numlines);
	    		y+=numlines;
	    		Object [] answers = q.getAnswers();
	    		buttons = new JButton[answers.length];
	    		bpanel.setLayout(new FlowLayout(FlowLayout.LEFT, 5, 2));
	    		int butheight =30;
	    		int butwidth = 0;
	    		for(int i = 0; i< answers.length; i++){
	    			buttons[i] = new JButton((String)answers[i]);
	    			buttons[i].setName((String)answers[i]);
	    			
	    			buttons[i].addActionListener(this);
	    			
	    			bpanel.add(buttons[i], -1);
	    			butwidth += (((String)answers[i]).length()+4);
	    			if(butwidth > 460){
	    				butwidth =0;
	    				butheight +=15;
	    			}
	    			if(backtracking && q.getAnswer()!=null){
	    				if(q.getAnswer().compareTo(answers[i])==0)
	    					buttons[i].setForeground(new Color(0x006400));
	    				}
	    			
	    		}
	
	    		bpanel.setBounds(10+q.getIndent(), y, 460-q.getIndent(), butheight);
	    		y+=butheight;
	    	
	    	}
	    	else {
	    		if(!(q.isSkipped() && toggle)){
	    			label.setIcon(yellowIcon);
	    			numlines *=22;
		    		question.setBounds(10+q.getIndent(), y, 460-q.getIndent(), numlines);
		    		question.setPreferredSize(new Dimension(400, 20));
		    		y+=numlines;
	    		}
	    		
		    	if(q.isSkipped()&& !toggle){
		    		label.setIcon(redIcon);
		    		question.setFont(new Font("arial", Font.PLAIN, 12));
	    		}
		    	
	    		else if(!q.isSkipped()&&!q.isAnswered()){
	    		}
		    	
	    		else if(q.isAnswered()){
	    			label.setIcon(checkmark);
	    			
		    		
	    		}
	    	}
    	
    	}
    	question.setVisible(true);
       	JLabel line =((JLabel)linesIter.next());
    	line.setBounds(0, y, 460, 4);
    	line.setVisible(true);
    	return y+10;
      }
    
    
    
	public void actionPerformed(ActionEvent evt) {
		
		if ((evt.getSource()).equals(seeDoc)){
			showDoc();
		}
		
		else if ((evt.getSource()).equals(quit)){
			JOptionPane.showMessageDialog(null, "End of deposition. Thanks for using StratPal.");
    		continues = false;
    		finish();
   		}

		else if ((evt.getSource()).equals(back)){
			back();
		}
		
		else if ((evt.getSource()).equals(forward)){
			forward();
		}
		
		else if((evt.getSource()).equals(exit)){
			handleExit();
		}
		
		else if((evt.getSource()).equals(save)){
			saveChanges();
		}
		
		else if ((evt.getSource()).equals(about)){
			displayAbout();
		}
		
		else if ((evt.getSource()).equals(togle)){
			toggle = !toggle;
			if(toggle)
				togle.setText("untoggle");
			else
				togle.setText("toggle");
			if(!done)
			repaintAll();
    	}
		
		else{	
			forwarding = false;
			qset.resetForwarding(forwarding);	
			setNext(((JButton)evt.getSource()).getName());
		}
			
	}


	/**
	 * If users clicked the About button, StratPal information is displayed.
	 *
	 */
	private void displayAbout() {
		JTextArea msg = new JTextArea("StratPal is an application to assist laywers develop and follow their deposition strategies.\nConceived by " +
		"Dennis Shasha, Designed and Implemented by Stacey Kuznetsov, 2005-2006.");
		msg.setWrapStyleWord(true);
		msg.setBackground(new Color(bgColor));
		JOptionPane.showMessageDialog(null, msg, "About StratPal", JOptionPane.INFORMATION_MESSAGE);
	
			return;
		}
		
	

	/**
	 * Handles when users wishes to end deposition.
	 *
	 */
	private void handleExit() {
		if(text.compareTo(doc.getText())!=0){
			Object []options = {"save changes", "continue editing", "quit"};
			int selectedValue = JOptionPane.showOptionDialog(null, "Are you sure you want to discard changes?", "Quit?",
		            JOptionPane.CLOSED_OPTION, JOptionPane.ERROR_MESSAGE, null, options, options[0]);
			if (selectedValue ==0){
				saveChanges();
				return;
			}
			else if(selectedValue == 1)
				return;
			}
		
		getContentPane().removeAll();
		getContentPane().add(pane, BorderLayout.CENTER);
		getContentPane().add(buttonP, BorderLayout.SOUTH);
		validate();
		repaint();
	}

	/**
	 * Saves changes to document if user edited it.
	 *
	 */
	private void saveChanges() {
		try {
			ByteArrayInputStream stream = new ByteArrayInputStream(doc.getText().getBytes());
			InputStreamReader reader = new InputStreamReader(stream);
			text = doc.getText();
			getContentPane().removeAll();
			getContentPane().add(pane, BorderLayout.CENTER);
			getContentPane().add(buttonP, BorderLayout.SOUTH);
			parser = new InputParser();
			qset = parser.parse(new BufferedReader(reader));
			current = null;
			forwarding = false;
			backtracking = false;
			if(done){
				buttonP.add(quit);
				buttonP.add(togle);
			}
			done = false;
			panel.removeAll();
			panel.add(bpanel);
			createQuestions();
			(new ScrollThread(panel, 0)).start();
		}catch(EmptyDepositionException e){
			e.printStackTrace();
			handleEmptyDeposition(e);
			
		}
		
		catch (Exception e) {
			e.printStackTrace();
			handleException(e);
		}
		
		
	}


	/**
	 * Displays error message if deposition document is empty.
	 * @param e exception
	 */
	private void handleEmptyDeposition(EmptyDepositionException e) {
		JOptionPane.showOptionDialog(null, e.getMessage(), "OPSE!!!",
	            JOptionPane.CLOSED_OPTION, JOptionPane.ERROR_MESSAGE, null, null, null);
	
		
	}


	/**
	 * displays original question document.
	 *
	 */
	private void showDoc() {
		getContentPane().remove(pane);
		getContentPane().remove(buttonP);
		if (text == null){
			try {
				InputStreamReader inStream = new InputStreamReader((new URL("http://newssohot.com/StratPal/input/"+urlstring)).openStream());
				BufferedReader bufferIn = new BufferedReader(inStream);
		    	doc = new JTextArea("");
		    	String line ="";
		    	text="";
		    	while(bufferIn.ready() && line != null){
		    		line = bufferIn.readLine();
		    		text+=(line+"\n");
		    	}
				bufferIn.close();
			}catch (MalformedURLException e) {
					
				e.printStackTrace();
			} catch (IOException e) {
				
				e.printStackTrace();
			}
		}
		doc.setText(text);
		doc.setFont(new Font("arial", Font.PLAIN, 12));
    	doc.setEditable(true);
    	doc.setLineWrap(true);
    	doc.setWrapStyleWord(true);
		doc.setAutoscrolls(true);
		
			
		JScrollPane p = new JScrollPane(doc);
		getContentPane().add(p, BorderLayout.CENTER);
		getContentPane().add(docOptions, BorderLayout.SOUTH);

		
		(new ScrollThread(doc, 0)).start();
		validate();
		repaint();
	
		
	}


	/** If user presses 'back' this function asks QuestionSet for the previous question. */
	
	public void back(){
	
		backtracking= true;
		if(overideNode!= null && current  == overideNode){
			qset.reset();
		}
		QNode node = qset.back(done);
		if(done){
			done = false;
			panel.removeAll();
				ListIterator qiter = questions.listIterator();
				ListIterator liter = labels.listIterator();
				ListIterator lineter = lines.listIterator();
				while(qiter.hasNext()){
					panel.add((JTextArea)qiter.next());
					panel.add((JLabel)liter.next());
					panel.add((JLabel)lineter.next());
				}
				panel.add(bpanel);
				buttonP.add(quit);
				buttonP.add(togle);
				
			
		}
	
		if (node != null ){
			current = node;
			done = false;
			repaintAll();
		}
	}
	
	/**
	 * Forwards to the next quesiton.
	 *
	 */
	public void forward(){
		forwarding = true;
		QNode node = qset.forward();
		if (node != null){
			repaintAll();
		}
	}


	/**
	 * Handles mouse click events.
	 */
	public void mouseClicked(MouseEvent click) {
		QNode node;
		if(!click.getComponent().getName().equals("JLabel")){
			JTextArea area = (JTextArea)click.getSource();
			node = qset.get(area.getName());
		}
		else{
			JLabel label = (JLabel)click.getSource();
			node= qset.get(label.getText());
		}
			
			
			if(!node.isCurrent()){
				try {
					if(done){
						panel.add(bpanel);
						done = false;
					}
					overideNode = node;
					System.out.println("overriding on  "+node.getName());
					qset.overide(node);
				} catch (EmptyDepositionException e) {
					e.printStackTrace();
				}
				repaintAll();
			}
	}


	/**
	 * Highlights question if user's mouse is over it.
	 */
	public void mouseEntered(MouseEvent focus) {
		JLabel label;
		if(focus.getComponent().getName().startsWith("JLabel")){
			label = ((JLabel)(focus.getSource()));
		}
		else{
			label =(JLabel)qLabelMap.get(focus.getSource());
			
			((JTextArea)focus.getSource()).setForeground(Color.BLACK);
		}
		
			oldIcon = label.getIcon();
			label.setIcon(greenIcon);
		
	}


	/**
	 * Resets question background to original color.
	 */
	public void mouseExited(MouseEvent focus) {
		JLabel label;
		if(focus.getComponent().getName().startsWith("JLabel")){
			label = ((JLabel)(focus.getSource()));
			
		}
		else{
			label =(JLabel)qLabelMap.get(focus.getSource());
			if(!((JTextArea)focus.getSource()).getText().equals(current.getText())||done)
				((JTextArea)focus.getSource()).setForeground(Color.gray);
			
		}
		if(qset.get(label.getText()).isCurrent())
			label.setIcon(greenIcon);
		else
			label.setIcon(oldIcon);
		repaint();
		
		
	}
	

	public void mousePressed(MouseEvent arg0) {
		// TODO Auto-generated method stub
		
	}


	public void mouseReleased(MouseEvent arg0) {
		// TODO Auto-generated method stub
		
	}
	


}

