#include "matrix.h"


/* 
   FUNCTION used by svdcmp
   Computes (a^2 + b^2 ) 1/2 without destructive under .o or over .ow.
*/

float pythag(float a, float b) {
  float absa,absb;
  absa=fabsf(a);
  absb=fabsf(b);
  if (absa > absb) return absa*sqrt(1.0+powf((absb/absa),2.0));
  else return (absb == 0.0 ? 0.0 : absb*sqrt(1.0+powf((absa/absb),2.0)));
}



/*   
     FUNCTION used by svdcmp
*/

float sign(float a,float b) {
  if(b>=0.0)
    return fabsf(a);
  else
    return -fabsf(a);
}



/*
   FUNCTION used by svdcmp
   max between two floats
*/

float fmax(float a,float b) {
  if(a > b)
    return a;
  else
    return b;
}



/* 
   min between two int
   FUNCTION used by svdcmp
*/

int imin(int a,int b) {
  if(a<b)
    return a;
  else
    return b;
}



/*
   Give a matrix a[1..m][1..n] this routines computes its singular value decomposition,
   A = U x W x V^t.
      mxn nxn nxn
   U replaces A on output.
   The diagonal matrix of singular values W is output as a vector w[1..n].
   The matrix V (not the transpose) is output as v[1..n][1..n] 
*/

void svdcmp(float **a, int m, int n, float *w, float **v){
  int flag,i,its,j,jj,k,l=0,nm=0;
  float anorm,c,f,g,h,s,scale,x,y,z,*rv1=NULL;

  rv1=(float *)malloc(sizeof(float)*(n+1));
  memset(rv1,0,sizeof(float)*(n+1));
  
  g=scale=anorm=0.0;
  for (i=1;i<=n;i++) {
    l=i+1;
    rv1[i]=scale*g;
    g=s=scale=0.0;
    if (i <= m) {
      for (k=i;k<=m;k++) {
        scale += fabsf(a[k][i]);
      }
      if (scale) {
        for (k=i;k<=m;k++) {
          a[k][i] /= scale;
          s += a[k][i]*a[k][i];
        }
        f=a[i][i];
        g = -sign(sqrtf(s),f);
        h=f*g-s;
        a[i][i]=f-g;
        for (j=l;j<=n;j++) {
	  for (s=0.0,k=i;k<=m;k++)
            s += a[k][i]*a[k][j];
          f=s/h;
          for (k=i;k<=m;k++)
            a[k][j] += f*a[k][i];
        }
        for (k=i;k<=m;k++) a[k][i] *= scale;
      }
    }
    w[i]=scale *g;
    g=s=scale=0.0;
    if (i <= m && i != n) {
      for (k=l;k<=n;k++) scale += fabsf(a[i][k]);
      if (scale) {
        for (k=l;k<=n;k++) {
          a[i][k] /= scale;
          s += a[i][k]*a[i][k];
        }
        f=a[i][l];
        g = -sign(sqrtf(s),f);
        h=f*g-s;
        a[i][l]=f-g;
        for (k=l;k<=n;k++) rv1[k]=a[i][k]/h;
        for (j=l;j<=m;j++) {
          for (s=0.0,k=l;k<=n;k++) s += a[j][k]*a[i][k];
          for (k=l;k<=n;k++) a[j][k] += s*rv1[k];
        }
        for (k=l;k<=n;k++) a[i][k] *= scale;
      }
    }
    anorm=fmax(anorm,(fabsf(w[i])+fabsf(rv1[i])));
  }
  for (i=n;i>=1;i--) {
    if (i < n) {
      if (g) {
        for (j=l;j<=n;j++)
          v[j][i]=(a[i][j]/a[i][l])/g;
        for (j=l;j<=n;j++) {
          for (s=0.0,k=l;k<=n;k++) s += a[i][k]*v[k][j];
          for (k=l;k<=n;k++) v[k][j] += s*v[k][i];
        }
      }
      for (j=l;j<=n;j++) v[i][j]=v[j][i]=0.0;
    }
    v[i][i]=1.0;
    g=rv1[i];
    l=i;
  }
  
  for (i=imin(m,n);i>=1;i--) {
    l=i+1;
    g=w[i];
    for (j=l;j<=n;j++) a[i][j]=0.0;
    if (g) {
      g=1.0/g;
      for (j=l;j<=n;j++) {
        for (s=0.0,k=l;k<=m;k++) s += a[k][i]*a[k][j];
        f=(s/a[i][i])*g;
        for (k=i;k<=m;k++) a[k][j] += f*a[k][i];
      }
      for (j=i;j<=m;j++) a[j][i] *= g;
    } else for (j=i;j<=m;j++) a[j][i]=0.0;
    ++a[i][i];
  }
  for (k=n;k>=1;k--) {
    for (its=1;its<=30;its++) {
      flag=TRUE;
      for (l=k;l>=1;l--) {
        nm=l-1;
        if ((float)(fabsf(rv1[l])+anorm) == anorm) {
          flag=FALSE;
          break;
        }
        if ((float)(fabsf(w[nm])+anorm) == anorm) break;
      }
      if (flag==TRUE) {
        c=0.0;
        s=1.0;
        for (i=l;i<=k;i++) {
          f=s*rv1[i];
          rv1[i]=c*rv1[i];
          if ((float)(fabsf(f)+anorm) == anorm) break;
          g=w[i];
          h=pythag(f,g);
          w[i]=h;
          h=1.0/h;
          c=g*h;
          s = -f*h;
          for (j=1;j<=m;j++) {
            y=a[j][nm];
            z=a[j][i];
            a[j][nm]=y*c+z*s;
            a[j][i]=z*c-y*s;
          }
        }
      }
      z=w[k];
      if (l == k) {
        if (z < 0.0) {
          w[k] = -z;
          for (j=1;j<=n;j++) v[j][k] = -v[j][k];
        }
        break;
      }
      if (its == 30) {
	perror("no convergence in 30 svdcmp iterations");
      }
      x=w[l];
      nm=k-1;
      y=w[nm];
      g=rv1[nm];
      h=rv1[k];
      f=((y-z)*(y+z)+(g-h)*(g+h))/(2.0*h*y);
      g=pythag(f,1.0);
      f=((x-z)*(x+z)+h*((y/(f+sign(g,f)))-h))/x;
      c=s=1.0;
      for (j=l;j<=nm;j++) {
        i=j+1;
        g=rv1[i];
        y=w[i];
        h=s*g;
        g=c*g;
        z=pythag(f,h);
        rv1[j]=z;
        c=f/z;
        s=h/z;
        f=x*c+g*s;
        g = g*c-x*s;
        h=y*s;
        y *= c;
        for (jj=1;jj<=n;jj++) {
          x=v[jj][j];
          z=v[jj][i];
          v[jj][j]=x*c+z*s;
          v[jj][i]=z*c-x*s;
        }
        z=pythag(f,h);
        w[j]=z;
        if (z) {
          z=1.0/z;
          c=f*z;
          s=h*z;
        }
        f=c*g+s*y;
        x=c*y-s*g;
        for (jj=1;jj<=m;jj++) {
          y=a[jj][j];
          z=a[jj][i];
          a[jj][j]=y*c+z*s;
          a[jj][i]=z*c-y*s;
        }
      }
      rv1[l]=0.0;
      rv1[k]=f;
      w[k]=x;
    }
  }
  free(rv1);
}



/* 
   It takes a matrix mat_output [m+1 X n+1], a vector w [n+1 columns] and a matrix
   v[n+1 X n+1] from the svdcmp, the number of words m, the number
   of graphs n and the SVD term k and 
   returns a matrix [m+1 X m+1] which is the product of 'mat_output' X 'w' X 'v' truncated to SVD terms
*/

float ** Reduction(float **mat_output,float *w,float **v,int m,int n,int k) {
  int i,j,l;
  float **out=NULL;
  out = (float **)malloc(sizeof(float *)*(m+1));
  for(i=0;i<=m;i++)
    out[i] = (float *)malloc(sizeof(float)*(m+1));
  for(i=0;i<=m;i++)
    for(j=0;j<=m;j++)
      out[i][j]=0;
  for(i=1;i<=m;i++)
    for(j=1;j<=k;j++)
      mat_output[i][j]=(float)(mat_output[i][j]*w[j]);
  for(i=1;i<=m;i++)
    for(j=1;j<=m;j++)
      for(l=1;l<=k;l++) {
	out[i][j]+=(float)(mat_output[i][l]*mat_output[j][l]);
      }
  return out;
}



/*
  It allocates memory for v and w
*/

void Allocating_vw(float **w,float ***v,int num_grafo) {
  int i,j;
  *w = NULL;
  *v = NULL;
  *w = (float *)malloc(sizeof(float)*(num_grafo+1));
  *v = (float **)malloc(sizeof(float *)*(num_grafo+1));
  for(i=0;i<num_grafo+1;i++) {
    (*v)[i] = (float *)malloc(sizeof(float)*(num_grafo+1));
    (*w)[i] = 0;
    for(j=0;j<num_grafo+1;j++)
      (*v)[i][j]=0;
  }
}

void free_matrix(int n, void **m) {
  int i;
  for(i=0;i<n;i++)
    free(m[i]);
  free(m);
}
