/*
  GraphBlast Algorithm
  graphblast.h responsible for graph matching
  Modified by Diego Reforgiato Recupero
*/
#ifndef GRAPHBLAST_H
#define GRAPHBLAST_H

#include "graph.h"
#include <ctype.h>

////////////////////////////////////////////
//
// Class: vDFS_GraphGrep
// Usage: GraphGrep main class. Doing DFS traversal and output an opfile contained the found intersected patterns
//
/////////////////////////////////////////////

vector<string> lab_path;
                        //The graph list for matching

vector<int> nGraphList;			//The graph list for matching
vector<string> hash_query;
int *cont_hash_query;                   //for graphgrep filtering
int *vet_h;   


extern bool ACC_VF;        // VF acceleration
extern bool GRAPHGREP_VF;
int flag_cc = 0;
int *stackdot;              // for parsing wildcards for inexact matching
int topdot;                 // for parsing wildcards for inexact matching
int stack_1[1000];          // for parsing wildcards for inexact matching
int top_stack = 0;          // for parsing wildcards for inexact matching

// structure for inexact matching for glide queries
typedef struct stack_wild {
  int x[500];
  int y[500];
}Stack_Wild;                // for parsing wildcards for inexact matching

// structure for inexact matching for glide queries
typedef struct emptyslot{   /* take tracks of the  allocated (memory) arrays.*/
  int numslotorig;          /* the dimension of the memory array when it is (all) empty*/
  char *startorig;          /* pointer to the beginning of the array*/
  int numslot;                         /* number of free left locations in the array */
  char *start;                         /* pointer to the first free location in memory */
  struct  emptyslot *next;
  struct emptyslot * prec;
}emptyslot;

// structure for inexact matching for glide queries
typedef struct StringSet_Struct{
  char *typeG;    /*the memory array used to store strings*/
  int  size;      /* size of typeG */
  double ave;
  emptyslot *head;
  int changearray; /* ==1 if we are in the following situation:
		      we are reading a string from a file;
		      part of it is allocated in an array; the array is full.
		      We continue to
		      1)read the string from the
		      file until the end of the string,
		      2) find a free memory where to store all the string.
		   */
}*StringSet;


////////////////////////////////////////////
//
// Class: Parse
// Usage: Used for parsing the wildcards for inexact matching
//
/////////////////////////////////////////////

class Parse {
private:
  int num;         /* id number of a node */
  int top;         /* the top of the stack (to parse)*/
  int *stack;      /*  stack to parse, inserire procedura per controllare dimensione*/
  int stacksize;
  int cyclesize;
  int numedges;    /* number of edges*/
  FILE *fptypes;   /* file types.dat */
  FILE *fpedges;   /* file edges.dat */
  char prec;       /* store the precedent character */
  char number[30]; /* the number of the cut edge */
  int *cycle;      /* the nodes involved in the circle */
  int flag;
  int wildvalue;   /* a value  of a sequence of wildcards*/
  int wildvaluemin;/* a minimum value of a sequence of wildcards*/
  char op;         /* g, e, l. es: find the nodes n2 and n1 connected through two nodes
		      n2/././n1 wildcard value 3, op==e 'equal'*/
  
public:
  Parse() {};                              // Default Constructor
  void ParseCreate();                      // creates space memory
  void ParseParsePath(FILE *fp);           // creates files type.dat and edge.dat
  void check(char c);                      // Creating parser table
  void EvaluateSequenceWildCards(int wild);// Creating parser table
  void StoreWildConnection(int top1);      // Creating parser table
  void InsertCycle();                      // Creating parser table
  void InsertElementInStack();             // Creating parser table
  void print();
  int ParseReturnNumNode();                // return the number of nodes of query
  int ParseReturnNumEdge();                // return the number of edges of query
};

int Parse::ParseReturnNumNode() {
  return num;
}

int Parse::ParseReturnNumEdge() {
  return numedges;
}

// debug method
void Parse::print() {
  fprintf(stderr,"\nnum:%d",num);
  fprintf(stderr,"\ntop:%d",top);
  fprintf(stderr,"\nstacksize:%d",stacksize);
  fprintf(stderr,"\ncyclesize:%d",cyclesize);
  fprintf(stderr,"\nnumedge:%d",numedges);
  fprintf(stderr,"\nprec:%c",prec);
  fprintf(stderr,"\nflag:%d wildvalue:%d wildvaluemin:%d op:%c",flag,wildvalue,wildvaluemin,op);
}

// init method
void Parse::ParseCreate() {
  int i;
  stacksize=1000;
  stack=(int *)calloc(sizeof(int),stacksize);
  assert(stack);
  cyclesize=1000;
  cycle=(int *)calloc(sizeof(int),cyclesize);
  assert(cycle);
  num=0;
  flag = 0;
  numedges=0;
  prec='w';
  top=-1;
  for(i=0;i<cyclesize;i++) {
    cycle[i]=-1;
    //stack[i]=0;
  }
  wildvalue=-1;
  wildvaluemin=0;
  wildhead = NULL;
}

// parse the paths of glide query creating the type.dat and edge.dat files
void Parse::ParseParsePath(FILE *fp) {
  char str[1000];
  int i;
  
  const char space=32;
  
  fptypes =fopen("type.dat","w+");
  assert(fptypes);
  fpedges = fopen("edge.dat","w+");
  assert(fpedges);
  
  fgets(str,1000,fp);
  while((str[0]=='\n' || str[0]=='\r') && !feof(fp))
    fgets(str,1000,fp);
  i=0;
  while(str[i]!='\0' && str[i]!='\n'&& str[i]!='\r') {
    if(str[i]!=space) {
      while(isdigit(str[i])!=0 && flag_cc==1 && flag==0) {
	fprintf(fptypes,"%c",str[i]);
	i++;
      }
      check(str[i]);
    }
    i++;
    if(i==1000 && !feof(fp))
      fgets(str,1000,fp);
  }
  if(prec=='e') {
    check('$'); /* end of the path, check for path-case b/wildcard/ */
  }

  fclose(fptypes);
  fclose(fpedges);
}

// check the glide syntax
void Parse::check(char c) {
  int top1;
  /* {g= greater, l= less, e==equal} */
  int *stack1;
  int i;
  
  if(top==stacksize-1) { /* check for overflow of memory */
    stack1=(int *) calloc(sizeof(int),2*stacksize+1);
    for(i=0; i<stacksize; i++)
      stack1[i]=stack[i];
    free(stack);
    stack=stack1;
    stacksize=2*stacksize;
  }
  
  if(prec=='e' && c=='$') { /*path-query ends with wildcards*/
    while(top >=0 && stack[top]<-1) {
      EvaluateSequenceWildCards(stack[top]);
      stack[top]=-2; /* empty slot */
      top = top-1;
    }			 
    top1=top;
    StoreWildConnection(top1);
    return;	
  }
  
  if(c=='*') {			/* >=0, value in stack = -10 */
    prec='w';
    top++;
    stack[top]= -10;
    return;
  }
  if(c=='+') {			/* >=1, value in satck = -11 */
    prec='w';
    top++;
    stack[top]= -11;
    return;
  }
  if(c=='.') {			/* 1, value in satck = -12 */
    prec='w';
    top++;
    stack[top]= -12;
    return;
  }
  if(c=='?') {			/* {0,1}, value in satck = -13 */
    prec='w';
    top++;
    stack[top]= -13;
    return;
  }
  
  
  if(isdigit(c)==0 && prec=='d' && flag==1) {
    InsertCycle();
    flag=0;
    prec='n';
  }
  
   
  if(c=='/') {/* the end of an element in the path (a type node, a wildcard)*/
    if(prec=='w') {/* the first character in the query is '/': no meaning for graph*/
      prec='e';
      return;
    }
    if(top==-1) {		/* The stack is empty */
      top=0;
      InsertElementInStack();
      return;	
    }
    
    if(stack[top]>=0) { /* It contains a node and not a special character  (, ) */
      fprintf(fpedges,"%d %d\n",stack[top],num);
      numedges++;
      InsertElementInStack();
      return;	
    }
    
    if(stack[top]==-1) { /* It contains ( */
      if(stack[top-1]>=0) {/* character */
	fprintf(fpedges,"%d %d\n",stack[top-1],num); 
	numedges++;
	top++;
	InsertElementInStack();
	return;	
      }
      top1=top-1;
      while(top1 >=0 && stack[top1]<=-1) { /* contain wildcards*/ 
	EvaluateSequenceWildCards(stack[top1]);
	top1 =top1-1;
      }			 
      top++; 
      InsertElementInStack();
      StoreWildConnection(top1); 
      return;	 
    }
    if(stack[top]<-1) {/* Contain wildcard */
      while(top >=0 && stack[top]<-1) {
	EvaluateSequenceWildCards(stack[top]);
	stack[top]=-2; /* empty slot */
	top = top-1;
      }			 
      top1=top;
      while(top1 >=0 && stack[top1]<=-1) {
	if(stack[top1]!=-1)
	  EvaluateSequenceWildCards(stack[top1]);
	top1=top1-1;
      }		  
      top++; 
      InsertElementInStack();
      StoreWildConnection(top1);
      return;	
    }
  }
  
  
  if(c=='(') { /*new branch*/
    stack_1[top_stack+1] = num;
    top_stack++;
    top=top+1;
    stack[top]=-1;
    prec='n';
    return;
  }
  
  if(c==')') {/*closing branch*/
    top_stack--;
    if(prec=='e') check('$');
    while(stack[top]!=-1 && top>=0) {
      stack[top]=-2;	/* -2 means empty slot */
      top=top-1;
    }
    stack[top]=-2;
    top=top-1;
    prec=')';
    return;
  }
  
  if(c=='%') {/*The number of the cycle-edge*/ 
    strcpy(number,"");
    flag=1;
    return;
  }
  if(isdigit(c)!=0) {/*Cycle-edge */
    if (flag==1) /*The number of the cycle-edge has more than 1 digit*/ 
      sprintf(number,"%s%c",number,c);
    else {
      fprintf(fptypes,"%c",c); 
      flag_cc = 1;
      return;
      //sprintf(number,"%c",c);
      //InsertCycle(p);
    }	
    prec='d';			/*Previous character was a number */
    return;
  } 
  
  stack_1[top_stack] = num;
  fprintf(fptypes,"%c",c); 
  flag_cc = 1;
  prec='n';
}

// evaluate the sequence of wildcards
void Parse::EvaluateSequenceWildCards(int wild) {
  char op1;
  
  if(wild == -10) {
    wild = 0;
    op1 = 'g';
  }
  if(wild == -11) {
    wild = 1;
    op1 = 'g';
  }
  if(wild == -12) {
    wild=1;
    op1 = 'e';
  }
  if(wild == -13) {
    wild=1;
    op1 = 'l';
  }
  if(wildvalue == -1) {
    wildvalue=wild;
    op=op1;
    return;
  }
  if(op == op1) {
    wildvalue = wildvalue + wild; 
    return;
  }
  if((op=='e' && op1=='g') || (op=='g' && op1=='e')) {
    op='g';
    wildvalue = wildvalue + wild; 
    return;
  }
  if(op1=='l' && op=='e') {
    wildvaluemin=wildvaluemin + wildvalue; /* minimo valore */
    wildvalue = wildvalue + wild; 
    op='l';
    return;
  }
  if(op1=='e' && op=='l') {
    wildvaluemin=wildvaluemin + wild; /* minimo valore */
    wildvalue = wildvalue + wild; 
    
    return;
  }
  if(op1=='l' && op=='g') {
    return;
  }
  if(op1=='g' && op=='l') {
    wildvalue = wildvaluemin + wild;
    op='g';
    return;
  }
}

// store the information of glide interconnections
void Parse::StoreWildConnection(int top1) {
  int node;
  int node2;
  listwildconnection *new1;
  
  if(top1==-1)
    node=-1;
  else
    node=stack[top1];
  if(node==-1)
    node = stack_1[top_stack];

  if(node <-1) node=-1;
  if(prec=='e')
    node2 =-1;
  else
    node2=num-1;
  
  new1 = (listwildconnection *) calloc(sizeof(listwildconnection),1);
  assert(new1!=NULL);
  
  new1->info[0]=node2; /*or num look InsertElementInStack()*/
  new1->info[1]=node;
  new1->info[2]=wildvalue;
  new1->op=op;
  wildvalue=-1;
  
  new1->next = wildhead;
  wildhead = new1;
}

// dealing with cycles
void Parse::InsertCycle() {
  int n;
  int *stack1;
  int i;
  
  n=atoi(number);
  
  if(n==cyclesize-1) {/* check for overflow of memory */
    stack1=(int *) calloc(sizeof(int),2*cyclesize);
    for(i=0; i<cyclesize; i++)
      stack1[i]=cycle[i];
    free(cycle);
    cycle=stack1;
    cyclesize=2*cyclesize;
  }
  
  
  if(cycle[n]==-1) {/*cycle[] empty, beginning of the edge*/
    cycle[n]=num;
  }
  else {
    fprintf(fpedges,"%d %d\n",num,cycle[n]);
    numedges++;
  }  
}

void Parse::InsertElementInStack() {
  stack[top]=num;
  fprintf(fptypes,"%c",'\n');
  num++;
  prec='n';
}

////////////////////////////////////////////
//
// Class: Query
// Usage: Used for querying during inexact matching
//
/////////////////////////////////////////////
class Query {
 private:
  char *p;               /* to read a string */
  int flag;
  int num;
  char *visitnode;
  StringSet ss;
 public:
  Query(int num);//,listwildconnection *l); // Constructor
  void CheckInOutTableWildConnection(char *str1); // main routine run when graphgrep is executed
  void CheckInOutTableWildConnection_VF(char *queryfilename); // main routine when graphgre_vf is executed
  int GetElementInString();
  void FindDot(char op,int *outids,int *G,int val,int num,int elm,int *pred,int **sol,int *cont_sol,int *num_sol,int start);             // Looking for some path which satisfies a/./+/*/?/ (one node)
  void FindDot2(char op,int *outids,int *G,int val,int num,int elm1,int elm2,int *pred,int **sol,int *cont_sol,int *num_sol);            // Looking for some path which satisfies a/./+/*/?/b (two nodes)
  Query::~Query();           // Destruptor
};

// malloc for emptyslot structure
static void MakeArray(StringSet s)
{
  emptyslot *slot;
  s->typeG = (char *)malloc(sizeof(char)*s->size);
  assert(s->typeG);

  if(s->head != NULL) {
    slot=(emptyslot *)malloc(sizeof(emptyslot));
    assert(slot);
    
    slot->numslotorig = s->size;
    slot->startorig = s->typeG;
    slot->numslot = s->size;
    slot->start = s->typeG;
    slot->next = s->head;
    s->head->prec = slot;
    slot->prec = NULL;
    s->head = slot;
  }
}


// malloc for newstringset structure
StringSet StringSetCreate() {
  StringSet newstringset;
  
  newstringset = (StringSet) malloc( sizeof(struct StringSet_Struct));
  assert(newstringset);
  newstringset->changearray = 0;
  newstringset->head = NULL;
  MakeArray(newstringset);      /* create typeG */
  newstringset->head=(emptyslot *)malloc(sizeof(emptyslot)); /* create a pointer array to a typeG */
  assert(newstringset->head);
  newstringset->head->numslotorig=newstringset->size;
  newstringset->head->startorig=newstringset->typeG;
  newstringset->head->numslot=newstringset->size;
  newstringset->head->start=newstringset->typeG;
  newstringset->head->next=NULL;
  newstringset->head->prec=NULL;
  return newstringset;
}

// init for StringSet
void StringSetInz(StringSet s) {
  
  emptyslot *p;
  
  p = s->head;
  while(p!=NULL) {
    p->start = p->startorig;
    p->numslot=p->numslotorig;
    p= p->next;
  } 
}

// check out of memory
static void CheckMemory(StringSet s) {
  int val;
  
  val =(int)(s->size*s->ave);
  if(s->head->numslot < val) {
    MakeArray(s);
  }
}

// basic list operation
void MoveFrontOldSlot(StringSet s, emptyslot *p) {
  emptyslot *slot;
  
  slot = p->next;
  p->prec->next = slot;
  p->next = s->head;
  s->head->prec = p;
  p->prec = NULL;
  s->head = p;
}

void CopyInsideArray(StringSet s, emptyslot *p) {
  strcpy(p->start,s->head->start);
  p->numslot = p->numslot - strlen(p->start);
  p->start = p->start + strlen(p->start);
}

void CopyInsideNewArray(StringSet s) {
  strcpy(s->head->start,s->head->next->start);
  s->head->numslot = s->head->numslot - strlen(s->head->start);
  s->head->start = s->head->start + strlen(s->head->start);
}

// read the list of Stringset
char *StringSetReadType( StringSet s, FILE *fp) {
  char *startreturned;
  char *oldstart;
  emptyslot *p;
  char *fgets_return;
  do{
    
    fgets_return = fgets(s->head->start,s->head->numslot,fp);
    if( fgets_return == NULL) {
      return NULL;
    }
    
    if(strlen(s->head->start) < s->head->numslot-1 || *(s->head->start + s->head->numslot-1) =='\n'){
      s->head->numslot = s->head->numslot - (strlen(s->head->start)+1);
 
      if(s->changearray==0)
	startreturned= s->head->start;
      if(s->changearray==1)
	startreturned=oldstart;
      s->head->start = s->head->start + strlen(s->head->start) + 1;
      CheckMemory(s);
      s->changearray=0;
      return startreturned;
    }
    /* The head is full and it finishes with incomplete string
       1) find a new free memory big enough for a bigger string
       2) copy the string in the head of this free memory*/
    p=s->head->next;
    if(s->changearray==1)
      s->head->start=oldstart;
    s->changearray=1;
    while(p != NULL && p->numslot < 2*strlen(s->head->start))
      p=p->next;
    if(p!=NULL) {
      oldstart=p->start;
      CopyInsideArray(s, p);
      MoveFrontOldSlot(s,p);
    }
    else {
      s->size = s->size *2;
      MakeArray(s);
      CopyInsideNewArray(s);
      oldstart=s->typeG;
    }
    
  }while(s->changearray==1 && !feof(fp));
}

// searching method dealing with the dot notation, first case
void Query::FindDot(char op,int *outids,int *G,int val,int num,int elm,int *pred,int **sol,int *cont_sol,int *num_sol,int start) {
  int i,m,o;
  if(start==elm && ((val==0 && op=='g') || op=='l')) { // It is for paths of length 0
    sol[*cont_sol] = (int *)malloc(sizeof(int)*(1));
    sol[*cont_sol][0]=elm;
    num_sol[*cont_sol] = 1;
    (*cont_sol)++;
    if((*cont_sol) > 5000) {
      cout << "Too many results!!!";
      exit(1);
    }
  }
  stackdot[topdot++]=elm;
  visitnode[elm] = 1;
  for(i=0;i<num;i++) {
    if(G[elm*num+i]==1 && outids[i]!=1 && visitnode[i]==0) {
      visitnode[i] = 1;
      pred[i]=elm;
      if((val==1 && op=='e') || (val<=1 && op=='g') || (val>=1 && op=='l')) {
	(flag) = 1;
	m = i;
	while(pred[m]!=-1) {
	  o = m;
	  m = pred[m];
	}
	sol[*cont_sol] = (int *)malloc(sizeof(int)*(topdot+1));
	for(m=0;m<topdot;m++) {
	  sol[*cont_sol][m] = stackdot[m];
	}
	sol[*cont_sol][m] = i;
	num_sol[*cont_sol] = topdot + 1;
	(*cont_sol)++;
      }
      FindDot(op,outids,G,val-1,num,i,pred,sol,cont_sol,num_sol,start);
    }
  }
  visitnode[stackdot[topdot-1]]=0;
  topdot--;
}

// searching method dealing for the dot notation, second case
void Query::FindDot2(char op,int *outids,int *G,int val,int num,int elm1,int elm2,int *pred,int **sol,int *cont_sol,int *num_sol) {
  int i,m,j;
  stackdot[topdot++]=elm1;
  visitnode[elm1] = 1;
  for(i=0;i<num;i++) {
    if(G[elm1*num+i]==1 && visitnode[i]==0) {// && outids[i]!=1 && q->visitnode[i]==0) {
      if(visitnode[i] == 0)
	pred[i] = elm1;
      if((val==0 && op=='e' && i==elm2) || (val<1 && op=='g' && i==elm2) || (val>=0 && op=='l' && i==elm2)) {
	sol[*cont_sol] = (int *)malloc(sizeof(int)*(topdot+1));
	sol[*cont_sol][0] = i;
	j = 1;
	for(m=topdot-1;m>=0;m--) {
	  sol[*cont_sol][j++] = stackdot[m];
	}
	num_sol[*cont_sol] = topdot + 1;
	(*cont_sol)++;
	if((*cont_sol) > 5000) {
	  cout << "Too many results!!!";
	  exit(1);
	}
	(flag) = 1;
      }
      else {
	if(outids[i]!=1 && (op=='g' || (val>0 && op=='e') || (val>0 && op=='l')) ) {
	  FindDot2(op,outids,G,val-1,num,i,elm2,pred,sol,cont_sol,num_sol);
	}
      }
    }
  }
  visitnode[stackdot[topdot-1]]=0;
  topdot--;
}

int subset_adj(int *s1,int *s2,int n1,int n2) {
  int i,j,k,dir,flag;
  for(i=0;i<n2;i++)
    if(s1[0]==s2[i]) {
      j=i;
      break;
    }
    else
      j=-1;
  if(j==-1)
    return 0;
  dir = 0;
  for(i=0;i<n2;i++)
    if(s1[1]==s2[i])
      if(j+1==i)
	dir=1;
      else
	if(j-1==i)
	  dir=-1;
	else
	  dir = 0;
  if(dir==0)
    return 0;
  if(n1==2 || n2==2)
    return 1;
  if(dir==1)
    for(k=2;k<n1;k++) {
      flag = 0;
      for(i=0;i<n2;i++)
	if(s1[k]==s2[i]) {
	  flag = 1;
	  if(i!=j+k)
	    return 0;
	}
      if(flag==0) {
	return 0;
      }
    }
  if(dir==-1)
    for(k=2;k<n1;k++) {
      flag = 0;
      for(i=0;i<n2;i++)
	if(s1[k]==s2[i]) {
	  flag = 1;
	  if(i!=j-k)
	    return 0;
	}
      if(flag==0) 
	return 0;
    }
  return 1;  
}

// dealing with our inexact matching constraints
int compare(int *s1,int *s2,int n1,int n2) {
  int i;
  if(n1<=n2) {
    if(subset_adj(s1,s2,n1,n2)==1)
      return 0;
  }
  else
    if(subset_adj(s2,s1,n2,n1)==1)
      return 0;
  /* The following lines deals with the starting and finals nodes two valid paths must not share */
  if(n1==2 || n2==2) {
    if(s1[1]==s2[1])
      return 0;
  }
  else
    if(s1[1]==s2[1] || s1[n1-2]==s2[n2-2]) { // there must not be present 2 paths with the same initial or final vertices
      return 0;
    }
  /* end dealing starting and finals nodes */
  return 1;
}

// intersect the obtained solutions in order to answer correctly to the whole glide query
int intersect_sol(FILE *fpout,int ***sol,int *cont_sol,int cont_w,int **num_sol,int grafo) {
  int i,i1,j,ris,top,top1,flag,k,k1;
  Stack_Wild s;
  for(i=0;i<cont_w;i++) {
    s.x[i] = 0;
    s.y[i] = i;
  }
  top = cont_w-1;
  while(1) {
    for(i=0;i<cont_sol[top];i++) {
      s.x[top] = i;
      //      Wildprint(s,cont_w);
      flag = 0;
      for(i1=0;i1<cont_w-1;i1++) {
      	for(j=i1+1;j<cont_w;j++) {
	  if(compare(sol[i1][s.x[i1]],sol[j][s.x[j]],num_sol[i1][s.x[i1]],num_sol[j][s.x[j]])==0) {
	    flag = 1;
	    break;
	  }
	  else {
	  }
	}
	if(flag==1)
	  break;
      }
      if(flag == 0) {
	fprintf(fpout,"g%d|",grafo);
	  for(k=0;k<cont_w;k++) {
	    for(k1=0;k1<num_sol[k][s.x[k]];k1++) {
	      fprintf(fpout,"%d ",sol[k][s.x[k]][k1]);
	    }
	    if(k < cont_w - 1)
	      fprintf(fpout,"- ");
	  }
	  fprintf(fpout,"\n");
      }
    }
    top1 = top - 1;
    while(s.x[top1]==cont_sol[top1]-1)
      top1--;
    if(top1==-1)
      return -1;
    s.x[top1]++;
    for(i=top1+1;i<cont_w;i++)
      s.x[i]=0;
  }
}

// main procedure for inexact matching when graphgrep_vf is run
void Query::CheckInOutTableWildConnection_VF(char *queryfilename) {
  listwildconnection *t,*w;
  int grafo,valid,cont_w,i,j,elm,val;
  int *T=NULL;
  int *outids=NULL,*outindx=NULL,*pred=NULL,*cont_sol=NULL;  /* outids[n]=1 if the node n is in answer*/
  FILE *fp,*fpout,*fpsp;
  char c;
  char str[100],count1[50];
  int **num_sol = NULL;
  int ***sol=NULL;
  time_t start,end;

  start = clock();
  
  fp = fopen(queryfilename,"r");
  strcpy(str,queryfilename);
  strcat(str,"put");
  fpout=fopen(str,"w+");
  
  while(!feof(fp)) {
    fscanf(fp,"%s",str);
    // reading graph header in query.out
    if(str[0]=='G') {
      fscanf(fp,"%c",&c);
      grafo = atoi(str+6);
      if(T!=NULL)
	free(T);
      sprintf(count1,".//graphblastdata//gn%d.sp",grafo);
      fpsp=fopen(count1,"r");
      assert(fpsp);
      fscanf(fpsp,"%d",&num);
      T=(int *)calloc(sizeof(int),num*num);
      assert(T);
      fread(T,sizeof(int)*num*num,1,fpsp);
      fclose(fpsp);
      if(outids) {
	free(outids);
	outids = NULL;
      }
      outids = (int *)malloc(sizeof(int)*num);
      memset(outids,0,sizeof(int)*num);
    }
    while(!feof(fp)) {
      char c1[5];
      int i_c1=0,el1,el2;
      if(c!='G')
	c = '1';
      while(c!='\n' && c!='G') {
	fscanf(fp,"%c",&c);
	if(c=='G' || feof(fp)) {
	  break;
	}
	if(isdigit(c)!=0) {
	  c1[i_c1++] = c;
	}
	c1[i_c1]='\0';
	if(c==',') {
	  el1 = atoi(c1);
	  i_c1=0;
	}
	if(c==')') {
	  el2 = atoi(c1);
	  i_c1=0;
	  outindx = (int *)realloc(outindx,sizeof(int)*(el1+1));
       	  outindx[el1] = el2;
	  outids[el2] = 1;
	}
      }
      if(c=='\n') {
	valid = 1;
	cont_w = 0;
	t = wildhead;
	while(t!=NULL) {
	  cont_w++;
	  t=t->next;
	}
	pred = (int *)malloc(sizeof(int)*num);
	for(i=0;i<num;i++)
	  pred[i] = -1;
	sol = (int ***)malloc(sizeof(int **)*cont_w);
	cont_sol = (int *)malloc(sizeof(int)*cont_w);
	num_sol = (int **)malloc(sizeof(int *)*cont_w);
	for(i=0;i<cont_w;i++) {
	  num_sol[i] = (int *)malloc(sizeof(int)*5000);
	  sol[i] = (int **)malloc(sizeof(int)*5000);
	}
	for(i=0;i<cont_w;i++)
	  for(j=0;j<5000;j++)
	    sol[i][j]=NULL;
	memset(cont_sol,0,sizeof(int)*cont_w);
	stackdot = (int *)malloc(sizeof(int)*num);
	cont_w = 0;
	w=wildhead;
	while(w!=NULL && valid==1) {
	  if(w->info[0]!=-1 && w->info[1]!=-1) {
	    flag = 0;
	    if(visitnode!=NULL) {
	      free(visitnode);
	      visitnode = NULL;
	    }
	    visitnode=(char *)calloc(num,sizeof(char));
	    memset(pred,-1,sizeof(int)*num);
	    topdot = 0;
	    FindDot2(w->op,outids,T,w->info[2],num,outindx[w->info[0]],outindx[w->info[1]],pred,sol[cont_w],&cont_sol[cont_w],num_sol[cont_w]);
	    valid = flag;
	    if(visitnode!=NULL) {
	      free(visitnode);
	      visitnode = NULL;
	    }
	  }
	  else {
	    if(w->info[0]!=-1) {
	      elm=outindx[w->info[0]];
	    }
	    else {
	      elm=outindx[w->info[1]];
	    }
	    
	    valid=0;
	    val = w->info[2];
	    flag = 0;
	    if(visitnode!=NULL) {
	      free(visitnode);
	    }
	    visitnode=(char *)calloc(num,sizeof(char));
	    memset(pred,-1,sizeof(int)*num);
	    topdot = 0;
	    FindDot(w->op,outids,T,val,num,elm,pred,sol[cont_w],&cont_sol[cont_w],num_sol[cont_w],elm);
	    if(visitnode!=NULL) {
	      free(visitnode);
	      visitnode = NULL;
	    }
	    valid = flag;
	  }
	  w=w->next;
	  cont_w++;
	}
	if(valid==1) {
	  if(cont_w>1) {
	    intersect_sol(fpout,sol,cont_sol,cont_w,num_sol,grafo);
	    fprintf(fpout,"\n");
	  }
	  else {
	    for(i=0;i<cont_sol[0];i++) {
	      fprintf(fpout,"g%d|",grafo);
	      for(j=0;j<num_sol[0][i];j++)
		fprintf(fpout,"%d ",sol[0][i][j]);
	      fprintf(fpout,"\n");
	    }
	    fprintf(fpout,"\n");
	  }
	}
	for(j=0;j<cont_w;j++) {
	  for(i=0;i<5000;i++)
	    if(sol[j][i]!=NULL) {
	      free(sol[j][i]);
	    }
	  free(sol[j]);
	  free(num_sol[j]);
	}
	free(num_sol);
	free(pred);
	free(sol);
	free(cont_sol);
	free(stackdot);
	if(outindx) {
	  free(outindx);
	  outindx = NULL;
	}
	memset(outids,0,sizeof(int)*num);
      }
      if(feof(fp))
	break;
      if(c=='G') {
	ungetc(c,fp);
	break;
      }
      fscanf(fp,"%c",&c);
      if(feof(fp))
	break;
      if(c=='G') {
	ungetc(c,fp);
	break;
      }
      ungetc(c,fp);
    }
  }
  if(outids)
    free(outids);

  end = clock();
  fprintf(stderr,"#Inexact Matching: %.2f\n\n",(float)(end-start)/CLOCKS_PER_SEC);
  return;
}

// main procedure for inexact matching when graphgrep is run
void Query::CheckInOutTableWildConnection(char *str1) {
  listwildconnection *w,*t;
  int j,pos,iz,i,oldcount=-1,grafo;
  int *T=NULL;
  int n1,n2,valid,elm,val;
  int *outids=NULL;              /* outids[n]=1 if the node n is in answer*/
  FILE *fp,*fpout,*fpsp;
  char *output,count[30],count1[30];
  int *conversion = NULL;
  int cont_conv = 0;
  int *outindx=NULL; /* outindx[ind]=n if the node n is in the ind column -answer-out table */
  int ind;
  int *pred=NULL;
  int cont_w;
  int ***sol=NULL;
  int **num_sol = NULL;
  int *cont_sol=NULL;
  time_t start,end;

  start = clock();
  w=wildhead;

  fp = fopen((char *)str1,"r");
  assert(fp);

  ss = StringSetCreate();
  StringSetInz(ss);

  sprintf((char *)str1,"%s%s",str1,"put");
  fpout=fopen((char *)str1,"w+");
  StringSetReadType(ss, fp);
  output=StringSetReadType(ss, fp);
  fprintf(stderr,"\nAA%sAA",output);
  i = j = 0;
  char ap[10];
  int cont_ap = 0;
  while(output[i]!='\n') {

    while(output[i]!=' ' && output[i]!='\n') {
      if(isdigit(output[i])!=0) {
	ap[cont_ap++]=output[i++];
      }
      else {
	if(output[i]==' ' || output[i]=='\n')
	  break;
	i++;
      }
    }
    ap[cont_ap]='\0';
    if(cont_ap!=0) {
      conversion = (int *)realloc(conversion,sizeof(int)*(cont_conv+1));
      conversion[j++] = atoi(ap);
      cont_conv++;
    }
    cont_ap = 0;
    i++;
  }

  assert(fpout);
  while(!feof(fp)) { // query0.out
    output = StringSetReadType(ss, fp);
    while(!feof(fp) && output[0]!='G' ) {   //read each qoutable
      output=StringSetReadType(ss, fp);
    }
    strcpy(count,"");
    iz = 15;
    if(output!=NULL) {
      while(output[iz]!='\t') {
	count[iz-15] = output[iz];
	iz++;
      }
      count[iz-15]='\0';
    }
    output=StringSetReadType(ss, fp);
    while(!feof(fp) && output[0]!='\n') {    //read each graph

      if(DEBUG_OPFILE){
	cout << count;
	cout.flush();
      }
      i = atoi(count);
      if(i!=oldcount) {
	grafo = i;
	oldcount = i;
	if(T!=NULL) 
	  free(T);
	sprintf(count1,".//graphblastdata//gn%d.sp",i);
	fpsp=fopen(count1,"r");
	assert(fpsp);
	fscanf(fpsp,"%d",&num);
	T=(int *)calloc(sizeof(int),num*num);
	assert(T);
	
	fread(T,sizeof(int)*num*num,1,fpsp);
	fclose(fpsp);
	
      }
      if(outids==NULL) {
	outids=(int*)calloc(sizeof(int),num);
	assert(outids);
      }
      else {
	free(outids);
	outids=(int*)calloc(sizeof(int),num);
	assert(outids);
      }
      
      if(outindx) {
	free(outindx);
	outindx = NULL;
      }

      for(i=0;i<num;i++)
	outids[i]=0;
      ind=0;
      i = 0;
      char str_app[10];
      while(i!=strlen(output)) {
	int kk = 0;
	while(output[i]!=' ' && output[i]!='\n') {
	  str_app[kk++]=output[i++];
	}
	str_app[kk]='\0';
	//	n1 = GetElementInString();
	if(!strcmp(str_app,""))
	  break;
	n1 = atoi(str_app);
	//	if(n1==-1)
	//break;
	outids[n1]=1;
	outindx = (int *)realloc(outindx,sizeof(int)*(ind+1));
	outindx[ind]=n1;
	ind++;
	if(output[i]==' ')
	  i++;
	else
	  break;
      }
      valid=1;
      cont_w = 0;
      t=wildhead;
      while(t!=NULL) {
	cont_w++;
	t=t->next;
      }

      pred = (int *)malloc(sizeof(int)*num);
      for(i=0;i<num;i++)
	pred[i] = -1;
      sol = (int ***)malloc(sizeof(int **)*cont_w);
      cont_sol = (int *)malloc(sizeof(int)*cont_w);
      num_sol = (int **)malloc(sizeof(int *)*cont_w);
      for(i=0;i<cont_w;i++) {
	num_sol[i] = (int *)malloc(sizeof(int)*5000);
	sol[i] = (int **)malloc(sizeof(int)*5000);
      }
      for(i=0;i<cont_w;i++)
	for(j=0;j<5000;j++)
	  sol[i][j]=NULL;
      memset(cont_sol,0,sizeof(int)*cont_w);
      stackdot = (int *)malloc(sizeof(int)*num);
      cont_w = 0;
      w=wildhead;
      while(w!=NULL && valid==1) {
	if(w->info[0]!=-1 && w->info[1]!=-1) {
	  for(i=0;i<cont_conv;i++) {
	    if(conversion[i] == w->info[0])
	      n1 = i;
	    if(conversion[i] == w->info[1])
	      n2 = i;
	  }
	  flag = 0;
	  
	  if(visitnode!=NULL) {
	    free(visitnode);
	    visitnode = NULL;
	  }
	  visitnode=(char *)calloc(num,sizeof(char));
	  
	  memset(pred,-1,sizeof(int)*num);
	  topdot = 0;
	  FindDot2(w->op,outids,T,w->info[2],num,outindx[n1],outindx[n2],pred,sol[cont_w],&cont_sol[cont_w],num_sol[cont_w]);
	  valid = flag;
	  if(visitnode!=NULL) {
	    free(visitnode);
	    visitnode = NULL;
	  }
	}
	else {
	  if(w->info[0]!=-1) {
	    for(i=0;i<cont_conv;i++) {
	      if(conversion[i] == w->info[0])
		n1 = i;
	    }
	    elm=outindx[n1];
	  }
	  else {
	    for(i=0;i<cont_conv;i++) {
	      if(conversion[i] == w->info[1])
		n2 = i;
	    }
	    elm=outindx[n2];
	  }
	  
	  valid=0;
	  n2=0;
	  
	  val = w->info[2];
	  flag = 0;
	  if(visitnode!=NULL) {
	    free(visitnode);
	  }
	  visitnode=(char *)calloc(num,sizeof(char));
	  memset(pred,-1,sizeof(int)*num);
	  topdot = 0;
	  FindDot(w->op,outids,T,val,num,elm,pred,sol[cont_w],&cont_sol[cont_w],num_sol[cont_w],elm);
	  if(visitnode!=NULL) {
	    free(visitnode);
	    visitnode = NULL;
	  }
	  valid = flag;
	}

	w=w->next;
	cont_w++;
      }
      if(valid==1) {
	if(cont_w>1) {
	  intersect_sol(fpout,sol,cont_sol,cont_w,num_sol,grafo);
	  fprintf(fpout,"\n");
	}
	else {
	  for(i=0;i<cont_sol[0];i++) {
   	    fprintf(fpout,"g%d|",grafo);
	    for(j=0;j<num_sol[0][i];j++)
	      fprintf(fpout,"%d ",sol[0][i][j]);
	    fprintf(fpout,"\n");
	  }
	  fprintf(fpout,"\n");
	}
      }
      if(!feof(fp)) {
	output=StringSetReadType(ss, fp);
      }
      for(j=0;j<cont_w;j++) {
	for(i=0;i<5000;i++)
	  if(sol[j][i]!=NULL) {
	    free(sol[j][i]);
	  }
	free(sol[j]);
	free(num_sol[j]);
      }
      free(num_sol);
      free(pred);
      free(sol);
      free(cont_sol);
      free(stackdot);
    }
  }
  if(outindx!=NULL) 
    free(outindx);
  if(outids!=NULL) 
    free(outids);
  if(T!=NULL) 
    free(T);
  fclose(fp);

  end = clock();
  fprintf(stderr,"#Inexact Matching: %.2f\n\n",(float)(end-start)/CLOCKS_PER_SEC);
  return;
}

// Query distruptor
Query::~Query() {
  int i;
  emptyslot *p;
  p = ss->head;
  while(p!=NULL) {
    ss->head=ss->head->next;
    free(p->startorig);
    free(p);
    p=ss->head;
  }
  free(ss);
  free(visitnode);
}

int Query::GetElementInString() {
  int indop=-1;
  char str[10];
  strcpy(str,"");
  while(p[0]==' ' && p[0]!='\0')
    p++;
  if(p[0]=='\n')
    return -1;
  while(p[0]!=' ' && p[0]!='\0') {
    sprintf(str,"%s%c",str,p[0]);
    p++;
  }
  indop=atoi(str);
  return indop;
}

// Query constructor
Query::Query(int Num) { 
  int i;
  
  num = Num;
  visitnode = (char *) calloc(sizeof(char),Num);
  assert(visitnode!=NULL);
  return;
}

// class for DFS visit
template <class GraphADT> class vDFS_GraphGrep {
  GraphADT &G;		//The Graph reference need to be visited
  DEGREE<GraphADT> degree;//Degree information of this graph
  ofstream fout;		//output file stream
  
  //data used for DFS traversal
  int depth,cnt,cntP,LP;
  int curLP,cut;
  vector<int> pre,post;	//pre, post vector(size of V)
  vector<int> nodepos;	//position of the nodes in the output
  vector<int> outorder;
  
  //for output LP
  STACK< pathWithIndex* > paths;
  
  //current nodes stack 
  deque<int> pathstack;
  
  //flag for new tree
  bool isNewTree;
  
  
  //tmp counters
  int IntersectCount;	
  int SelectCount;
  int CartisianCount;
  int SQ;
  int iTableCount;
  int	iTotalNode;
  
  //used in the opfile
  vector<int> vTestUnique;
  deque<int> selectNode;		//store the nodes need to be selected
  
  fingerprint fp;			//fingerprint of this graph
  
  //The output function of the operations
  void outputIntersect(pathWithIndex&,pathWithIndex&, int,int);
  void outputTestUnique(pathWithIndex&,int,int);
  void outputCartesianProduct();
  void outputSelect(int,int);
  void outputSelectLoop(pathWithIndex& pw1,int,int,int);
  
  //utility functions that handle pathes
  pathWithIndex* join(pathWithIndex& p1,pathWithIndex& p2,int joint);	//join 2 given paths in a given joint
  void checkLoop(pathWithIndex& p);					//check if there's a loop in the path
  void createPath(int LenthPath,int joint=-1,int tmpLP=0);		//create a path from curren pathstack
  
  //utility functions
  //void show(char* , Edge );
  //void show(char* );
  //void show(const char* , int );
  //void show(const vector<int>& out);
  //void showNodePos(const pathADT& p);
  //void showDepth();
  string showTable(const vector<int>& tab);
  //	int GetHash(const vector<int>& tab);
  
  //DFS traversal function(Right side)
  void dfsR(Edge e);
  
  
 public:
  ~vDFS_GraphGrep(){}
  
  //Constructor
  // Params: 1. GraphADT Object 
  //         2. Lengthpath 
  vDFS_GraphGrep(GraphADT &G,int lp);
  
  //int operator[](int v) const{return ord[v];}
  unsigned int* getFingerPrint(){  return fp.getFP();  };	//get the fingerprint of this graph
  void writeOutputOrder(char* outfile);			//writing node position of the output into a given file
};

/////////////////////////////////////////////////////////////
//	
//	Implementation detail of vDFS_GraphGrep templete class. 
//	Put here because of templete limitation
//
/////////////////////////////////////////////////////////////
template <class GraphADT>
string vDFS_GraphGrep<GraphADT>::showTable(const vector<int>& tab){
	
  string ret;
  
  for(int j=0;j<tab.size();j++){
    ret+=G.getLabel(tab[j]);
  }
  
  
  if(QUERY_WITH_DEGREE){
    //degree information
    ret+="^";
    for(int i=0;i<tab.size();i++){
      char tmp[10];
      sprintf(tmp,"_%d",degree[tab[i]]);
      ret+=tmp;
    }
  }
  
  if(!QUIET_MODE){
    cout << "table:" << ret<<endl;
  }
  
  return ret;	
}

/*template <class GraphADT>
  int vDFS_GraphGrep<GraphADT>::GetHash(const vector<int>& tab){
  
  int h = 0;
  string c;
  
  for(int j=0;j<tab.size();j++)
  c+=G.getLabel(tab[j]);
  
  for(int j=0;j<c.size();j++) {
  h = (64*h + c[j])%HASHP;
  }
  return h;
  }
*/

template <class GraphADT>
void vDFS_GraphGrep<GraphADT>::writeOutputOrder(char* outfile){
  FILE* fp=fopen(outfile,"a");
  assert(fp);
  
  for(int i=0;i<outorder.size()*2-2;i++)
    fprintf(fp,"*");
  
  fprintf(fp,"\n");
  for(int j=0;j<outorder.size();j++)
    fprintf(fp,"%d ",outorder[j]);
  
  fprintf(fp,"\n");
  for(int k=0;k<outorder.size()*2-2;k++)
    fprintf(fp,"*");
  
  
  fprintf(fp,"\n");
  fclose(fp);
}

// class for intersecting partial solutions when graphgrep is run
template <class GraphADT>
void vDFS_GraphGrep<GraphADT>::outputIntersect(pathWithIndex& pw1,pathWithIndex& pw2,int pos1,int pos2){
  const pathADT& p1=*(pw1.getPath());
  const pathADT& p2=*(pw2.getPath());
  char tabname1[100],tabname2[100];
  
  if(pw1.getIndex()==-1)
    sprintf(tabname1,"^%s",showTable(p1).c_str());
  else
    sprintf(tabname1,"qoutable%d",pw1.getIndex());
  
  if(pw2.getIndex()==-1)
    sprintf(tabname2,"^%s",showTable(p2).c_str());
  else
    sprintf(tabname2,"qoutable%d",pw2.getIndex());
  
  if(DEBUG_OPFILE){
    cout<<"INTERSECT "<<"qoutable"<< iTableCount<<" "<< tabname1<< " "<<pos1;
    cout<<" " <<tabname2<< " "<<pos2<<endl;		
  }
  
  if(pos1==-1)
    throw GraphGrepExcp("INTERSECT Position -1 at outputIntersect()");
  
  fout<<"INTERSECT "<<"qoutable"<< iTableCount<<" "<< tabname1<< " "<<pos1;
  fout<<" " <<tabname2<< " "<<pos2<<"\n";
  
  iTableCount++;
}

// dealing wih cartesian product when graphgrep is run
template <class GraphADT>
void vDFS_GraphGrep<GraphADT>::outputCartesianProduct(){		
  
  if(CartisianCount<2){		//only 1 tree exist in the last
    pathWithIndex& pw1=*paths.pop();
    pathADT& p1=*(pw1.getPath());
    
    //writing output node position information
    for(int i=0;i<p1.size();i++)
      outorder.push_back(p1[i]);
  }else{						//Is a forest
    char tabname1[100],tabname2[100];
    
    pathWithIndex& pw1=*paths.pop();
    pathWithIndex& pw2=*paths.pop();
    
    pathADT& p1=*(pw1.getPath());
    pathADT& p2=*(pw2.getPath());
    
    //writing output node position information
    for(int i=0;i<p1.size();i++)
      outorder.push_back(p1[i]);
    
    for(int j=0;j<p2.size();j++)
      outorder.push_back(p2[j]);
    
    if(pw1.getIndex()==-1)
      sprintf(tabname1,"^%s",showTable(p1).c_str());
    else
      sprintf(tabname1,"qoutable%d",pw1.getIndex());
    
    if(pw2.getIndex()==-1)
      sprintf(tabname2,"^%s",showTable(p2).c_str());
    else
      sprintf(tabname2,"qoutable%d",pw2.getIndex());
    
    if(DEBUG_OPFILE){
      cout<<"CARTESIANPRODUCT "<<"qoutable"<< iTableCount<<" "<< tabname1<<" "<< tabname2<<endl  ;
      cout.flush();
    }
    fout<<"CARTESIANPRODUCT "<<"qoutable"<< iTableCount<<" "<< tabname1<<" "<< tabname2<<"\n" ;
    
    iTableCount++;
    
    CartisianCount--;
    
    while(CartisianCount>1){
      pathWithIndex& pwNxt=*paths.pop();
      pathADT& pNxt=*(pwNxt.getPath());
      
      if(pwNxt.getIndex()==-1)
	sprintf(tabname1,"^%s",showTable(pNxt).c_str());
      else
	sprintf(tabname1,"qoutable%d",pwNxt.getIndex());
      
      if(DEBUG_OPFILE){
	cout<<"CARTESIANPRODUCT "<<"qoutable"<< iTableCount<<" "<<"qoutable"<< iTableCount-1<<" "<< tabname1<<endl  ;
	cout.flush();
      }
      fout<<"CARTESIANPRODUCT "<<"qoutable"<< iTableCount<<" "<<"qoutable"<< iTableCount-1<<" "<< tabname1<<"\n";
      
      //insert position to output order
      for(int i=0;i<pNxt.size();i++)
	outorder.push_back(pNxt[i]);
      
      iTableCount++;CartisianCount--;
    }
    
  }
  
}

template <class GraphADT>
void vDFS_GraphGrep<GraphADT>::outputSelect(int pos1,int pos2){	
  if(DEBUG_OPFILE){
    cout<<"SELECT "<<"qoutable"<< iTableCount<<" "<<"qoutable"<< iTableCount-1<<" "  ;
    cout<< pos1<<" "<<pos2<<endl;
  }
  
  fout<<"SELECT "<<"qoutable"<< iTableCount<<" "<<"qoutable"<< iTableCount-1<<" "  ;
  fout<< pos1<<" "<<pos2<<"\n";
  
  
  iTableCount++;
}

template <class GraphADT>
void vDFS_GraphGrep<GraphADT>::outputSelectLoop(pathWithIndex& pw1,int pos1,int pos2,int len){	
  const pathADT& p1=*(pw1.getPath());
  char tabname1[100];
  
  if(pw1.getIndex()==-1)
    sprintf(tabname1,"^%s",showTable(p1).c_str());
  else
    sprintf(tabname1,"qoutable%d",pw1.getIndex());
  
  if(DEBUG_OPFILE){
    cout<<"SELECT "<<"qoutable"<< iTableCount<<" "<< tabname1<<" "  ;
    cout<< pos1<<" "<<pos2<<endl;
  }
  
  fout<<"SELECT "<<"qoutable"<< iTableCount<<" "<< tabname1<<" "  ;
  fout<< pos1<<" "<<pos2<<"\n";
  
  pw1.setIndex(iTableCount);
  iTableCount++;
}

template <class GraphADT>
void vDFS_GraphGrep<GraphADT>::createPath(int LenthPath,int joint,int tmpLP){		//output a path of LengthPath
  pathADT	*path=new pathADT(LenthPath+1);	//LP equals nodes+1
  
  for(int i=0;i<=LenthPath;i++){		//pop node to parent(curLP+1)
    (*path)[i]=pathstack[0];
    pathstack.pop_front();
  }
  
  SQ +=LenthPath;
  pathWithIndex *pIndex= new pathWithIndex(path);
  
  int isize=selectNode.size();
  for(int j=0;j<isize;j++){
    (*pIndex).addSelectNode(selectNode[0]);
    selectNode.pop_front();
  }
  
  //show(*path);
  if(!isNewTree && paths.size()>0 ){
    paths.push(join(*paths.pop(),*pIndex,joint));		
  }else{
    //cout<< "New tree, Paths:"<<pIndex->getIndex()<< " itableCount:"<<iTableCount<<endl;
    
    paths.push(pIndex);
    isNewTree=false;
  }
  
  lab_path.push_back(showTable(*path).c_str());
  //	cout << lab_path[lab_path.size()-1];
  
  unsigned int FP=fp.addPattern(showTable(*path));
  
  if(!QUIET_MODE){
    cout<<showTable(*path).c_str()<< " FP:"<<FP<<endl;
  }
  
  //show(*path);
  //cout<<"FP:"<<FP<<"\t";
  
  curLP=tmpLP;
  
  string st = showTable(*path);

  int i;
  for(i=0;i<hash_query.size();i++)
    if(hash_query[i]==st) {
      cont_hash_query[i]++;
      return;
    }
  hash_query.push_back(st);
  cont_hash_query = (int *)realloc(cont_hash_query,sizeof(int)*(i+1));
  cont_hash_query[i]=1;
}

template <class GraphADT>
void vDFS_GraphGrep<GraphADT>::outputTestUnique(pathWithIndex& pw,int oldLen,int tmpselectCount){
  const pathADT& p=*(pw.getPath());
  int len=p.size();
  
  if(iTableCount>0){
    if(DEBUG_OPFILE)
      cout<<"TESTUNIQUE "<<"qoutable"<< iTableCount<<" "<<"qoutable"<< iTableCount-1<<" "  ;
    
    fout<<"TESTUNIQUE "<<"qoutable"<< iTableCount<<" "<<"qoutable"<< iTableCount-1<<" "  ;
  }else{	//for the case that there's no other paths involved in the graph
    char tabname[100];
    if(pw.getIndex()==-1)
      sprintf(tabname,"^%s",showTable(p).c_str());
    else
      sprintf(tabname,"qoutable%d",pw.getIndex());
    
    if(DEBUG_OPFILE)
      cout<<"TESTUNIQUE "<<"qoutable"<< iTableCount<<" "<<tabname<<" "  ;
    
    fout<<"TESTUNIQUE "<<"qoutable"<< iTableCount<<" "<<tabname<<" "  ;	
    pw.setIndex(iTableCount);
  }
  
  int count=0;
  
  //Generate Testunique position from last Testunique position(iTotalNode-1) to current length
  //!! Because the each position will be unique after Testunique, we don't need to check the old position
  for(int i=((iTotalNode-1)<0?0:(iTotalNode-1));i<((len-tmpselectCount)>oldLen?(len):(len-tmpselectCount+1));i++) {
    if(DEBUG_OPFILE)
      cout << i<<" ";
    fout << i<<" ";
    count++;
  }
  //cout.flush();	
  
  if(count==0){
    if(DEBUG_OPFILE)
      cout<<-1;
    fout<<-1;
  }
  
  if(DEBUG_OPFILE)
    cout<<endl;
  
  fout<<"\n";
  
  iTableCount++;
  iTotalNode=len;
}

template <class GraphADT>
void vDFS_GraphGrep<GraphADT>::checkLoop(pathWithIndex& pw1){	//check if there's a loop in the path
  pathADT& p1=*(pw1.getPath());
  int lenP1=p1.size();
  
  vector<int>& selectNodes=pw1.getSelectNodes();	//get the need-select node of this path
  int ssize=selectNodes.size();
  vector<int> selectNodePair((ssize)*2,-1);	//allocate vector for the possible loops
  
  vector<int> VisitedNode;
  
  int count=0;
  
  int i=0;
  int size=lenP1;
  typedef vector<int> vInt;
  vInt::iterator it=p1.begin(),
    it_end=p1.end();
  
  while(i<size){
    int n=p1[i];
    
    for(int x1=0;x1<ssize;x1++){
      if(n==selectNodes[x1]){
	bool stopflag=false;
	int k=selectNodes[x1];
	
	for(int j=0;j<VisitedNode.size();j++){	//already visited select node
	  if(k==VisitedNode[j]){
	    stopflag=true;
	    break;
	  }
	}
	
	if(stopflag)	break;
	
	if(selectNodePair[x1]>-1){		//loop
	  selectNodePair[x1+ssize]=-1;	//reset it
	  
	  VisitedNode.push_back(k);
	  count++;	//found loop
	  
	  selectNodes[x1]=-1;	//mark self-loop node
	  
	  outputSelectLoop(pw1,selectNodePair[x1],i,lenP1);
	  
	  if(it+1!=it_end)		//!!maybe some bug here
	    p1.erase(it,it+1);
	  else
	    p1.pop_back();
					
	  i--;
	  size--;
	  
	}else		//first met
	  selectNodePair[x1]=i;		
	
	vTestUnique[n]=0;
      }
    }
    i++;
    it++;
  }
}

template <class GraphADT>
pathWithIndex* vDFS_GraphGrep<GraphADT>::join(pathWithIndex& pw1,pathWithIndex& pw2,int joint){
  const pathADT& p1=*(pw1.getPath());
  const pathADT& p2=*(pw2.getPath());
  const vector<int>& selectNodes1=pw1.getSelectNodes();
  const vector<int>& selectNodes2=pw2.getSelectNodes();
  
  //	show(selectNodes1);
  //	show(selectNodes2);
  
  //	show(p1);
  //	show(p2);
  
  //Check loop before doing join
  //if(pw1.getPath()->size()>=LP)		
  checkLoop(pw1);
  
  checkLoop(pw2);
  
  //	show(p1);
  //	show(p2);
  
	
  int lenP1=p1.size(),lenP2=p2.size();
  int pos1=-1,pos2=-1;
  
  //if(joint==0)
  joint=p2[0];
  
  int iSelect=pw2.getSelectCount();
  vector<int> selectNodePair((iSelect)*2,-1);
  vector<int> selectPairResult((iSelect)*2,-1);
  
  pathADT* retP=new pathADT(lenP1);			//need to be free in some place
  
  for(int i=0;i<lenP1;i++) {
    int n=p1[i];		
    (*retP)[i]=n;		//add nodes into return path
    
    if(nodepos[n]==-1 ){		//modify the current position corresonding to P1
      nodepos[n]=i;
    }
    
    if(vTestUnique[n]>0)	//testunique checking control
      vTestUnique[n]++;
    
    if(n==joint)
      pos1=i;
    
    if(iSelect>0){				//check back/cross path
      for(int x1=0;x1<iSelect;x1++){		//for each select1 node
	int k=selectNodes2[x1];
	if(n==selectNodes2[x1]){
	  selectNodePair[x1]=i;
	  vTestUnique[n]=0;
	}
      }
    }
    
  }
  
  //if merge path, simply join at p1:size-1 & p2:0
  //if join children, join at parent node
  if(joint>-1){
    vTestUnique[joint]=0;
  }else{
    vTestUnique[p1[lenP1-1]]=0;
  }
  
	
  int tmpselectcount=0;
  //below is the same as path1
  for(int j=0;j<lenP2;j++) {
    int n=p2[j];
    //(*retP)[j+lenP1]=n;		//new solution
    
    if(vTestUnique[n]>0)
      vTestUnique[n]++;
    
    nodepos[n]=j+lenP1;	//modify the current position correspoing to (P1+P2)
    
    if(iSelect>0){				//check select node
      if(n==joint)	//not push_back the joint node
	pos2=j;
      else
	(*retP).push_back(n);
      
      for(int x1=0;x1<iSelect;x1++){		//for each select1 node
	if(n==selectNodes2[x1]){
	  int k=selectNodes2[x1];
	  int inode=selectNodePair[x1];
	  if(inode>-1){		//back/cross path						
	    
	    selectNodePair[x1+iSelect]=j+lenP1-1-tmpselectcount;	//normal case
	    
	    selectPairResult[tmpselectcount]=selectNodePair[x1];
	    selectPairResult[tmpselectcount+iSelect]=j+lenP1-1-tmpselectcount;
	    
	    tmpselectcount++;
	    //selectNodePair[x1+iSelect]=j;
	    (*retP).pop_back();
	  }
	  vTestUnique[n]=0;
	}
      }
      
    }else{
      if(n==joint)
	pos2=j;
      else
	(*retP).push_back(n);
    }
    
    
  }
  
  if(joint>-1){
    vTestUnique[joint]=0;
    outputIntersect(pw1,pw2,pos1,pos2);
  }else{
    vTestUnique[p2[0]]=0;
    outputIntersect(pw1,pw2,lenP1-1,0);
  }
  
  pathWithIndex* pIndex= new pathWithIndex(retP,iTableCount-1);
  
  for(int r=0;r<iSelect;r++){		//output selection
    if(selectPairResult[r]!=-1 && selectPairResult[r+iSelect]!=-1){	//involved in the join paths... output it
      outputSelect(selectPairResult[r],selectPairResult[r+iSelect]);
    }
  }
  
  //if(iTableCount%2==0)
  outputTestUnique(*pIndex,lenP1,tmpselectcount);
  
  pIndex->setIndex(iTableCount-1);
  
  return pIndex;
}

template <class GraphADT>
void vDFS_GraphGrep<GraphADT>::dfsR(Edge e){
  int w=e.w; 
  //if(e.v!=e.w)
  //show(" tree",e);
  
  pathstack.push_back(w);
  
  pre[w] =cnt++; depth++;
  
  typename GraphADT::adjIterator A(G,w);
  
  int ichild=0;	

  for(int t=A.beg();!A.end();t=A.nxt()) {
    if(t!=-1){	//there's a path
      Edge x(w,t);
      
      G.remove(x);	//marked visted paths
      ichild++;
      curLP++;
      
      if(curLP>=(LP+1)){
	pathstack.push_back(w);
	createPath(curLP-1,0,1);
	cut++;
      }
      
      if(ichild>1){		//has more than 1 child in this node, push parent node into stack
				//show("=========================");
	pathstack.push_back(w);
      }
      
      if(pre[t]==-1){					//normal node in the path 			
	//show(" tree",x);
	
      }else if(post[t]== -1){		
	//show(" back path(select)",x);	//select point
	SelectCount++;
	selectNode.push_back(t);
	
      }else if(pre[t] >pre[w]){	//shouldn't happen, already remove all visited paths
				//show(" down",x);
      }else{					//cross path...select point
	int pret=pre[t];
	int postt=post[t];
	int prew=pre[w];
	int postw=post[w];
	
	//show(" cross",x);
	
	selectNode.push_back(t);
      }
      
      dfsR(x);		//no matter what ... keep going down
      
    }
  }
  
  if(ichild==0 && curLP>=1){	//leaf
    
    if(cut>0){
      createPath(curLP,-1,0);
    }else{
      createPath(curLP,0,0);
    }
    curLP=0;
    cut=0;
  }
  
  
  post[w]=cntP++; depth--;
  
}

//Constructor
// Params: 1. GraphADT Object 
//         2. Lengthpath 
template <class GraphADT>
vDFS_GraphGrep<GraphADT>::vDFS_GraphGrep(GraphADT &G,int lp):G(G),degree(G),cnt(0),cntP(0),depth(0),pre(G.V(),-1), post(G.V(),-1),curLP(0),paths(0),LP(lp),cut(0),nodepos(G.V(),-1),vTestUnique(G.V(),-1)
,iTableCount(0),iTotalNode(0),isNewTree(false),CartisianCount(0),fp(NUM_INTS),SQ(0) {
  fout.open("opfile");
  IntersectCount=0;
  SelectCount=0;
  
  for(int v=0;v<G.V();v++){	// numero vertici query
    if(pre[v]==-1){
      isNewTree=true;
      dfsR(Edge(v,v));
      if(paths.size()==1 && iTableCount==0){ //Special case, only has 1 path
	checkLoop(*paths[0]);
	outputTestUnique(*paths[0],0,0);
      }
      if(pathstack.size()>0)	//for single node that didn't create inside dfsR()
	createPath(curLP,0,0);
      
      CartisianCount++;
      curLP=0;
    }
  }

  int itmpCartisianCount=CartisianCount;
  //output CartesianProduct
  outputCartesianProduct();
  
  if(!QUIET_MODE){
    cout<<"CartisianCount:"<<itmpCartisianCount<<endl;
    cout<<"SPIT qoutable"<<iTableCount-1<<endl;
    cout<<"#Intersect count:"<<IntersectCount<<endl;
    cout<<"#Select count:"<<SelectCount<<endl;
  }
  
  //cout<<"CartisianCount:"<<itmpCartisianCount<<endl;
  cout<<"nV:"<<G.V()<<"\n";
  cout<<"SQ:"<<SQ <<"\n";
  cout << "numPath: " << paths.size() << "\n";
  
  //outputing the output node position information
  //for(int i=0;i<outorder.size();i++){
  //cout<< outorder[i] << " ";
  //}
  
  fout<<"SPIT qoutable"<<iTableCount-1<<"\n";
  
  fout.close();
}

#endif
