/*
  GraphBlast Algorithm
  graph.h responsible for graph matching
  2006(c) Modified by Diego Reforgiato Recupero
  2007(c) Modified by Dmitry Skripin
*/
#ifndef GRAPH_H
#define GRAPH_H

#include "common.h"
#include "algebra.h"


/////////////////////////////////////////////
//	Graph Related classes
/////////////////////////////////////////////

////////////////////////////////////////////
//
// Class: Edge
// Usage: An Edge contains 2 int nodes and without direction. Used in graph traversal.
//
/////////////////////////////////////////////
class Edge{
public:
  int v,w;
  Edge(int v=-1,int w=-1):v(v), w(w){}
  Edge(string s1,string s2){
    v=atoi(s1.c_str());
    w=atoi(s2.c_str());
  }
  
  bool operator== (Edge e) {
    if(e.v==v && e.w==w)
      return true;
    else if(e.w==v && e.v==w)
      return true;
    
    return false;			
  }	
};

////////////////////////////////////////////
//
// Class: Graph
// Usage: An interface that provide basic functions that a graph should have
//
/////////////////////////////////////////////
class GRAPH{
  private :
 public:
  GRAPH();
  GRAPH(int,bool);
  ~GRAPH();
  int V() const;
  int E() const;
  bool directed() const;
  int insert(Edge);
  int remove(Edge);
  bool edge(int, int);
  const string& getLabel(int v);
  void addNode(int v,const string& name);
  void sort();
  class adjIterator{
  public:
    adjIterator(const GRAPH&, int);
    int beg();
    int nxt();
    bool end();
  };
};

////////////////////////////////////////////
//
// Class: DenseGRAPH
// Usage: An implementation class of Graph using adjcent matrix, usually efficient while graph is densed
//
/////////////////////////////////////////////
class DenseGRAPH{
  int Vcnt,Ecnt;
  bool digraph;
  vector <vector <bool> > adj;
  vector <string> vlabels;
  DenseGRAPH();
  string _name;
  string _graphName;
 public:
  DenseGRAPH(int V,bool digraph=false):adj(V),vlabels(V), Vcnt(V), Ecnt(0), digraph(digraph){
    for(int i=0;i<V;i++)
      adj[i].assign(V,false);
  }
  int V() const{return Vcnt;}
  int E() const{return Ecnt;}
  bool directed() const {return digraph;}
  
  void setName(char* name){_name=string(name);};
  void setName(string name){_name=name;};
  void setGraphName(string name){_graphName=name;};
  string getName(){return _name;}
  string getGraphName(){return _graphName;}
  
  void insert(Edge e){
    int v=e.v, w=e.w;
    if(adj[v][w]==false) Ecnt++;
    adj[v][w]=true;
    if(!digraph) 
      adj[w][v]=true;
  }
  void remove(Edge e){
    int v=e.v,w=e.w;
    if(adj[v][w]==true) Ecnt--;
    adj[v][w]=false;
    if(!digraph) adj[w][v]=false;
    
  }
  
  const string& getLabel(int v){return vlabels[v];}
  void addNode(int v,const string& name){vlabels[v]=string(name);}
  void addNode(int v,const char* name){vlabels[v]=string(name);}
  
  bool edge(int v,int w) const{return adj[v][w];}
  class adjIterator;
  friend class adjIterator;
};

class DenseGRAPH::adjIterator{
  const DenseGRAPH &G;
  int i,v;
 public:
  adjIterator(const DenseGRAPH &G, int v): G(G), v(v), i(-1){}
  int beg(){i=-1; return nxt();}
  int nxt(){
    for(i++;i<G.V();i++)
      if(G.adj[v][i]==true) return i;
    return -1;
  }
  bool end(){return i>=G.V();}
};

////////////////////////////////////////////
//
// Class: SparseMultiGRAPH 
// Usage: An implementation class of Graph using link list, usually efficient while graph is sparse
//
/////////////////////////////////////////////
class SparseMultiGRAPH{
  int Vcnt,Ecnt;
  bool digraph;
  vector <string> vlabels;
  
  struct node{
    int v;
    string label;
    bool visited;
    node* next;
    node(int x,node* t):visited(false),v(x),next(t){}
  };
  
  typedef node* link;
  
  //	vector <link> adj;
  string _name;
  string _graphName;
 public:
  vector <string> vlab() {return vlabels;}
  vector <link> adj;
  SparseMultiGRAPH(int V,bool digraph=false):adj(V),vlabels(V), Vcnt(V), Ecnt(0), digraph(digraph){
    for(int ijt=0;ijt<V;ijt++)
      adj[ijt] = 0;	    
    //			adj.assign(V,0);
  }
  int V() const{return Vcnt;}
  int E() const{return Ecnt;}
  bool directed() const {return digraph;}
  int size() {return vlabels.size();}
  void setName(char* name){_name=string(name);};
  void setName(string name){_name=name;};
  void setGraphName(string name){_graphName=name;};
  string getName(){return _name;}
  string getGraphName(){return _graphName;}
  
  
  link addLink(int n, link t){
    if(!t){		//start from null
      return new node(n,NULL);
    }else{		//have elements already 
      link pre=t;
      link cur=t;	//next elem
      bool isfound=false;
      
      if(cur->v >n){	//insert node is the smaller, insert to first elem
	return new node(n,cur);		
      }
      
      while(cur){
	if(cur->v < n){	//current elm is smaller,go next
	  pre=cur;
	  cur=cur->next;
	}else if(cur->v == n){	//equal node, should not insert again
	  return NULL;
	}else{				//find spot(current is bigger)
	  pre->next=new node(n,cur);
	  return t;
	}
      }
      
      pre->next=new node(n,cur);	//go to the end 
      return t;
    }
  }
  
  void insert(Edge e){
    int v=e.v, w=e.w;
    link tmp=NULL, tmp1=NULL;
    
    tmp=addLink(w,adj[v]);	//add link for w ->v
    if(tmp!=NULL){
      adj[v]=tmp;
      
      if(!digraph){
	tmp1=addLink(v,adj[w]);
	if(tmp1!=NULL)
	  adj[w]=tmp1;
      }
      
      Ecnt++;			
    }
    
  }
  
  void remove(Edge e){
    int v=e.v, w=e.w;
    link t=adj[v];
    link pre=adj[v];
    while(t){	//found all child
      if(t->v==w ){	//found it		
	t->visited=true;
	break;
      }
      pre=t;
      t=t->next;
    }
    
    
    if(!digraph){
      link t1=adj[w];
      link pre1=adj[w];
      while(t1){	//found all child
	if(t1->v==v){	//found it
	  t1->visited=true;
	  break;
	}
	pre1=t1;
	t1=t1->next;
      }
    }	
    
    Ecnt--;
  }
  
  string getLabel(int v){
    return vlabels[v];
  }
  
  void addNode(int v,const string& name){
    vlabels[v]=sep_words(name)[0];
  }
  
  void addNode(int v,const char* name){
    vlabels[v]=sep_words(string(name))[0];
  }
  
  bool edge(int v,int w) const;
  class adjIterator;
  friend class adjIterator;
};

class SparseMultiGRAPH::adjIterator{
  const SparseMultiGRAPH &G;
  int v;
  link t;
 public:
  adjIterator(const SparseMultiGRAPH &G, int v):G(G),v(v){t=0;}
  int beg(){
    t=G.adj[v];
    return (t && !t->visited)? t->v: -1;
  }
  int nxt(){
    link tmp=t;
    if(t) {
      t=t->next;
      while(t&& t->visited){
	t=t->next;
      }
    }
    
    return t? t->v:-1;	
  }
  bool end(){
    return t==0;
  }
};

/***********************************************
!!!!!!!! Here is the GraphADT defination  !!!!!!
************************************************/
typedef SparseMultiGRAPH GraphADT;
//typedef DenseGRAPH GraphADT;

//class PathTable;
//class PathTableWithDegree;

typedef vector<int> pathADT;
typedef map<string,PathTable*> Map_PathTable;
typedef map<string,PathTableWithDegree*> Map_PathTableWithDegree;

////////////////////////////////////////////
//
// Class: pathWithIndex 
// Usage: Store a path found/generated with index information while DFS traverersal in GraphGrep
//
/////////////////////////////////////////////
class pathWithIndex{
  int index;	//Index of this path (qoutable#)
  pathADT* path;	//Contains nodes of this path
  vector<int> selectNodes;	//Node need to be selected in this path
 public:
  pathWithIndex():index(-1),path(NULL){};
  pathWithIndex(pathADT* p):index(-1),path(p){};
  pathWithIndex(pathADT* p,int i):index(i),path(p){};
  ~pathWithIndex(){delete path;}
  
  pathADT* getPath(){	return path;}
  int getIndex(){		return index;}
  void setIndex(int i=1){		index=i;}
  
  void addSelectNode(int node){
    selectNodes.push_back(node);
  }
  vector<int>& getSelectNodes(){return selectNodes;}
  int getSelectCount(){return selectNodes.size();}
};

////////////////////////////////////////////
//
// Class: fingerprint 
// Usage: Fingerprint implementation using Frowns/daylight fingerprint
//
/////////////////////////////////////////////
class fingerprint {
 public:
  ~fingerprint(){
    delete _fp;
  };
  
  fingerprint(int size=256){
    _fp=new unsigned int[size];
    memset(_fp,0,NUM_INTS*sizeof(unsigned int));
  };
  
  unsigned int addPattern(const string& str){
    unsigned int res=0;
    unsigned int const shift=INT_SIZE/((DEFAULT_LP+1)*2);
    unsigned int const mask=~0U<<(INT_SIZE-shift);
    
    string::const_iterator it=str.begin(),
      it_end=str.end();
    while (it!=it_end){
      res= (res&mask) ^ (res<<shift) ^ (*it++);
    }
    frownFingerPrint(res);		
    
    if(!QUIET_MODE){
      cout<< "Pattern:" <<str.c_str()<< " FP:"<<res<<endl;
    }
    
    return res;
  }
  
  unsigned int* getFP(){
    return _fp;
  }
  
 private:
  unsigned int* _fp;
  
  void frownFingerPrint(unsigned int sum) {	  
    unsigned int seed = sum;
    unsigned int index,bit;
    
    srand(seed);
    index = (unsigned int)(( ((double)(rand())/RAND_MAX)*NUM_INTS));
    bit = (unsigned int)(( ((double)(rand())/RAND_MAX)*INT_SIZE));
    
    if(index==NUM_INTS)	index=0;
    if(bit==INT_SIZE)		bit=0;
    
    //cout<<"index:"<<index<<" bit:"<<bit<<endl;
    _fp[index]=_fp[index] | 1<<bit;
  }
};


/////////////////////////////////////////////////////////////
///////////////	Start Template classes		/////////////////
/////////////////////////////////////////////////////////////

////////////////////////////////////////////
//
// Class: IO 
// Usage: Print the GraphADT structure to screen
//
/////////////////////////////////////////////
template <class GraphADT>
class IO{
public:
	static void show(const GraphADT &);
};


template <class GraphADT>
void IO<GraphADT>::show(const GraphADT &G){
  for(int s=0;s<G.V();s++){
    cout.width(2); cout<<s<<":";
    
    typename GraphADT::adjIterator A(G,s);
    for(int t=A.beg();!A.end();t=A.nxt()){
      cout.width(2); cout<<t<<" ";
    }
    cout <<endl;
  }
};

////////////////////////////////////////////
//
// Class: DEGREE
// Usage: Provide restore/retrive the degree information of a GraphADT object
//
/////////////////////////////////////////////
template <class GraphADT> class DEGREE {
  const GraphADT &G;
  vector <int> degree;
 public:
  DEGREE(const GraphADT &G):G(G),degree(G.V(),0){
    for (int v=0;v<G.V();v++){
      typename GraphADT::adjIterator A(G,v);
      for(int w=A.beg();!A.end();w=A.nxt())
	degree[v]++;
    }
  }
  int operator[] (int v) const{return degree[v];}
};


////////////////////////////////////////////
//
// Class: STACK
// Usage: A stack data-structure class implemented using link list
//
/////////////////////////////////////////////
template <class Item>
class STACK{
 private:
  struct node{
    Item item; node* next;
    node(Item x, node* t){
      item=x;
      next=t;
    }		
  };
  typedef node* link;
  link head;
  int count;
 public:
  STACK(int):count(0){head=0;}
  int empty() const{return head==0;}
  void push(Item x){head=new node(x,head);count++;}
  Item pop(){
    Item v=head->item;
    link t=head->next;
    delete head; head=t;
    
    count--;
    return v;
  }
  int size(){
    int i=count;
    return count;
  }
  Item operator[] (int v) const{
    if(v>count) throw GraphGrepExcp("out of bound");
    Item vitem=head->item;
    link t=head->next;
    for(int i=0;i<v;i++){
      vitem=t->item;
      t=t->next;
    }
    return vitem;
  }
};

#endif
