/*****************************************************************************/
/*!
 * \file ml_interface.c
 *
 * Interface file required by OCaml	 						
 * 
 * <hr>
 *
 * License to use, copy, modify, sell and/or distribute this software
 * and its documentation for any purpose is hereby granted without
 * royalty, subject to the terms and conditions defined in the \ref
 * README file provided with this distribution.
 * 
 * <hr>
 * 
 */
/*****************************************************************************/

#include "c_interface.h"
#include <stdio.h>
#include <caml/mlvalues.h>
#include <caml/alloc.h>
#include <caml/memory.h>
#include <caml/fail.h>

/* ---------------------------------------------------------------------- */
/*  Debug                                                                 */
/* ---------------------------------------------------------------------- */

value ml_vc_get_error_string(){
  CAMLparam0();
  CAMLreturn(copy_string(vc_get_error_string()));
}

void ml_reset_error_status(){
   CAMLparam0();
   vc_reset_error_status();
   // CAMLreturn(Val_int(1));*)
   CAMLreturn0;
}


// BEGIN C INTERFACE COPY

value ml_vc_destroyValidityChecker(value vc_v){
  CAMLparam1(vc_v);
  vc_reset_error_status();;
  VC vc = (VC) vc_v;
  vc_destroyValidityChecker(vc);
  CAMLreturn(Val_unit);
}

value ml_vc_createValidityCheckerCVC(value dagify){
  CAMLparam1(dagify);
  vc_reset_error_status();;
  Flags flags = vc_createFlags();
  vc_setBoolFlag(flags, "proofs", 1);
  vc_setBoolFlag(flags, "dagify-exprs", 0);
  //  vc_setBoolFlag(flags, "arith3", 1);
  vc_setBoolFlag(flags, "smart-clauses", 0);
  VC vc = vc_createValidityChecker(flags);
  if(vc_get_error_status() < 0) failwith("ml_vc_createValidityCheckerCVC");
  else CAMLreturn((value) vc);
}

value ml_vc_createValidityCheckerSMT(value dagify){
  CAMLparam1(dagify);
  vc_reset_error_status();;
  Flags flags = vc_createFlags();
  vc_setBoolFlag(flags, "proofs", 1);
  vc_setBoolFlag(flags, "dagify-exprs", 0);
  //  vc_setBoolFlag(flags, "arith3", 1);
  vc_setBoolFlag(flags, "smart-clauses", 0);
  vc_setStringFlag(flags,"lang","smtlib");
  VC vc = vc_createValidityChecker(flags);
  if(vc_get_error_status() < 0) failwith("ml_vc_createValidityCheckerSMT");
  else CAMLreturn((value) vc);
}


value ml_vc_getProofOfFile(value vc_v, value fileName){
  CAMLparam2(vc_v,fileName);
  vc_reset_error_status();
  VC vc = (VC) vc_v;
  Expr r = vc_getProofOfFile(vc,String_val(fileName));
  if(vc_get_error_status() < 0) failwith("ml_vc_getProofOfFile");
  else CAMLreturn((value) r);
}

value ml_vc_getProofAssumptions(value vc_v){
  CAMLparam1(vc_v);
  vc_reset_error_status();
  VC vc = (VC) vc_v;
  Expr r = vc_getProofAssumptions(vc);
  if(vc_get_error_status() < 0) failwith("ml_vc_getProofAssumptions");
  else CAMLreturn((value) r);
}

value ml_vc_getProofQuery(value vc_v){
  CAMLparam1(vc_v);
  vc_reset_error_status();
  VC vc = (VC) vc_v;
  Expr r = vc_getProofQuery(vc);
  if(vc_get_error_status() < 0) failwith("ml_vc_getProofQuery");
  else CAMLreturn((value) r);
}



// Basic Types
value ml_vc_boolType(value vc_v){
  CAMLparam1(vc_v);
  VC vc = (VC) vc_v;
  Type r = vc_boolType(vc);
  if(vc_get_error_status() < 0) failwith("ml_vc_boolType");
  else CAMLreturn ((value) r);
}

value ml_vc_realType(value vc_v){
  CAMLparam1(vc_v);
  VC vc = (VC) vc_v;
  Type r = vc_realType(vc);
  if(vc_get_error_status() < 0) failwith("ml_vc_realType");
  else CAMLreturn ((value) r);
}

value ml_vc_intType(value vc_v){
  CAMLparam1(vc_v);
  VC vc = (VC) vc_v;
  Type r = vc_intType(vc);
  if(vc_get_error_status() < 0) failwith("ml_vc_intType");
  else CAMLreturn ((value) r);
}

// array types
value ml_vc_arrayType(value vc_v, value typeIndex, value typeData){
  CAMLparam3(vc_v,typeIndex,typeData);
  VC vc = (VC) vc_v;
  Type ti = (Type) typeIndex;
  Type td = (Type) typeData;
  Type ta = vc_arrayType(vc,ti,td);
  if(vc_get_error_status() < 0) failwith("ml_vc_arrayType");
  else CAMLreturn((value) ta);
}

// skipping tuple types
// skipping record types

// skipping function types
// skipping user types


/////////////////////////////////////////////////////////////////////////////
// Expr manipulation methods                                               //
/////////////////////////////////////////////////////////////////////////////

// Return the ExprManager
value ml_vc_getEM(value vc_v){
  CAMLparam1(vc_v);
  VC vc = (VC) vc_v;
  ExprManager em = vc_getEM(vc);
  if(vc_get_error_status() < 0) failwith("ml_vc_getEM");
  else CAMLreturn ((value) em);
}

value ml_vc_getKindInt(value vc_v,value kind_name){
  CAMLparam2(vc_v,kind_name);
  VC vc = (VC) vc_v;
  char* name = String_val(kind_name);
  int kind = vc_getKindInt(vc,name);
  if(vc_get_error_status() < 0) failwith("ml_vc_getKindInt");
  else CAMLreturn(Val_int(kind));
}


value ml_vc_getKindString(value vc_v,value kind_v){
  CAMLparam2(vc_v,kind_v);
  VC vc = (VC) vc_v;
  int kind = Int_val(kind_v);
  const char* name = vc_getKindString(vc,kind);
  if(vc_get_error_status() < 0) failwith("ml_vc_getKindString");
  else CAMLreturn(copy_string(name));
}

 
// Create a variable with a given name and type (the type
// cannot be a function type).
value ml_vc_varExpr(value vc_v, value name_v, value type_v){
  CAMLparam3(vc_v,name_v,type_v);
  VC vc = (VC) vc_v;
  char* name = String_val(name_v);
  Type r = (Type) type_v;
  Expr x = vc_varExpr(vc,name,r);
  if(vc_get_error_status() < 0) failwith("ml_vc_varExpr");
  else CAMLreturn((value) x);
}

// Get the expression and type associated with a name.  If there is no such
// Expr, a NULL Expr is returned.
value ml_vc_lookupVar(value vc_v, value name_v, value type_v){
  CAMLparam3(vc_v,name_v,type_v);
  VC vc = (VC) vc_v;
  char* name = String_val(name_v);
  Type t = (Type) type_v;
  Expr e = vc_lookupVar(vc,name,t);
  if(vc_get_error_status() < 0) failwith("ml_vc_lookupVar");
  else CAMLreturn((value) e);
}

// Get the type of the Expr.
value ml_vc_getType(value vc_v,value e_v){
  CAMLparam2(vc_v,e_v); 
  VC vc = (VC) vc_v;
  Expr e = (Expr) e_v;
  Type t = vc_getType(vc,e);
  if(vc_get_error_status() < 0) failwith("ml_vc_getType");
  else CAMLreturn((value) t);  
}

//Get the Op of the Fuction
value ml_vc_getFun(value vc_v,value e_v){
  CAMLparam2(vc_v,e_v); 
  VC vc = (VC) vc_v;
  Expr e = (Expr) e_v;
  Expr t = vc_getFun(vc,e);
  if(vc_get_error_status() < 0) failwith("ml_vc_getFun");
  else CAMLreturn((value) t);  
}

// Create an equality expression.  The two children must have the same type.
value ml_vc_eqExpr(value vc_v, value expr1_v, value expr2_v){
  CAMLparam3(vc_v,expr1_v,expr2_v);
  VC vc = (VC) vc_v;
  Expr expr1 = (Expr) expr1_v;
  Expr expr2 = (Expr) expr2_v;
  Expr expr1_EQ_expr2 = vc_eqExpr(vc,expr1,expr2);
  if(vc_get_error_status() < 0) failwith("ml_vc_eqExpr");
  else CAMLreturn((value) expr1_EQ_expr2);
}

// Boolean expressions

// The following functions create Boolean expressions.  The children provided
// as arguments must be of type Boolean.

value ml_vc_trueExpr(value vc_v){
  CAMLparam1(vc_v);
  VC vc = (VC) vc_v;
  Expr trueExpr = vc_trueExpr(vc);
  if(vc_get_error_status() < 0) failwith("ml_vc_trueExpr");
  else CAMLreturn((value) trueExpr);
}

value ml_vc_falseExpr(value vc_v){
  CAMLparam1(vc_v);
  VC vc = (VC) vc_v;
  Expr falseExpr = vc_falseExpr(vc);
  if(vc_get_error_status() < 0) failwith("ml_vc_falseExpr");
  else CAMLreturn((value) falseExpr);
}
value ml_vc_notExpr(value vc_v, value expr_v){
  CAMLparam2(vc_v,expr_v);
  VC vc = (VC) vc_v;
  Expr expr = (Expr) expr_v;
  Expr notExpr = vc_notExpr(vc,expr);
  if(vc_get_error_status() < 0) failwith("ml_vc_notExpr");
  else CAMLreturn((value) notExpr);
}

value ml_vc_andExpr(value vc_v, value expr1_v, value expr2_v){
  CAMLparam3(vc_v,expr1_v,expr2_v);
  VC vc = (VC) vc_v;
  Expr expr1 = (Expr) expr1_v;
  Expr expr2 = (Expr) expr2_v;
  Expr expr1_AND_expr2 = vc_andExpr(vc,expr1,expr2);
  if(vc_get_error_status() < 0) failwith("ml_vc_andExpr");
  else CAMLreturn((value) expr1_AND_expr2);
}

// skipping andN

value ml_vc_orExpr(value vc_v, value expr1_v, value expr2_v){
  CAMLparam3(vc_v,expr1_v,expr2_v);
  VC vc = (VC) vc_v;
  Expr expr1 = (Expr) expr1_v;
  Expr expr2 = (Expr) expr2_v;
  Expr expr1_OR_expr2 = vc_orExpr(vc,expr1,expr2);
  if(vc_get_error_status() < 0) failwith("ml_vc_orExpr");
  else CAMLreturn((value) expr1_OR_expr2);
}

// skipping orN

value ml_vc_impliesExpr(value vc_v, value expr1_v, value expr2_v){
  CAMLparam3(vc_v,expr1_v,expr2_v);  
  VC vc = (VC) vc_v;
  Expr expr1 = (Expr) expr1_v;
  Expr expr2 = (Expr) expr2_v;
  Expr expr1_IMP_expr2 = vc_impliesExpr(vc,expr1,expr2);
  if(vc_get_error_status() < 0) failwith("ml_vc_impliesExpr");
  else CAMLreturn((value) expr1_IMP_expr2);
}

value ml_vc_iffExpr(value vc_v, value expr1_v, value expr2_v){
  CAMLparam3(vc_v,expr1_v,expr2_v);
  VC vc = (VC) vc_v;
  Expr expr1 = (Expr) expr1_v;
  Expr expr2 = (Expr) expr2_v;
  Expr expr1_IFF_expr2 = vc_iffExpr(vc,expr1,expr2);
  if(vc_get_error_status() < 0) failwith("ml_vc_iffExpr");
  else CAMLreturn((value) expr1_IFF_expr2);
}

value ml_vc_iteExpr(value vc_v, value if_v, value then_v, value else_v){
  CAMLparam4(vc_v,if_v,then_v,else_v);
  VC vc = (VC) vc_v;
  Expr if_e = (Expr) if_v;
  Expr then_e = (Expr) then_v;
  Expr else_e = (Expr) else_v;
  Expr ite_e = vc_iteExpr(vc,if_e,then_e,else_e);
  if(vc_get_error_status() < 0) failwith("ml_vc_iteExpr");
  else CAMLreturn((value) ite_e);
}

// Arithmetic

// Create a rational number with numerator n and denominator d.  d cannot be 0.
value ml_vc_ratExpr(value vc_v,value n_v,value d_v){
  CAMLparam3(vc_v,n_v,d_v);
  VC vc = (VC) vc_v;
  int n = Int_val(n_v);
  int d = Int_val(d_v);
  Expr e = vc_ratExpr(vc,n,d);
  if(vc_get_error_status() < 0) failwith("ml_vc_ratExpr: ");
  else CAMLreturn((value) e);
}

// Same as above, except that n and d are given as separate strings.  They are
// converted to arbitrary-precision integers according to the given base.
value ml_vc_ratExprFromStr(value vc_v,value n_s_v,value d_s_v, value base_v){
  CAMLparam4(vc_v,n_s_v,d_s_v,base_v);
  VC vc = (VC) vc_v;
  char* n = String_val(n_s_v);
  char* d = String_val(d_s_v);
  int base = Int_val(base_v);
  Expr e = vc_ratExprFromStr(vc,n,d,base);
  if(vc_get_error_status() < 0) failwith("ml_vc_ratExpr");
  else CAMLreturn((value) e);
}

// Unary minus.  Child must have real type.
value ml_vc_uminusExpr(value vc_v,value child_v){
  CAMLparam2(vc_v,child_v);
  VC vc = (VC) vc_v;
  Expr child = (Expr) child_v;
  Expr e = vc_uminusExpr(vc,child);
  if(vc_get_error_status() < 0) failwith("ml_vc_uminusExpr");
  else CAMLreturn((value) e);
}


// plus, minus, mult.  Children must have real type.
value ml_vc_plusExpr(value vc_v,value left_v,value right_v){
  CAMLparam3(vc_v,left_v,right_v);
  VC vc = (VC) vc_v;
  Expr left = (Expr) left_v;
  Expr right = (Expr) right_v;
  Expr e = vc_plusExpr(vc,left,right);
  if(vc_get_error_status() < 0) failwith("ml_vc_plusExpr");
  else CAMLreturn((value)e);
}

value ml_vc_minusExpr(value vc_v,value left_v,value right_v){
  CAMLparam3(vc_v,left_v,right_v);
  VC vc = (VC) vc_v;
  Expr left = (Expr) left_v;
  Expr right = (Expr) right_v;
  Expr e = vc_minusExpr(vc,left,right);
  if(vc_get_error_status() < 0) failwith("ml_vc_minusExpr");
  else CAMLreturn((value)e);
}

value ml_vc_multExpr(value vc_v,value left_v,value right_v){
  CAMLparam3(vc_v,left_v,right_v);
  VC vc = (VC) vc_v;
  Expr left = (Expr) left_v;
  Expr right = (Expr) right_v;
  Expr e = vc_multExpr(vc,left,right);
  if(vc_get_error_status() < 0) failwith("ml_vc_multExpr");
  else CAMLreturn((value)e);
}


value ml_vc_powExpr(value vc_v,value pow_v,value base_v){
  CAMLparam3(vc_v,pow_v,base_v);
  VC vc = (VC) vc_v;
  Expr pow = (Expr) pow_v;
  Expr base = (Expr) base_v;
  Expr e = vc_powExpr(vc,pow,base);
  if(vc_get_error_status() < 0) failwith("ml_vc_powExpr");
  else CAMLreturn((value)e);
}

value ml_vc_divideExpr(value vc_v,value numerator_v,value denominator_v){
  CAMLparam3(vc_v,numerator_v,denominator_v);
  VC vc = (VC) vc_v;
  Expr num = (Expr) numerator_v;
  Expr denom = (Expr) denominator_v;
  Expr e = vc_powExpr(vc,num,denom);
  if(vc_get_error_status() < 0) failwith("ml_vc_divideExpr");
  else CAMLreturn((value)e);
}



// The following functions create less-than, less-than or equal,
// greater-than, and greater-than or equal expressions of type Boolean.
// Their arguments must be of type Real.

value ml_vc_ltExpr(value vc_v,value left_v,value right_v){
  CAMLparam3(vc_v,left_v,right_v);
  VC vc = (VC) vc_v;
  Expr left = (Expr) left_v;
  Expr right = (Expr) right_v;
  Expr e = vc_ltExpr(vc,left,right);
  if(vc_get_error_status() < 0) failwith("ml_vc_ltExpr");
  else CAMLreturn((value)e);
}

value ml_vc_leExpr(value vc_v,value left_v,value right_v){
  CAMLparam3(vc_v,left_v,right_v);
  VC vc = (VC) vc_v;
  Expr left = (Expr) left_v;
  Expr right = (Expr) right_v;
  Expr e = vc_leExpr(vc,left,right);
  if(vc_get_error_status() < 0) failwith("ml_vc_leExpr");
  else CAMLreturn((value)e);
}

value ml_vc_gtExpr(value vc_v,value left_v,value right_v){
  CAMLparam3(vc_v,left_v,right_v);
  VC vc = (VC) vc_v;
  Expr left = (Expr) left_v;
  Expr right = (Expr) right_v;
  Expr e = vc_gtExpr(vc,left,right);
  if(vc_get_error_status() < 0) failwith("ml_vc_gtExpr");
  else CAMLreturn((value)e);
}

value ml_vc_geExpr(value vc_v,value left_v,value right_v){
  CAMLparam3(vc_v,left_v,right_v);
  VC vc = (VC) vc_v;
  Expr left = (Expr) left_v;
  Expr right = (Expr) right_v;
  Expr e = vc_geExpr(vc,left,right);
  if(vc_get_error_status() < 0) failwith("ml_vc_geExpr");
  else CAMLreturn((value)e);
}

// arrays
value ml_vc_readExpr(value vc_v,value array_v, value index_v){
  CAMLparam3(vc_v,array_v,index_v);
  VC vc = (VC) vc_v;
  Expr array = (Expr) array_v;
  Expr index = (Expr) index_v;
  Expr read_expr = vc_readExpr(vc,array,index);
  if(vc_get_error_status() < 0) failwith("ml_vc_readExpr");
  else CAMLreturn((value) read_expr);
}

value ml_vc_writeExpr(value vc_v,value array_v, 
                      value index_v,value newValue_v){
  CAMLparam4(vc_v,array_v,index_v,newValue_v);
  VC vc = (VC) vc_v;
  Expr array = (Expr) array_v;
  Expr index = (Expr) index_v;
  Expr newValue = (Expr) newValue_v;
  Expr write_expr = vc_writeExpr(vc,array,index,newValue);
  if(vc_get_error_status() < 0) failwith("ml_vc_writeExpr");
  else CAMLreturn((value) write_expr);
}

// skipping records


// skipping print

/////////////////////////////////////////////////////////////////////////////
// Context-related methods                                                 //
/////////////////////////////////////////////////////////////////////////////


// Assert a new formula in the current context.  The formula must have
// Boolean type.
value ml_vc_assertFormula(value vc_v,value e_v){
  CAMLparam2(vc_v,e_v);
  VC vc = (VC) vc_v; 
  Expr e = (Expr) e_v;
  vc_assertFormula(vc,e);
  if(vc_get_error_status() < 0) failwith("ml_vc_assertFormula");
  else CAMLreturn(Val_unit);
} 
  

// Simplify e with respect to the current context
value ml_vc_simplify(value vc_v,value e_v){
  CAMLparam2(vc_v,e_v);
  VC vc = (VC) vc_v; 
  Expr e = (Expr) e_v; 
  Expr ret = vc_simplify(vc,e);
  if(vc_get_error_status() < 0) failwith("ml_vc_simplify");
  else CAMLreturn((value)ret); 
}

// Check validity of e in the current context.  If the result is true, then
// the resulting context is the same as the starting context.  If the result
// is false, then the resulting context is a context in which e is false.  e
// must have Boolean type.
value ml_vc_query(value vc_v, value expr_v){
  CAMLparam2(vc_v,expr_v);
  VC vc = (VC) vc_v;
  Expr expr = (Expr) expr_v;
  vc_query(vc, expr);
  if(vc_get_error_status() < 0) failwith("ml_vc_query");
  else CAMLreturn(Val_int(1));  
}

// skip counterexample
// skip inconsistent

value ml_vc_getProof(value vc_v){
  CAMLparam1(vc_v);  
  VC vc = (VC) vc_v;
  Expr p = vc_getProof(vc);
  if(vc_get_error_status() < 0) failwith("ml_vc_getProof");
  else CAMLreturn((value)p);
}

// skip push,pop,scopelevel

/* ====================================================================== */
/*  Disorganized                                                          */
/* ====================================================================== */


value ml_expr_toString(value expr_v){
  CAMLparam1(expr_v);
  Expr expr = (Expr) expr_v;
  char* s = vc_exprString(expr);
  if(vc_get_error_status() < 0) failwith("ml_expr_toString");
  else CAMLreturn(copy_string(s));
}

value ml_type_toString(value type_v){
  CAMLparam1(type_v);
  Type type = (Type) type_v;
  char* s = vc_typeString(type);
  if(vc_get_error_status() < 0) failwith("ml_type_toString");
  else CAMLreturn(copy_string(s));
}

value ml_expr_arity(value expr_v){
  CAMLparam1(expr_v);
  Expr expr = (Expr) expr_v;
  int n = vc_arity(expr);
  if(vc_get_error_status() < 0) failwith("ml_expr_vc_arity");
  else CAMLreturn(Val_int(n));
}

value ml_expr_getKind(value expr_v){
  CAMLparam1(expr_v);
  Expr expr = (Expr) expr_v;
  int n = vc_getKind(expr);
  if(vc_get_error_status() < 0) failwith("ml_expr_getKind");
  else CAMLreturn(Val_int(n));
}

value ml_expr_child(value expr_v, value i_v){
  CAMLparam2(expr_v,i_v);
  Expr expr = (Expr) expr_v;
  int n = vc_arity(expr);
  int i = Int_val(i_v);
  Expr child;
  if(i<n){
    child = vc_getChild(expr,i);
  } else {
    printf("CHILD ARG OUT OF RANGE: vc_arity=%d -- arg=%d",n,i);
    failwith("ml_expr_child");
  }
  if(vc_get_error_status() < 0) failwith("ml_expr_child");
  else CAMLreturn((value)child);
}

value ml_expr_isClosure(value expr_v){
  CAMLparam1(expr_v);
  Expr expr = (Expr) expr_v;
  int b = vc_isClosure(expr);
  if(vc_get_error_status() < 0) failwith("ml_expr_isClosure");
  else CAMLreturn(Val_bool(b));
}

value ml_expr_isQuantifier(value expr_v){
  CAMLparam1(expr_v);  
  Expr expr = (Expr) expr_v;
  int b = vc_isQuantifier(expr);
  if(vc_get_error_status() < 0) failwith("ml_expr_isQuantifier");
  else CAMLreturn(Val_bool(b));
}

value ml_expr_isLambda(value expr_v){
  CAMLparam1(expr_v);
  Expr expr = (Expr) expr_v;
  int b = vc_isLambda(expr);
  if(vc_get_error_status() < 0) failwith("ml_expr_isLambda");
  else CAMLreturn(Val_bool(b));
}

value ml_expr_getNumVars(value expr_v){
  CAMLparam1(expr_v);
  Expr expr = (Expr) expr_v;
  int i = (int)vc_getNumVars(expr);
  if(vc_get_error_status() < 0) failwith("ml_expr_getNumVars");
  else CAMLreturn(Val_int(i));
}

value ml_expr_getVar(value expr_v, value i_v){
  CAMLparam2(expr_v,i_v);
  Expr expr = (Expr) expr_v;
  int i = Int_val(i_v);
  Expr var = vc_getVar(expr,i);
  if(vc_get_error_status() < 0) failwith("ml_expr_getVar");
  else CAMLreturn((value)var);
}

value ml_expr_getBody(value expr_v){
  CAMLparam1(expr_v);
  Expr expr = (Expr) expr_v;  
  Expr body = vc_getBody(expr);
  if(vc_get_error_status() < 0) failwith("ml_expr_getBody");
  else CAMLreturn((value) body);
}

value ml_expr_getExistential(value expr_v){
  CAMLparam1(expr_v);
  Expr expr = (Expr) expr_v;  
  Expr body = vc_getExistential(expr);
  if(vc_get_error_status() < 0) failwith("ml_expr_getExistential");
  else CAMLreturn((value) body);
}


value ml_type_toExpr(value type_v){
  CAMLparam1(type_v);
  Type t = (Type) type_v;
  if(vc_get_error_status() < 0) failwith("ml_type_toExpr");
  else CAMLreturn((value) vc_toExpr(t));
}

value ml_expr_isVar(value expr_v){
  CAMLparam1(expr_v);
  Expr e = (Expr) expr_v;
  if(vc_get_error_status() < 0) failwith("ml_expr_isVar");
  else CAMLreturn(Val_bool(vc_isVar(e)));
}


value ml_expr_getInt(value expr_v){
  CAMLparam1(expr_v);
  Expr expr = (Expr) expr_v;
  int i = vc_getInt(expr);
  if(vc_get_error_status() < 0) failwith("ml_expr_getInt");
  else CAMLreturn(Val_int(i));
}

value ml_expr_compare(value expr1_v,value expr2_v){
  CAMLparam2(expr1_v,expr2_v);
  Expr expr1 = (Expr) expr1_v;
  Expr expr2 = (Expr) expr2_v;
  int i = vc_compare_exprs(expr1,expr2);
  if(vc_get_error_status() < 0) failwith("ml_expr_compare");
  else CAMLreturn(Val_int(i));
}


//value ml_getExprAttr(value expr_v,value int_v){
//  CAMLparam2(expr_v,int_v);
//  Expr expr = (Expr) expr_v;
//  int i = Int_val(int_v); 
//  if(vc_get_error_status() < 0) failwith("ml_getExprAttr");
//  else CAMLreturn((value) vc_getExprAttr(expr,i));
//}
