function [diagnostic,R] = solveconstraints(F, X, fi, radius, solver)
% [diagnostic,R] = solveconstraints(F, X[, fi, radius, solver])
%   solves the set F of constraints in the unkown ranking matrix X 
%   and returns its value R with optional input argument fi = 'float' 
%   ['int'], the rigid feasibility radius [inf is none], and the LMI 
%   solver ['lmilab']. Returns a feasibility diagnostic and the 
%   result matrix R.
if nargin < 3, fi = 'float'; end; % real ranking function
if nargin < 4, radius = inf; end;  % no rigid feasibility radius
if nargin < 5, solver = 'lmilab'; end; % default solver
% yalmip note: The >= constraints are any to the solver without any
% modification, and may be treated strictly by some solvers making no
% difference betwwen >= and >. With > and constant shift on, the
% > constraints are relaxed or tightened according to the sign of the
% shift.
if nargin < 5, error('missing argument.'); end;
n = size(X,1)-1;
F % display the constraints
if fi(1:3) == 'int' % choose solver
  solver = 'bnb'; ans = solvesdp(F,[],sdpsettings('solver',...
                                     solver,'savesolveroutput',1));
  correct = (ans.problem == 0);
else
  t = sdpvar(1,1);
  if size(solver,2)>=3 & solver(1:3) == 'lmi'
    if (radius == inf), radius = -1; end;
    ans = solvesdp(F,-t,sdpsettings('solver','lmilab',...
      'lmilab.feasradius',radius,'shift',t));
    [primal,dual] = checkset(F);
    correct = (ans.problem == 0) & (all(primal >= -1e-8) | ...
              all(dual >= -1e-8));
  else if size(solver,2)>=4 & solver(1:4) == 'sdpa'
    ans = solvesdp(F,-t,sdpsettings('solver','sdpa',...
      'radius',radius,'shift',t,'savesolveroutput',1));
    display(ans);
    correct = (ans.problem == 0);
  else if size(solver,2)>=4 & solver(1:4) == 'dsdp'
    ans = solvesdp(F,-t,sdpsettings('solver','dsdp',...
      'dsdp.maxtrustradius',radius,'shift',t,'savesolveroutput',1));
    correct = (ans.problem == 0);
  else if size(solver,2)>=4 & solver(1:4) == 'csdp'
    ans = solvesdp(F,[],sdpsettings('solver','csdp',...
      'radius',radius,'savesolveroutput',1));
    correct = (ans.problem == 0);
  else if size(solver,2)>=5 & solver(1:5) == 'sdpt3'
    ans = solvesdp(F,[],sdpsettings('solver','sdpt3'));
    display(ans);
    correct = (ans.problem == 0);
  else if size(solver,2)>=5 & solver(1:5) == 'sdplr'
    ans = solvesdp(F,-t,sdpsettings('solver','sdplr',...
      'radius',radius,'shift',t,'savesolveroutput',1));
    display(ans.solveroutput.info(:));
    correct = (ans.problem == 0);
  else if size(solver,2)>=5 & solver(1:5) == 'mosek'
    ans = solvesdp(F,-t,sdpsettings('solver','mosek',...
      'radius',radius,'shift',t,'savesolveroutput',1));
    display(ans);
    correct = (ans.problem == 0);
  else if  size(solver,2)>=5 & solver(1:5) == 'sdpt3'
    ans = solvesdp(F,-t,sdpsettings('solver','sdpt3',...
      'radius',radius,'shift',t,'savesolveroutput',1));
    display(ans(:));
    correct = (ans.problem == 0);
  else if  size(solver,2)>=6 & solver(1:6) == 'penbmi'
    ans = solvesdp(F,-t,sdpsettings('solver','penbmi',...
      'radius',radius,'shift',t,'savesolveroutput',1));
    correct = (ans.problem == 0);
  else
    ans = solvesdp(F,-t,sdpsettings('solver',solver,...
      'radius',radius,'shift',t,'savesolveroutput',1));
    correct = (ans.problem == 0) & ((ans.solveroutput.info.pinf...
                        == 0) | (ans.solveroutput.info.dinf == 0));
  end;end;end;end;end;end;end;end;end;
end;
if correct
  diagnostic = ['feasible (' solver ')']; R = double(X);
else
  [primal,dual] = checkset(F);
  disp('Primal residuals:'); disp(primal); 
  disp('Dual residuals:'); disp(dual); 
  diagnostic = ['infeasible problem  (' solver ')']; 
  R = zeros(n+1,n+1);
 end;
 
%******************************************************************************
% Copyright 2004  Patrick.Cousot@ens.fr
%******************************************************************************
% Permission to use, copy, modify, and distribute this software and its
% documentation for any purpose and without fee is hereby granted,
% provided that the above copyright notice appear in all copies and that
% both that the copyright notice and warranty disclaimer appear in
% supporting documentation.
% 
% P. Cousot disclaims all warranties with regard to this software,
% including all implied warranties of merchantability and fitness.  In
% no event shall P. Cousot be liable for any special, indirect or
% consequential damages or any damages whatsoever resulting from loss of
% use, data or profits, whether in an action of contract, negligence or
% other tortuous action, arising out of or in connection with the use or
% performance of this software.
%******************************************************************************
