
package lecture03;

import java.util.Date;

/**
 * An array-based implementation of the PrintJobQueue interface.
 * @author Joanna Klukowska
 * @version Feb 10, 2014 
 *
 */
public class PrintJobQueueArray implements PrintJobQueue {
	
	protected final static int DEFAULT_CAPACITY = 16;
	protected PrintJob [] queue;
	protected int capacity;
	protected int size;
	protected int front;
	protected int back;
	
	

	/**
	 * Construct a PrintJobQueueArray object with a specified capacity 
	 * (capacity allows to predetermine the initial size 
	 * of array used for storage of the queue).
	 * @param capacity
	 *   initial capacity of an array to be used to store 
	 *   the queue
	 */
	public PrintJobQueueArray(int capacity) {
		this.capacity = capacity;
		queue = new PrintJob [capacity];
		front = 0;
		back = capacity -1; //this way the next item entered will
		                    //wrap to the zero location
	}
	
	/**
	 * Construct a PrintJobQueueArray object. 
	 */
	public PrintJobQueueArray() {
		this(DEFAULT_CAPACITY);
	}



	/* (non-Javadoc)
	 * @see lecture05_adt.PrintJobQueue#enqueue(lecture05_adt.PrintJob)
	 */
	@Override
	public void enqueue(PrintJob item) {
		//make sure we are not adding a null job
		if (item == null )
			return;
		//check if the queue is big enough
		//to add another element
		if ( size == capacity ) 
			makeLarger();
		//advance back to point to the next availabe spot
		back = (back+1)%capacity;
		queue[back] = item;
		size++;
	}

	/* (non-Javadoc)
	 * @see lecture05_adt.PrintJobQueue#dequeue()
	 */
	@Override
	public PrintJob dequeue() {
		if (size == 0)
			return null;
		PrintJob nextJob = queue[front];
		//optionally we could set the removed location
		//to null:
		//  queue[front] = null;
		front = (front+1)%capacity;
		size--;
		return nextJob;
	}
	
	/* (non-Javadoc)
	 * @see lecture05_adt.PrintJobQueue#peak()
	 */
	@Override
	public PrintJob peek() {
		if (size == 0)
			return null;
		PrintJob nextJob = queue[front];
		return nextJob;
	}
	
	
	/*
	 * Allocates an array twice the size of the current array used for 
	 * storing the queue and copies all the data to the new array.
	 */
	private void makeLarger ()
	{
		//allocate larger array
		PrintJob [] newQueue = new PrintJob [capacity * 2 ];
		//copy the data over to the new array
		int current = front;
		for (int i = 0; i < capacity; i++)
		{
			newQueue[i] = queue[current];
			current = (current+1) % capacity;
		}
		//reset list reference to the new array
		queue = newQueue;
		//reset the capacity to the new value
		capacity = 2*capacity;
		//reset front and back
		front = 0;
		back = size-1;
		
	}



	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuffer queueString = new StringBuffer();
		int current;
		for (int i = 0; i < size; i++ ) {
			current = (front+i)%capacity;
			queueString.append( String.format("%10s, %3d/%3d pages, %20s %n",
			                      queue[current].getUser(), 
			                      queue[current].getNumPagesLeft(),
			                      queue[current].getNumOfPages(),
			                      new Date( queue[current].getTimeReceived() )   ) );
		}
		
		return queueString.toString(); 
	}


	
	

}
