package Jet.Refres;

import java.util.*;
import java.io.*;

import Jet.JetTest;
import Jet.Control;
import Jet.Console;
import Jet.Pat.Pat;
import Jet.Lisp.*;
import Jet.Tipster.*;
import Jet.Chunk.Chunker;

import AceJet.Gazetteer;
import AceJet.WordNetInterface;

/**
 *  evaluate reference resolution procedure against a key file
 *  with references annotated.
 */

public class CorefEval {
	
	static final String home =
	    "C:/Documents and Settings/Ralph Grishman/My Documents/";
	static final String ACEdir = home + "ACE/";
	// static final String collection = ACEdir + "training nwire sgm 10.txt";
	// static final String keyCollection = ACEdir + "training nwire coref 10.txt";
	// static final String parseCollection = ACEdir + "parses/training nwire 10 parses.txt";
	static final String collection = ACEdir + "training nwire sgm.txt";
	static final String keyCollection = ACEdir + "training nwire coref.txt";
	static final String parseCollection = ACEdir + "parses/training nwire parses.txt";
	// static final String collection = ACEdir + "training nwire sgm.txt";
	// static final String keyCollection = ACEdir + "training nwire coref.txt";
	// static final String collection = "acedata/training nwire sgm.txt";
	// static final String keyCollection = "acedata/training nwire coref.txt";
	static final String baselineCollection = home + "jet temp/coref/coref baseline.txt";
	static final boolean writeBaseline = false;
	static final boolean compareToBaseline = false;
	static final boolean useParser = true;
	static final boolean useParseCollection = true;
	
	public static void main (String[] args) throws IOException {
		// initialize WordNet
		WordNetInterface.initialize();
		// initialize Jet
		System.out.println("Starting ACE Jet...");
		if (useParser) {
			if (useParseCollection) {
				JetTest.initializeFromConfig("props/ace use parses.properties");
			} else {
				JetTest.initializeFromConfig("props/ace parser.properties");
			}
		} else {
			JetTest.initializeFromConfig("props/ME ace.properties");
		}
		new Console();
		// load ACE type dictionary and gazetteer
		AceJet.EDTtype.readTypeDict();
		AceJet.Ace.gazetteer = new Gazetteer();
		AceJet.Ace.gazetteer.load();
		// turn off traces
		Pat.trace = false;
		Resolve.trace = false;
		// open text/parse and key collections
		DocumentCollection col;
		if (useParser & useParseCollection) {
			col = new DocumentCollection(parseCollection);
		} else {
			col = new DocumentCollection(collection);
		}
		DocumentCollection keyCol = new DocumentCollection(
			compareToBaseline ? baselineCollection : keyCollection);
		col.open();
		keyCol.open();
		CorefScorer scorer = new CorefScorer();
		for (int docCount = 0; docCount < col.size(); docCount++) {
			// process file 'currentDoc'
			ExternalDocument doc = col.get(docCount);
			System.out.println ("\nProcessing document " + docCount + ": " + doc.fileName());
			Console.println ("\nProcessing document " + docCount + ": " + doc.fileName());
			// read test document
			doc.setAllTags(true);
			doc.open();
			// process document
			AceJet.Ace.monocase = AceJet.Ace.allLowerCase(doc);
			Control.processDocument (doc, null, docCount == -1, docCount);
			AceJet.Ace.tagReciprocalRelations (doc);
			// remove temporarily -- SuperResolve not part of Jet project
			// SuperResolve.resolveEntities(doc);
			// read coref key
			ExternalDocument keyDoc = keyCol.get(docCount);
			keyDoc.setAllTags(true);
			keyDoc.open();
			// create 'entity' annotations for key
			buildEntitiesFromMentions(keyDoc);
			// score document
			scorer.score(doc, keyDoc);
			System.out.println ("Recall = " + scorer.recall);
			System.out.println ("Precision = " + scorer.precision);
			System.out.println (scorer.mappedMentionCount + " mapped mentions");
			if (writeBaseline) {
				buildMentionsFromEntities(doc);
				doc.removeAnnotationsOfType ("token");
				doc.removeAnnotationsOfType ("constit");
				doc.removeAnnotationsOfType ("tagger");
				doc.removeAnnotationsOfType ("ENAMEX");
				doc.removeAnnotationsOfType ("entity");
				doc.removeAnnotationsOfType ("ng");
			}
		}		
		System.out.println ("");
		System.out.println ("Overall Recall = " + scorer.overallRecall);
		System.out.println ("Overall Precision = " + scorer.overallPrecision);
		float FO = 2.f / (1.f / scorer.overallRecall + 1.f / scorer.overallPrecision);
		System.out.println ("F = " + FO);
		System.out.println (scorer.overallMappedMentionCount + " total mapped mentions");
		if (writeBaseline) {
			col.saveAs (baselineCollection);
		} else {
			CorefCompare.compareCollections(col, keyCol);
			new CollectionView (col);
		}
	}
	
	/**
	 *  command-line callable coreference evaluation (invoked by jet -CorefEval).
	 *  Passed an array of two file names:  the system response collection and
	 *  the key collection.
	 **/  

	public static void task (String[] args) {
		if (args.length != 3) {
			System.out.println 
			  ("CorefEval requires 2 arguments: jet -CorefEval <response collection> <key collection>");
			System.exit(1);
		}
		// open text and key collections
		String collection = args[1]; // ACEdir + "training nwire bad coref 10.txt";
		String keyCollection = args[2]; // ACEdir + "training nwire coref 10.txt";
		DocumentCollection col = new DocumentCollection(collection);
		DocumentCollection keyCol = new DocumentCollection(keyCollection);
		col.open();
		keyCol.open();
		CorefScorer scorer = new CorefScorer();
		for (int docCount = 0; docCount < col.size(); docCount++) {
			// process file 'currentDoc'
			// if (docCount > 1) continue;
			ExternalDocument doc = col.get(docCount);
			System.out.println ("\nProcessing document " + docCount + ": " + doc.fileName());
			// read test document
			doc.setAllTags(true);
			doc.open();
			// read coref key
			ExternalDocument keyDoc = keyCol.get(docCount);
			keyDoc.setAllTags(true);
			keyDoc.open();
			// create 'entity' annotations
			buildEntitiesFromMentions(doc);
			buildEntitiesFromMentions(keyDoc);
			// score document
			scorer.score(doc, keyDoc);
			System.out.println ("Recall = " + scorer.recall);
			System.out.println ("Precision = " + scorer.precision);
			System.out.println (scorer.mappedMentionCount + " mapped mentions");		
		}		
		System.out.println ("");
		System.out.println ("Overall Recall = " + scorer.overallRecall);
		System.out.println ("Overall Precision = " + scorer.overallPrecision);
		float FO = 2.f / (1.f / scorer.overallRecall + 1.f / scorer.overallPrecision);
		System.out.println ("F = " + FO);
		System.out.println (scorer.overallMappedMentionCount + " total mapped mentions");
	}	
	
	/**
	 *  buildEntitiesFromMentions takes a Document with annotations 
	 *  of the form <br>
	 *      mention entity=entity-ID                                <br>
	 *  where coreferential entities share the same entity-ID, and generates
	 *  annotations of the form                                     <br>
	 *      entity mentions=Vector(mentions)                        <br>
	 */

	public static void buildEntitiesFromMentions (Document doc) {
		HashMap entityIdToMentions = new HashMap();
		Vector mentions = doc.annotationsOfType("mention");
		for (int i=0; i<mentions.size(); i++) {
			Annotation mention = (Annotation) mentions.get(i);
			Object entityId = mention.get("entity");
			if (entityId == null) {
				System.out.println ("CorefEval.buildEntities:  mention annotation without entity id");
			} else {
				Vector mentionList = (Vector) entityIdToMentions.get(entityId);
				if (mentionList == null)
					mentionList = new Vector();
				mentionList.add(mention);
				entityIdToMentions.put(entityId, mentionList);
			}
		}
		Set entityIds = entityIdToMentions.keySet();
		Iterator it = entityIds.iterator();
		while (it.hasNext()) {
			String entityId = (String) it.next();
			Vector mentionList = (Vector) entityIdToMentions.get(entityId);
			Annotation firstMention = (Annotation) mentionList.get(0);
			doc.annotate("entity", firstMention.span(), new FeatureSet("mentions", mentionList));
		}
	}

  /**
	 *  buildEntitiesFromLinkedMentions takes a Document with annotations 
	 *  following the MUC standard <br>
	 *      coref id=mention-ID ref=prior-mention-ID                <br>
	 *  where coreferential entities share the same entity-ID, and generates
	 *  annotations of the form                                     <br>
	 *      entity mentions=Vector(mentions)                        <br>
	 */
	 
	private static HashMap idToMention, mentionToEntity;
	
	public static void buildEntitiesFromLinkedMentions (Document doc) {
		Vector corefs = doc.annotationsOfType("coref");
		if (corefs == null) return;
		idToMention = new HashMap();
		for (int i=0; i<corefs.size(); i++) {
			Annotation coref = (Annotation) corefs.get(i);
			Object id = coref.get("id");
			if (id == null) {
				System.out.println ("(buildEntitiesFromRefs) coref annotation without id");
			} else {
				idToMention.put(id, coref);
			}
		}
		mentionToEntity = new HashMap();
		for (int i=0; i<corefs.size(); i++) {
			Annotation mention = (Annotation) corefs.get(i);
			mentionToEntity(mention, doc, 0);
		}
	}
	
	private static Annotation mentionToEntity (Annotation mention, Document doc, int level) {
		if (level > 100) {
			System.out.println ("(mentionToEntity) loop of REF pointers");
			return null;
		}
		Annotation entity = (Annotation) mentionToEntity.get(mention);
		if (entity != null)
			return entity;
		String ref = (String) mention.get("ref");
		if (ref == null) {
			Vector v = new Vector();
			v.add(mention);
			entity = 
			    new Annotation("entity", mention.span(), new FeatureSet ("mentions", v));
			doc.addAnnotation(entity);
		} else {
			Annotation referent = (Annotation) idToMention.get(ref);
			if (referent == null)
				System.out.println ("(mentionToEntity) undefined REF pointer");
			entity = mentionToEntity (referent, doc, level+1);
			Vector v = (Vector) entity.get("mentions");
			v.add(mention);
		}
		mentionToEntity.put(mention, entity);
		return entity;
	}
	
	/**
	 *  buildMentionsFromEntities takes a Document with coreference information 
	 *  in the form of mention attributes on entities and generates Annotations
	 *  of the form <br>
	 *      mention entity=entity-ID                                <br> 
	 *  over the <I>heads</I> of mentions, where coreferential mentions are
	 *  linked by having the same entity-ID.
	 */

	public static void buildMentionsFromEntities (Document doc) {
		Vector entities = doc.annotationsOfType("entity");
		for (int i=0; i<entities.size(); i++) {
			Annotation entity = (Annotation) entities.get(i);
			Integer entityID = new Integer(i);
			Vector mentions = (Vector) entity.get("mentions");
			for (int j=0; j<mentions.size(); j++) {
				Annotation mention = (Annotation) mentions.get(j);
				Annotation mentionHead = Resolve.getHeadC(mention);
				if (mentionHead.type() != "mention") {
					mentionHead = new Annotation ("mention", mentionHead.span(), null);
					doc.addAnnotation(mentionHead);
				}
				mentionHead.put("entity", entityID);
			}
		}
	}
}