function [cndd,Bsize,B] = dcond(A,blk,Z,tol)
% DCOND  estimates whether an SQLP is dual degenerate at its computed
%        solution.
%        It constructs a certain matrix B with m rows that has the
%        property the SQLP is dual nondgenerate if and only if the
%        number q of columns of B is less than or equal to m and B has
%        maximal rank.
%        The columns of B are partitioned into blocks corresponding
%        to some of the blocks of the SQLP. The SQLP is said to be
%        primal nondegenerate if the columns of B rows are linearly
%        independent.
%        Let ncols denote the number of columns of B.
%        If m < ncols, the SQLP is dual degenerate by definition,
%        and cndd is set to infinity.
%        If ncols = 0, i.e. the the matrix B is empty, then the SQLP
%        is dual degenerate and we set cndd to infinity unless m = 0
%        in which case it is nondegenerate and we set cndd = 0.
%        If ncols > 0 and m >= ncols, cndd is set to the 2-norm
%        condition number of B.  A very large condition number is a
%        strong indication that the SQLP is dual degenerate.
%        The tolerance tol is used to decide which columns to include
%        in B.  If strict complementarity does not hold, it is advisable
%        not to make tol too small, e.g. sqrt(abstol).
%
% [cnddual,Bsize,B] = dcond(A,blk,Z,tol)
%
% input variables:
%     - A            structure of constraints data
%     -     A.s      constraint matrix for SD
%     -     A.q      constraint matrix for QC
%     -     A.l      constraint matrix for LP
%     - blk          block info structure
%     -     blk.s    block info for SD
%     -     blk.q    block info for QC
%     -     blk.l    block info for LP
%     - Z            initial guess for dual structure
%     -     Z.s      SD dual variable
%     -     Z.q      QC dual variable
%     -     Z.l      LP dual variable
%     - tol          tolerance
%
% output variables:
%     - cndd         the condition number of the dual problem
%     - Bsize        partition info for columns of B
%     -     Bsize.s  # columns of B coming from SD component
%     -     Bsize.q  # columns of B coming from QC component
%     -     Bsize.l  # columns of B coming from LP component
%     - B            matrix whose condition number is cndd

% SDPPACK Version 0.9 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton, S. Schmieta
% Last modified: 6/2/97
%
 if isfield(blk,'s')
    n.sm = sum(blk.s);
    n.sv = sum(blk.s .* (1+blk.s))/2;
    nblk_s = length(blk.s);
 else
    n.sm = 0;
    n.sv = 0;
 end
 if isfield(blk,'q')
    n.q = sum(blk.q);
    nblk_q = length(blk.q);
 else
    n.q = 0;
 end
 if isfield(blk,'l')
    n.l = sum(blk.l);
 else
    n.l = 0;
 end
 if n.sm + n.q + n.l <= 0
    fprintf('dcond: the block structure is empty.  Aborting...\n');
    return
 end
%
% Note:  n.sm = 0 means that the mixed problem does not contain any SD part
%        n.q  = 0 means that the mixed problem does not contain any QC part
%        n.l  = 0 means that the mixed problem does not contain any LP part
%
 m = 0;
 if n.sm > 0
    m = size(A.s,1);
 end
 if n.q > 0
    m = max(m,size(A.q,1));
 end
 if n.l > 0
    m = max(m,size(A.l,1));
 end
 ncols = 0;       % the number of columns of B
 B = zeros(m,0);  % empty matrix with m rows
%
% First the semidefinite part
% The part of B coming from the SD component of the SQLP consists of
% representations of the matrices
%                 Q1' Ak Q1,  k = 1,...,m,
% where the columns of Q1 form an orthonormal basis for the null
% space of Z (corresponding to eigenvalues smaller than tol).
%
 if n.sm > 0   % SQL program has an SD component
    [lam,Q] = blkeig(Z.s,blk.s);
    I = lam < tol;
    if sum(I) > 0
% need to know block structure of Q1
       fin = 0;                              % for call to svec later
       next = 0;
       for i = 1:nblk_s
          bsize = blk.s(i);
          start = fin + 1;
          fin = fin + bsize;
          tmp = sum(I(start:fin));
          if tmp > 0              % block i of Q has contribution to Q1
             next = next + 1;
             Q1blk(next) = tmp;
          end
       end
       Q1 = Q(:,find(I));         % e-vectors for zero eigenvalues of Z
       ncols = sum(Q1blk .* (1+Q1blk))/2;
       B = zeros(m,ncols);
       for k=1:m
           Ak = smat(A.s(k,:),blk.s);
           B(k,:) = svec(Q1'*Ak*Q1,Q1blk)';% Q1blk defines block structure of Q1
       end
    end
 end
 Bsize.s = ncols;
%
% Now the QC part
 if n.q > 0   % SQL program has an QC component
    fin = 0;
    for k = 1:nblk_q
       bsize = blk.q(k);       % what we called nk+1 in comments above
       start = fin + 1;
       fin = fin + bsize;
       z = Z.q(start:fin);
       zbar = z(2:bsize);
       if z(1)*z(1) - zbar'*zbar < tol    % otherwise z is in the interior,
                                          % so move on to the next block
          Ak = A.q(:,start:fin);
          if z(1) < tol      % z is considered zero
             if ncols > 0
                B = cat(2,B,full(Ak));
                ncols = ncols + bsize;
             else
                B = full(Ak);
                ncols = bsize;
             end
          else                            % z on the boundary of the cone
             grad = [z(1); -zbar];   % col. vector
             if ncols > 0
                B = cat(2,B,full(Ak*grad));
                ncols = ncols + 1;
             else
                B = full(Ak*grad);
                ncols = 1;
             end
          end
       end   % if not in interior
    end   % for k
 end   % if n.q
 Bsize.q = ncols - Bsize.s;
%
% Finally the linear part
% This is very simple: we simply append to B the columns of the constraint
% matrix A.l corresponding to the zero entries of the vector Z.l
%
 if n.l > 0   % SQL program has an LP component
    I = Z.l < tol;     % indices of nonzero entries of X.l
    if ncols > 0
       B = cat(2,B,full(A.l(:,find(I))));
       ncols = ncols + sum(I);
    else
       B = full(A.l(:,find(I)));
       ncols = sum(I);
    end
 end
 Bsize.l = ncols - Bsize.s - Bsize.q;
%
 if ncols == 0
    if m == 0
       cndd = 0;
    else
       cndd = Inf;
    end
 elseif m < ncols
    cndd = Inf;
 else
    cndd = cond(B);
 end;
 fprintf('dual condition number = %11.3e\n',cndd);
%
% END function
