% SQL  driver for the function fsql.m
%
%   Mixed Conic Programming Solver
%
%   The following variables must be available in the Matlab workspace:
%     - A                  structure of constraints data
%     -     A.s            constraint matrix for SD
%     -     A.q            constraint matrix for QC
%     -     A.l            constraint matrix for LP
%     - b                  rhs of primal constraints
%     - C                  cost structure
%     -     C.s            cost matrix for SD
%     -     C.q            cost matrix for QC
%     -     C.l            cost matrix for LP
%     - blk                block info structure
%     -     blk.s          block info for SD
%     -     blk.q          block info for QC
%     -     blk.l          block info for LP
%     - X                  initial guess for primal structure
%     -     X.s            SD primal variable
%     -     X.q            QC primal variable
%     -     X.l            LP primal variable
%     - y                  initial guess for dual variable
%     - Z                  initial guess for dual structure
%     -     Z.s            SD dual slack
%     -     Z.q            QC dual slack
%     -     Z.l            LP dual slack
%     - opt                structure of options
%     -     maxit          maximum iterations allowed
%     -     tau            fraction of step to boundary of psd cone
%     -     steptol        tolerance for short steps
%     -     abstol         absolute tolerance on gap
%     -     reltol         relative tolerance on gap
%     -     gapprogtol     tolerance on sufficient progress
%     -     feasprogtol    tolerance on sufficient progress
%     -     bndtol         upper bound on size of solution
%     -     uselyapsol     flag: 1 = use mex-file lyapsol
%                                0 = do not use mex-file lyapsol
%                               -1 = let fsql decide
%     -     prtlevel       verbosity level
%     -     validate       data validation flag

% SDPPACK Version 0.9 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton, S. Schmieta
% Last modified: 6/9/97
%
% termination flag:
%    termflag = -2  ==> X is blowing up
%    termflag = -1  ==> Z is blowing up
%    termflag =  0  ==> success
%    termflag =  1  ==> either X or Z lies outside the product of cones
%    termflag =  2  ==> either X.s or Z.s has a nonpositive eigenvalue even though
%                       chol found the matrix to be positive definite
%    termflag =  3  ==> SchurComp numerically singular
%    termflag =  4  ==> new point rejected: worse than current point
%    termflag =  5  ==> shortsteps
%    termflag =  6  ==> exceeded maximum number of iterations
%    termflag =  7  ==> data failed validation test
%
% Also, if maxit > 0:
%    termflag =  1 and iter = 0 ==> initial X or Z lies outside the product of cones
%    termflag =  2 and iter = 0 ==> either Z.s has a nonpositive eigenvalue even though
%                                   chol found the matrix to be positive definite
%    termflag =  3 and iter = 0 ==> A was rank deficient; use preproc
%    termflag =  4 and iter = 0 ==> Initial pt. may be too close to bdry
%    termflag =  5 and iter = 0 ==> Initial pt. may be too close to bdry
%
% If maxit = 0, we assume the user wants to do just data validation.
%    termflag =  6  ==> data passed validation test
%
 flops(0);
 tstart = clock;
 cputstart = cputime;
%
 tau_orig = opt.tau;
 scalefac_orig = scalefac;
 for runs = 1:3
    fprintf('\ntau = %8.4f,     scalefac = %8.0f\n\n',opt.tau,scalefac);
    [X,y,Z,iter,compval,feasval,objval,termflag] = ...
        fsql(A,b,C,blk,X,y,Z,opt);
    if iter == 0
       if termflag == 4 | termflag == 5
          fprintf('\nsql: Initial point may be too close to boundary\n');
       elseif termflag == 6 & opt.maxit == 0
          fprintf('\nsql: Data passed validation tests\n');
       elseif termflag == 7 & opt.prtlevel == 0
          fprintf('\nsql: Data failed validation tests\n');
          fprintf('For more information, rerun sql with prtlevel = 1\n');
       end
       break
    end
   if autorestart & (termflag == 4 | termflag == 5)...
      & (compval(iter+1) + feasval(iter+1,1) + feasval(iter+1,2) > 5.0e-3)
       if runs < 3
          fprintf('\nrestarting...\n');
       end
       if runs == 1
          opt.tau = min(opt.tau,.99);
       elseif runs == 2
          opt.tau = min(opt.tau,.9);
       end
       scalefac = 100*scalefac;
       init;
    else
       break
    end
 end    % runs
 opt.tau = tau_orig;
 scalefac = scalefac_orig;
%
 if iter > 0
    if termflag == 6
       fprintf('\nsql: reached maximum number of iterations\n');
    end
    compval = compval(1:(iter+1));
    feasval = feasval(1:(iter+1),:);
    objval = objval(1:(iter+1),:);
    cputend = cputime - cputstart;
    elapsedtime = etime(clock,tstart);
    if runs > 1
       fprintf('\nsql: number of runs = %3.0f\n',min(runs,3));
    end
    fprintf('\nsql: elapsed time               = %- 9.3f seconds\n',elapsedtime);
    fprintf('sql: elapsed cpu time           = %- 9.3f seconds\n',cputend);
    fprintf('sql: number of iterations       = %- 4.0f\n',iter);
    fprintf('sql: final value of X . Z       = %- 11.3e\n',compval(iter+1));
    fprintf('sql: final primal infeasibility = %- 11.3e\n',feasval(iter+1,1));
    fprintf('sql: final dual infeasibility   = %- 11.3e\n',feasval(iter+1,2));
    fprintf('sql: primal objective value     = %- 24.16e\n',objval(iter+1,1));
    fprintf('sql: dual objective value       = %- 24.16e\n',objval(iter+1,2));
 end

