function w = arwimul(x,v,blk)
% ARWIMUL computes w_i = X_i^{-1} v_i, 1 <= i <= nblk, where
%           X_i = arw(x_i), the arrow matrix associated to X_i.
%
% w = arwimul(x,v,blk)
%
% Single block case:
% if x = (x0,x1,...,xn) then
%             [ x0 x1 x2 ... xn ]
%             [ x1 x0  0 ...  0 ]
%    arw(x) = [ x2  0 x0 ...  0 ]
%             [      ...        ]
%             [ xn  0  0 ... x0 ]
% Let xbar = (0,x1,...,xn) (or (x1,...,xn) depending upon context),
% let e0 = (1,0,...,0) of length n+1,  and let r = x0^2 - ||xbar||^2.
% We have
%    arw(x)^{-1} = 1/r [   x0         -xbar^T         ]
%                      [ -xbar  1/x0(rI + xbar*xbar^T ]
%
%   1      1  ||xbar||^2                                   1
% = -- I + -{ ---------- e0*e0^T - e0*xbar^T - xbar*e0^T + -- xbar*xbar^T }
%   x0     r       x0                                      x0
%
% hence computing w = arw(x)^{-1} v, for a vector v, is very fast.
%
% input variables:
%     - x         vector of quadratic blocks
%     - v         vector of length n
%     - blk       block structure vector
%
% output variables:
%     - w         vector of length n

% SDPPACK Version 0.9 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton, S. Schmieta
% Last modified : 5/23/97
%
 n =length(x);
 nblk = length(blk);
 w = zeros(n,1);
 fin = 0;
 for i = 1:length(blk)
    start = fin + 1;
    fin = fin + blk(i);
    xtmp = x(start:fin);
    vtmp = v(start:fin);
    x1 = xtmp(1);
    v1 = vtmp(1);
    xtmp(1) = 0;
    s = xtmp'*xtmp;
    t = xtmp'*vtmp;
    alpha = x1*x1 - s;
    wtmp = t*xtmp/x1;
    wtmp = wtmp - v1*xtmp;
    wtmp(1) = v1*s/x1 - t;
    wtmp = wtmp/alpha;
    wtmp = wtmp + vtmp/x1;
    w(start:fin) = wtmp;
 end
%
% END function
