% DSDP   driver for the function fdsdp
%
%   Semi Definite Programming Solver using either the XZ or the
%   XZ + ZX method for problems with diagonal constraints, namely
%   m = n = sum(blk) and the constraint matrices are of the form
%
%            Ak = Ek = ek ek^T,   1 <= k <= n
%
%   Note: dsdp calls either fdsdp.m or fsqlp.m depending upon
%   the value of the flag useXZ; when calling fsql.m, dsdp must
%   build the matrix A.s.
%   The following variables must be available in the Matlab workspace:
%      - C.s             cost matrix
%      - b               rhs of primal constarints
%      - blk.s           SD block structure vector
%      - X.s             initial guess for primal variable
%      - y               initial guess for dual variable
%      - Z.s             initial guess for dual slack variable
%      - opt             structure of options
%      -    maxit        maximum iterations allowed
%      -    tau          fraction of step to boundary of psd cone
%      -    steptol      tolerance for short steps
%      -    abstol       absolute tolerance on gap
%      -    reltol       relative tolerance on gap
%      -    gapprogtol   tolerance on sufficient progress
%      -    feasprogtol  tolerance on sufficient progress
%      -    bndtol       upper bound on size of solution
%      -    prtlevel     verbosity level (accessed only by fdsdp.m or fsdp.m)
%      - useXZ           flag: 1 = call fdsdp, 0 = call fsql

% SDPPACK Version 0.9 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton. S. Schmieta
% Last modified: 6/8/97
%
% termination flag:
%    termflag = -2  ==> X is blowing up
%    termflag = -1  ==> Z is blowing up
%    termflag =  0  ==> success
%    termflag =  1  ==> either X or Z is not strictly positive definite, i.e.
%                       either chol(X) or chol(Z) failed, or Z is singular
%                       (i.e. inversion of Z failed)
%    termflag =  2  ==> XZ method: not used
%                       XZ+ZX method: Z has eigenvalues <= 0 even chol(Z)did not fail
%    termflag =  3  ==> XZ method: SchurComp is numerically indefinite or singular
%                       XZ+ZX method: SchurComp is numerically singular
%    termflag =  4  ==> new point rejected: worse than current point
%    termflag =  5  ==> shortsteps
%    termflag =  6  ==> exceeded maximum number of iterations
%    termflag =  7  ==> data failed validation test
%
% Also:
%    termflag =  1 and iter = 0 ==> initial X or Z is not strictly positive definite
%    termflag =  2 and iter = 0 ==> XZ method: not used
%                                   XZ+ZX method: initial Z has eigenvalues <= 0
%                                   even chol(Z)did not fail
%    termflag =  4 and iter = 0 ==> Initial pt. may be too close to bdry
%    termflag =  5 and iter = 0 ==> Initial pt. may be too close to bdry
%
% If maxit = 0, we assume the user wants to do just data validation.
%    termflag =  6  ==> data passed validation test
%
 flops(0);
 tstart = clock;
 cputstart = cputime;
%
 tau_orig = opt.tau;
 scalefac_orig = scalefac;
 if useXZ,
    if opt.tau > 0.99
       fprintf('\ndsdp: tau is set to %7.5f\n',opt.tau);
       fprintf('   the XZ method does not perform well with such a high ');
       fprintf('value of tau!\n');
       choice = input('   Do you wish to proceed? <y/n> ','s');
       if choice ~= 'y'
          break;
       end;
    end;
    fprintf('\ndsdp: using XZ method...\n\n');
    for runs = 1:3,
       fprintf('\ntau = %8.4f,     scalefac = %8.0f\n\n',opt.tau,scalefac);
       [X.s,y,Z.s,iter,compval,feasval,objval,termflag] = ...
           fdsdp(C.s,b,blk.s,X.s,y,Z.s,opt);
       if iter == 0,
          if termflag == 4 | termflag == 5,
             fprintf('\ndsdp: Initial point may be too close to boundary\n');
          elseif termflag == 6 & opt.maxit == 0,
             fprintf('\ndsdp: Data passed validation test!\n');
          elseif termflag == 7 & opt.prtlevel == 0,
             fprintf('\ndsdp: Data failed validation tests\n');
             fprintf('For more information, rerun dsdp with prtlevel = 1\n');
          end;
          break;
       end;
       if autorestart & (termflag == 4 | termflag == 5)...
         & (compval(iter+1) + feasval(iter+1,1) + feasval(iter+1,2) > 1.0e-3),
          if runs < 3,
             fprintf('\nrestarting...\n');
          end;
          opt.tau = min(opt.tau,.9);
          scalefac = 100*scalefac;
          dinit;
       else,
          break;
       end;
    end;  % runs
 else,  % use XZ+ZX method: must construct the matrix A
    sparseblocks = 1;
    n = sum(blk.s);
    n2 = (sum(blk.s .* (blk.s+1)))/2;
    blk.q = [];
    blk.l = [];
    A.s = sparse(n,n2);
    for i = 1:n,
       Ai = sparse(n,n);
       Ai(i,i) = 1;
       A.s(i,:) = svec(Ai,blk.s,sparseblocks)';
    end;
    fprintf('\ndsdp: using XZ+ZX method...\n\n');
    for runs = 1:3,
       fprintf('\ntau = %8.4f,     scalefac = %8.0f\n\n',opt.tau,scalefac);
       [X,y,Z,iter,compval,feasval,objval,termflag] = ...
           fsql(A,b,C,blk,X,y,Z,opt);
       if iter == 0,
          if termflag == 4 | termflag == 5,
             fprintf('\ndsdp: Initial point may be too close to boundary\n');
          elseif termflag == 6 & opt.maxit == 0,
             fprintf('\ndsdp: Data passed validation test!\n');
          elseif termflag == 7 & opt.prtlevel == 0,
             fprintf('\ndsdp: Data failed validation tests\n');
             fprintf('For more information, rerun dsdp with prtlevel = 1\n');
          end;
          break;
       end;
       if autorestart & (termflag == 4 | termflag == 5)...
         & (compval(iter+1) + feasval(iter+1,1) + feasval(iter+1,2) > 1.0e-3),
          if runs < 3,
             fprintf('\nrestarting...\n');
          end;
          if runs == 1,
             opt.tau = min(opt.tau,.99);
          elseif runs == 2,
             opt.tau = min(opt.tau,.9);
          end;
          scalefac = 100*scalefac;
          dinit;
       else,
          break;
       end;
    end;  % runs
 end;
 opt.tau = tau_orig;
 scalefac = scalefac_orig;
%
 if iter > 0,
    if termflag == 6,
       fprintf('\ndsdp: reached the maximum number of iteration\n');
    end;
    compval = compval(1:(iter+1));
    feasval = feasval(1:(iter+1),:);
    objval = objval(1:(iter+1),:);
    elapsedtime = etime(clock,tstart);
    cputend = cputime - cputstart;
    if runs > 1,
       fprintf('\ndsdp: number of runs = %3.0f\n',min(runs,3));
    end;
    fprintf('\ndsdp: elapsed time               = %- 9.3f seconds\n',elapsedtime);
    fprintf('dsdp: elapsed cpu time           = %- 9.3f seconds\n',cputend);
    fprintf('dsdp: number of iterations       = %- 4.0f\n',iter);
    fprintf('dsdp: final value of X . Z       = %- 11.3e\n',compval(iter+1));
    fprintf('dsdp: final primal infeasibility = %- 11.3e\n',feasval(iter+1,1));
    fprintf('dsdp: final dual infeasibility   = %- 11.3e\n',feasval(iter+1,2));
    fprintf('dsdp: primal objective value     = %- 24.16e\n',objval(iter+1,1));
    fprintf('dsdp: dual objective value       = %- 24.16e\n',objval(iter+1,2));
 end;
