function [lambda,Pi,x,y] = getEigValProjVec(A,lambda0)
%
% Return the eigenvalue of A closest to lambda0, assumed to be simple.
% Also return the corresponding eigenprojector Pi=x*y', as well as
% right and left eigenvectors x, y, normalized so y'*x = 1.
%
% input: 
%   A: square matrix (the perturbed matrix) 
%   lambda0: (scalar, the relevant eigenvalue of the base matrix)
% output: 
%   lambda: the eigenvalue of A closest to lambda0
%   Pi: corresponding eigenprojector x*y',
%   x, y: corresponding right and left eigenvectors, normalized by y'*x = 1
% 
n = length(A); % A is square
%
% compute all the right and left eigenvectors and eigenvalues, although we 
% need only the largest eigenvalue in modulus and the corresponding eigenvectors
% These satisfy A*X = X*Lambda and Y'*A = Lambda*Y', with Lambda diagonal
%
[X,Lambda,Y] = eig(A); 
Lambda = diag(Lambda); % make Lambda a vector
[~,indx] = min(abs(Lambda - lambda0));
lambda = Lambda(indx); % closest eigenvalue to lambda0
%
% check that the eigenvalue is not a multiple eigenvalue
% we could use a tolerance here, but it's not clear what to use, so we just
% check for exact equality of eigenvalues
%
indxothers = find(Lambda == lambda);
if length(indxothers) > 1
    error('largest eigenvalue of A in modulus is a multiple eigenvalue')
end
x = X(:,indx); % relevant right eigenvector
y = Y(:,indx); % relevant left eigenvector
ytx = y'*x;
x = x/(y'*x);  % so that y'*x = 1
Pi = x*y'; % relevant eigenprojector
normPi = norm(x)*norm(y);
if normPi > 1e6
    fprintf('Warning: eigenvalue condition number is %e\n', normPi)
end