function [x_discrep, y_discrep] = ...
    eigvecvary(x0,y0,xpert,ypert,h,dx,dy,normalization,displaylevel)
%
% compute and numerically verify right and left eigenvector derivatives 
%   dx(t)/dt  and dy(t)/dt
% where x(t), y(t) are the smoothly varying eigenvector functions of
%   A(t) = A0 + t dA
% normalized by y(t)'*x(t)=1, whose existence is known from Theorem 2 of
% Greenbaum, Li, Overton paper, using formlas given in Section 3.3.
% When normalization = 0: no additional normalization is done. 
% Otherwise, use right and left *normalized* eigenvector derivatives
%   dxhat(t)/dt and dyhat(t)
% where xhat(t) = alpha(t)x(t), yhat(t) = beta(t)y(t)
% with alpha(t) and beta(t) chosen so that xhat(t) and yhat(t) 
% have properties defined by normalization 1,2,3 or 4 specified below.
%
% input
%  x0, y0: the relevant right and left eigenvectors of the base matrix
%  xpert,ypert: the relevant right and left eigenvectors of the perturbed
%    matrix
%  h: the difference quantity for the difference quotient
%  dx, dy: the derivatives of the eigenvectors with normalization 0, as
%    given in Theorem 2
%  normalization: 0, 1, 2, 3 or 4: 
%    normalization 0: uses the analytic x(t), y(t)' defined in Section 3.3
%    otherwise: see below for definition
%  displaylevel: 1 to display scalars only, 2 to display vectors

%
% We consider various normalizations
%    xhat(t)=alpha(t)x(t), yhat(t)=beta(t)y(t)
% for which we can compute and verify the derivatives.
% We first compute the derivatives
%
normalization_x_indx = nan; % only used by normalizations 1 and 2
normalization_y_indx = nan; % only used by normalization 1
%
if normalization == 0
    %
    % not normally used in practice, as it *depends* on x0, y0, but this
    % makes xhat, yhat' actually equal to the analytic functions x, y'
    % defined in terms of the eigenprojector (see Section 3.3 of the paper)
    % alpha(t) = 1, beta(t) = 1
    %
    dxhat = dx;
    dyhat = dy;
    %
elseif normalization == 1
    % Let j be the index maximizing |x0_i| and
    % let k be the index maximizing |y0_i|.
    % Define the normalization such that
    % xhat(t)_j = 1, yhat(t)_k = 1 so need
    % alpha(t)=1/(e_j'*x(t)), beta(t)=1/(e_k'*y(t))
    % all derivatives are evaluated at t=0
    [~,j] = max(abs(x0));
    normalization_x_indx = j;
    [~,k] = max(abs(y0));
    normalization_y_indx = k;
    alpha = 1/x0(j);
    beta = 1/y0(k);
    dalpha = -dx(j)/(x0(j)^2); 
    dxhat = dalpha*x0 + alpha*dx;
    dbeta = -dy(k)/(y0(k)^2);
    dyhat = dbeta*y0 + beta*dy;
    %
elseif normalization == 2
    % Defining j as in normalization 1,
    % want xhat(t)_j = 1, yhat(t)'*xhat(t)=1 and since the latter
    % equation says alpha(t)conj(beta(t))y(t)'*x(t) = 1,but y(t)'*x(t) = 1,
    % we need alpha(t)=1/(x(t)_1), beta(t)=conj{x(t)_1).
    % hence dalpha is as before and dbeta (at t=0) is conj(dx_j).
    % not defined if x0(1) = 0.
    %
    [~,j] = max(abs(x0));
    normalization_x_indx = j;
    alpha = 1/x0(j);
    dalpha = -dx(j)/(x0(j)^2);
    dxhat = dalpha*x0 + alpha*dx;
    beta = conj(x0(j));
    dbeta = conj(dx(j));
    dyhat = dbeta*y0 + beta*dy;
    %
elseif normalization == 3
    %
    % want xhat(t)^T xhat(t) = 1, yhat(t)'*xhat(t)=1 and since the latter
    % equation says alpha(t)conj(beta(t))y(t)'*x(t) = 1 need
    % alpha(t)=(x(t)^T x(t))^(-1/2), beta(t)=conj{(x(t)^T x(t))^(+1/2)/(y(t)'*x(t)}
    % hence dalpha (at t=0) is -(x^T x)^(-3/2)(x^T dx) and dbeta (at t=0) is 
    % conj((x^T x)^(-1/2)(x^T dx) - (x'x)^(1/2)*(y'*dx + dy'*x)) 
    %
    % not defined if the following is zero
    xTx = x0.'*x0;  % not the squared norm, could be zero
    xTdx = x0.'*dx;
    %
    % alpha, beta and all the derivatives are evaluated at t=0
    %
    alpha = xTx^(-1/2); % often, but not always, turns out to be one since
                        % apparently Matlab often normalizes the
                        % eigenvectors this way even in the complex case --
                        % although this is not always possible...
    beta = conj(xTx^(1/2));
    dalpha = -(xTx)^(-3/2)*xTdx;
    dxhat = dalpha*x0 + alpha*dx;
    dbeta = conj(xTdx*(xTx)^(-1/2));
    dyhat = dbeta*y0 + beta*dy; 
    %
elseif normalization == 4 % not unique
    %
    % want xhat(t)'*xhat(t) = 1, yhat(t)'*xhat(t)=1 
    % but the first equation implies only that |alpha(t)|=1/||x(t)||.
    % We can choose alpha(t)=1/||x(t)|| but we won't be able to verify the
    % derivatives numerically because we don't know how to actually compute
    % xhat: in the other cases it was uniquely defined.
    % We also need beta(t)=conj{||x(t)||/(y(t)'*x(t)}.
    % So dalpha (at t=0) is -(x'*x)^(-3/2)(REAL(x'*dx)) and dbeta (at t=0) is 
    % conj(||x||^(-1/2)(REAL(x'*dx)) - ||x||*(y'*dx + dy'*x)) 
    %
    normx = norm(x0);  
    xctdx = x0'*dx;
    % alpha, beta and all the derivatives are evaluated at t=0
    alpha = 1/normx; 
    beta = conj(normx);
    dalpha = -(normx)^(-3)*real(xctdx);
    dxhat = dalpha*x0 + alpha*dx;
    dbeta = conj(alpha*real(xctdx) - (normx)*(y0'*dx + dy'*x0));
    dyhat = dbeta*y0 + beta*dy; 
else
    error('invalid normalization code')
end
%
% compare the derivative formulas above with difference quotients
%
% first get the normalized eigenvectors of A0
%
if normalization == 0
    xhat0 = x0;
    yhat0 = y0;
else % normalization_x_indx,normalization_y_indx are only used for normalizations 1 and 2
    [xhat0,yhat0] = getEigVecNormalized(x0,y0,normalization,...
        normalization_x_indx,normalization_y_indx);
end
%
% then get the eigenvalues and normalized eigenvectors of the pertubed matrix
% normalization_x_indx,normalization_y_indx are only used for normalizations 1 and 2
% xhat0 is used for normalization = 3
% x0 and y0 are used for normalization = 0 
%
[xhatpert,yhatpert] = getEigVecNormalized(xpert,ypert,normalization,...
    normalization_x_indx,normalization_y_indx,xhat0,x0,y0);
if displaylevel > 1
    fprintf('\nRIGHT EIGENVECTOR\n')
    fprintf('                    derivative                                     difference quotient\n')
    displayVecs(dxhat, (xhatpert-xhat0)/h);
    fprintf('\nLEFT EIGENVECTOR\n')
    fprintf('                    derivative                                     difference quotient\n')
    displayVecs(dyhat, (yhatpert-yhat0)/h);
end
%
% compute and display the discrepancy norms
%
dif_quo_x = (xhatpert-xhat0)/h;
dif_quo_y = (yhatpert-yhat0)/h;
x_discrep = norm(dxhat - dif_quo_x)/norm(dxhat);
y_discrep = norm(dyhat - dif_quo_y)/norm(dyhat);
if displaylevel > 0
    fprintf('\nrelative norms of discrepancies between derivatives and difference quotients are\n\n')
    fprintf('right eigenvector: %e\n',x_discrep)
    fprintf('left eigenvector:  %e\n',y_discrep)
end
    