function [lambda0,x0,y0,Pi0,X1,Y1] = applyLemma1(A0)
%
% apply Lemma 1 of the Greenbaum, Li, Overton paper.
% assumes that the largest eigenvalue of A0 in modulus is simple,
% but makes no assumption on multiplicities or Jordan structure of other 
% eigenvectors. Computation follows Stewart's 2001 book Eigendcompositions,
% Thm 1.18 and eq. (3.10)
%
% input: 
%   A0: square matrix (the base matrix)
% output:
%   lambda0: largest eigenvalue in modulus of A0, assumed to be simple
%   x0, y0: corresponding right and left eigenvectors normalized so
%           that y0'*x0 = 1
%   Pi0: the eigenprojector x0*y0'
%   X1, Y1: matrices satisfying Lemma 1 in Greenbaum, Li, Overton paper,
%   i.e.,  X=[x0 X1], Y = [y0, Y1], Y'*X = I, and Y'*A0*X is block diagonal 
%   with the scalar lambda0 as the first block and an (n-1) by (n-1) matrix  
%   B1 as the second block. Note that the columns of X1 and Y1 are not 
%   necessarily right or left eigenvectors corresponding to the other 
%   eigenvalues, as A0 might not be diagonalizable
%

%
% unlike in Stewart, we don't start with the Schur decomposition, because
% then the eigenvalue ordering is tricky. We simply start by computing
% an eigenvalue with largest modulus (to define a specific eigenvalue) and 
% a corresponding right eigenvector
%
n = length(A0); % A0 is square
[V,Lambda] = eig(A0); % right eigenvectors and eigenvalues
Lambda = diag(Lambda); % make Lambda a vector
[~,indx] = max(abs(Lambda)); 
lambda0 = Lambda(indx); % largest eigenvalue in modulus, for convenience
indxothers = find(Lambda ~= lambda0);
if length(indxothers) ~= n-1
    error('largest eigenvalue of A in modulus is a multiple eigenvalue')
end
v = V(:,indx); % relevant right eigenvector, no special normalization
[U,~] = qr(v); % U is n by n and unitary
%
% v is an eigenvector of A0, and the first column of U is a scalar multiple 
% of this with norm 1, so we have U'*A0*U is block upper triangular, with 
% the 1,1 block equal to lambda0. Let b denote the upper right block (a row
% vector) and let B1 denote the 2,2 block (which is n-1 by n-1).
%
UtA0U = U'*A0*U;
normA0 = norm(A0,1);
if abs(UtA0U(1,1) - lambda0) > 1e-8*normA0
    error('applyLemma1: why is 1,1 entry not equal to lambda0?')
end
if norm(UtA0U(2:n,1)) > 1e-8*normA0
    error('applyLemma1: why is 2,1 block not small in norm?')
end
b = UtA0U(1,2:n);
B1 = UtA0U(2:n,2:n);
%
% now we need to solve the Sylvester equation lambda0*q - q*C = -b for q
%
q = sylvester(lambda0,-B1,-b);
z = zeros(n-1,1);
I = eye(n-1);
H = [1 q; z I];
Hinv = [1 -q; z I]; % inverse of H
X = U*H;
Y = (Hinv*U')';
%
% by construction, Y'*A0*X is block diagonal with blocks lambda0 and B1, and
% since Y' is the inverse of X, not only is the first column of X a right
% eigenvector, but also the first column of Y is a left eigenvector
%
x0 = X(:,1); % right eigenvector for lambda0
y0 = Y(:,1); % left eigenvector, satisfying y0'*x0 = 1
Pi0 = x0*y0'; % eigenprojector for lambda0
X1 = X(:,2:n); % not eigenvectors but satisfying Y1'*X1 = I
Y1 = Y(:,2:n); 


