/*
 * Functions that interface with the C time.h library, and perform date/ time
 * manipulations.
 */

#include <time.h>

#include "timing.h"

#if defined(CLOCK_HIGHRES)
	#define CLOCK CLOCK_HIGHRES
#elif defined(CLOCK_REALTIME)
	#define CLOCK CLOCK_REALTIME
#else
	#error No suitable clock found.  Check docs for clock_gettime.
#endif

/*
 * Converts the clock resolultion stored in the timespec structure into a long
 * double (seconds).
 */
long double timespec_to_ldbl( struct timespec ts )
{
	return ts.tv_sec + 1.0E-9 * ts.tv_nsec;
}

/*
 * Calculates the difference between start and end times, measured in seconds.
 */
long double timespec_diff( struct timespec sta, struct timespec end )
{
	long double delta;

	delta = (end.tv_nsec - (double)sta.tv_nsec) * 1.0E-9;
	delta += end.tv_sec - (double)sta.tv_sec;

	return delta;
}

/*
 * Gets resolution for the defined CLOCK.  The clock resolution, which is stored
 * in the timespec structure, is converted to a long double (seconds) before it
 * is returned by the function.
 */
long double timer_resolution( void )
{
	struct timespec ts;

	clock_getres( CLOCK, &ts );

	return timespec_to_ldbl( ts );
}

/*
 * Gets time for the defined CLOCK.
 */
void get_time( struct timespec *ts )
{
	clock_gettime( CLOCK, ts );
}
