#if !defined(LAPACK_H_)
	#define LAPACK_H_ 1

#if defined(__cplusplus)
	extern "C" {
#endif

	// Prototypes for LAPACK and BLAS routines:

	// ILAENV determines the optimal block size for the local environment.
	int ilaenv_( const int *ISPEC, const char *NAME, const char *OPTS,
		const int *N1, const int *N2, const int *N3, const int *N4 );

	// DCOPY copies vector X to vector Y, where N is the number of elements, and
	// INCX and INCY are increments of elements of X and Y, respectively.
	int dcopy_( const int *N, const double *X, const int *INCX, 
		double *Y, const int *INCY );

	// DSWAP interchanges vectors SX and SY, where N is number of elements, and  
	// INCX and INCY are increments of elements of SX and SY, respectively.
	int dswap_( const int *N, const double *SX, const int *INCX, 
		double *SY, const int *INCY );

	// IDAMAX returns the index of the element of DX having the maximum absolute
	// value (maximum magnitude), where N is the number of elements and INCX 
	// is the increment of elements of DX.
	int idamax_( const int *N, const double *DX, const int *INCX );

	// DGEMM performs matrix multiplication, C = beta*C + alpha * op(A) * op(B), 
	// where alpha and beta are scalars, 
	// C, A and B are M-by-N, M-by-K and K-by-N matrices, respectively,
	// and op(X) = X or op(X) = X'.
	int dgemm_( const char *TRANSA, const char *TRANSB,
		const int *M, const int *N, const int *K, 
		const double *ALPHA, const double *A, const int *LDA,
		const double *B, const int *LDB,
		const double *BETA, double *C, const int *LDC );

	// DGEMV performs matrix-vector operation y = beta*y + alpha * op(A) * x,
	// where alpha and beta are scalars, x and y are vectors, A is an M-by-N
	// matrix, and op(A) = A or op(A) = A'.
	int dgemv_( const char *TRANS, const int *M, const int *N,
		const double *ALPHA, const double *A, const int *LDA,
		const double *X, const int *INCX,
		const double *BETA, double *Y, const int *INCY );

	// DSYRK performs symmetric rank k operation 
	// C = alpha*op(A)*op(A)' + beta*C, where alpha and beta are scalars, 
	// C is an N-by-N symmetric matrix and A is an N-by-K matrix, and
	// op(X) = X or op(X) = X'.
	int dsyrk_( const char *UPLO, const char *TRANS, const int *N,
		const int *K, const double *ALPHA, const double *A, const int *LDA,
		const double *BETA, double *C, const int *LDC );

	// DTRSM solves a triangular system of the form 
	// op(A)*X = alpha*B or X*op(A) = alpha*B,
	// where alpha is a scalar, X and B are m-by-n matrices, A is a triangular 
 	// matrix, and op(A) = A or op(A) = A'.
	int dtrsm_( const char *SIDE, const char *UPLO, const char *TRANSA, 
		const char *DIAG, const int *M, const int *N, const double *ALPHA, 
		const double *A, const int *LDA, double *B, const int *LDB );

	// DTRTRS solves a triangular system of the form A*X = B or A'*X = B,
	// where A is a triangular matrix of order N, and B is an N-by-NRHS matrix.
	int dtrtrs_( const char *UPLO, const char *TRANS, const char *DIAG,
		const int *N, const int *NRHS, const double *A, const int *LDA, 
		double *B, const int *LDB, int *INFO );

	// DGETRF computes an LU factorization of a general M-by-N matrix A using
	// partial pivoting with row interchanges.  The factorization takes the form
	// A = P*L*U, where P is the permutation matrix encoded in the vector IPIV,
	// L is unit lower triangular and U is upper triangular.  On exit, L and U
	// overwrite A.
	int dgetrf_( const int *M, const int *N, double *A, const int *LDA, 
		int *IPIV, int *INFO );

	// DPOTF2 computes the Cholesky factorization of an N-by-N real symmetric
	// positive definite matrix A.  The factorization takes the form A = L*L' or
	// A = U'*U, where L is lower triangular and U is upper triangular.  On exit, 
	// L or U overwrites A.  This is LAPACK's unblocked version of Cholesky
	// factorization.
	int dpotf2_( const char *UPLO, const int *N, double *A, const int *LDA, 
		int *INFO );

	// DPOTRF computes the Cholesky factorization of an N-by-N real symmetric
	// positive definite matrix A.  The factorization takes the form A = L*L' or
	// A = U'*U, where L is lower triangular and U is upper triangular.  On exit, 
	// L or U overwrites A.
	int dpotrf_( const char *UPLO, const int *N, double *A, const int *LDA, 
		int *INFO );

	// DSYTRF computes the factorization of a real symmetric matrix A using
	// the Bunch-Kaufman pivoting.  The factorization takes the form 
	// A = (P*L)*D*(P*L)' or A = (P*U)*D*(P*U)', where P is the permuation 
	// matrix encoded in IPIV, L is unit lower triangular, U is unit upper
	// triangular and D block diagonal with block order 1 or 2.  On exit,
	// (P*L) or (P*U) and D overwrite A.  This is LAPACK's unblocked version
	// of symmetric indefinite factorization.
	int dsytf2_(const char *UPLO, const int *N, double *A, const int *LDA, 
		int *IPIV, int *INFO);

	// DSYTRF computes the factorization of a real symmetric matrix A using
	// the Bunch-Kaufman pivoting.  The factorization takes the form 
	// A = (P*L)*D*(P*L)' or A = (P*U)*D*(P*U)', where P is the permuation 
	// matrix encoded in IPIV, L is unit lower triangular, U is unit upper
	// triangular and D block diagonal with block order 1 or 2.  On exit,
	// (P*L) or (P*U) and D overwrite A.
	int dsytrf_(const char *UPLO, const int *N, double *A, const int *LDA, 
		int *IPIV, double *WORK, int *LWORK, int *INFO);

#if defined(__cplusplus)
	}
#endif

#endif
