classdef box < handle & matlab.mixin.Copyable
    %box class for subdivision algorithms
    
    properties
        boxdimensions %D*2 vector with min and max coordinate of 
            %the dimensions in each row[xmin,xmax; ymin,ymax,...]

        length0 %D*1 logical vector keeping track of the directions of 0 length
            %i.e. representing a lower dimensional box / face in D-dimensional space

        depth = 0;
    end
    properties (NonCopyable)
        parent %parent box
        children %2*2*...*2 array containing 2^d many children boxes
            %children(1,1,...,1) corresponds to the child with minimal
            %coordinates in each direction
            %d is the number of directions in which the box does not have 0 length
        
        plotboxhandle %handle to the plotted box boundary
    end
    
    methods
        function this = box(boxdimensions,length0)
            %box(boxdimensions)
            if nargin > 0
                this.boxdimensions = boxdimensions;
                if nargin > 1
                    this.length0 = length0;
                else
                    this.length0 = boxdimensions(:,1) == boxdimensions(:,2);
                end
            end
        end
        function disp(this)
            if length(this) == 1
                disp(this.boxdimensions);
            else
                disp([num2str(size(this)),' matrix of boxes']);
            end
        end
        function boxes = split(this)
            if ~isempty(this.children) || isempty(this.boxdimensions)
                boxes = [];
%                 warning("no box split performed");
                return
            end
            D = size(this.boxdimensions,1);
            d = D-sum(this.length0);

            childrenarry(2^d) = copy(this);
            if D == 1 || d == 1
                childrenarry = reshape(childrenarry,[2*ones(1,d),1]);
            else
                childrenarry = reshape(childrenarry,2*ones(1,d));
            end
            
            vLim = 2*ones(1,d);
            v     = ones(1, d);
            ready = false;
            while ~ready
                Index = arrayindexing.sub2indV(vLim, v);
                %If v(i) = 1, then the child will get the smaller coordinate in dimension i
                vcomplete = ones(1,D);
                vcomplete(~this.length0) = v;
                childrenarry(Index).boxdimensions = [this.boxdimensions(:,1),sum(this.boxdimensions,2)/2]+((vcomplete-1)'.*(this.boxdimensions(:,2)-this.boxdimensions(:,1))/2)*[1,1];
                childrenarry(Index).length0 = this.length0;
                childrenarry(Index).depth = this.depth+1;
                childrenarry(Index).parent = this;

                % Update the index vector:
                [v,ready] = arrayindexing.updateindexvec(v,vLim);
            end
            
            this.children = childrenarry;
            this.plotboxhandle = [];
            boxes = childrenarry(:)';
        end
        
        function c = center(this)
            c = sum(this.boxdimensions,2)/2;
        end
        function r = radius(this)
            r = norm(this.center-this.boxdimensions(:,1));
        end
        function C = corners(this)
            D = size(this.boxdimensions,1);
            d = D-sum(this.length0);
            C = zeros(D,2^d);
            
            for i = 1:2^d
                v = arrayindexing.ind2subV(2*ones(1,d),i);
                corner = zeros(D,1);
                corner(this.length0) = this.boxdimensions(this.length0,1);
                vind = 0;
                for j = 1:D
                    if ~this.length0(j)
                        vind = vind+1;
                        corner(j) = (v(vind)==1)*this.boxdimensions(j,1)+(v(vind)==2)*this.boxdimensions(j,2);
                    end
                end
                C(:,i) = corner;
            end
        end
        function f = facets(this)
            D = size(this.boxdimensions,1);
            d = D-sum(this.length0);
            dind = d;
            for i = D:-1:1
                if ~this.length0(i)
                    for j = 2:-1:1
                        B = copy(this);
                        B.boxdimensions(i,3-j) = B.boxdimensions(i,j);
                        B.length0 = this.length0;
                        B.length0(i) = true;
                        f(dind,j) = B;
                    end
                    dind = dind-1;
                end
            end
        end
        
        function Bscaled = scale(this,factor)
            classtype = class(this);
            scaledboxdimensions = interval(this.boxdimensions(:,1),this.boxdimensions(:,2)).scale(factor); 
            extractedbounds = scaledboxdimensions.extractbounds; %#ok<NASGU> 
            Bscaled = eval([classtype,'(scaledboxdimensions.extractbounds);']);
            Bscaled.length0 = this.length0;
        end
        
        %Returns a box without the removed dimension
        function Bnew = removedim(this,dim)
            classtype = class(this);
            dims = this.boxdimensions;
            len0 = this.length0;
            dims(dim,:) = [];
            len0(dim,:) = [];
            Bnew = eval([classtype,'(dims);']);
            Bnew.length0 = len0;
        end

        %Returns a box with an inserted dimension at the specified location
        %Inserts to the first/last position if the requested position is
        %out of bounds
        function Bnew = insertdim(this,dim,rowdims)
            classtype = class(this);
            dims = this.boxdimensions;
            len0 = this.length0;
            if dim < 1
                dim = 1;
            elseif dim > length(this.boxdimensions) + 1
                dim = length(this.boxdimensions) + 1;
            end
            dims = [dims(1:dim-1,:); rowdims; dims(dim:end,:)];
            len0 = [len0(1:dim-1,:); rowdims(1) == rowdims(2); len0(dim:end,:)];
            Bnew = eval([classtype,'(dims);']);
            Bnew.length0 = len0;
        end

        function Q = leaves(this)
            if isempty(this.children)
                Q = this;
            else
                Q = [];
                for i = 1:numel(this.children)
                    Q = [Q,leaves(this.children(i))]; %#ok<*AGROW> 
                end
            end
        end
        function int = interval(this)
            int = interval(this.boxdimensions(:,1),this.boxdimensions(:,2));
        end

        function str = box2str(this)
            str = int2str(this.interval);
        end

        %TODO
        function Bcap = cap(this,B)
            %to be implemented
            Bcap = this;
        end
        function Bcup = cup(this,B)
            %to be implemented
            Bcup = this;
        end
        
        function varargout = coord(this)
            %Splits the box into its coordinate interval components
            %For 3D boxes:
            %[x,y,z] = B.splitcoordinates
            D = size(this.boxdimensions,1);
            varargout = cell(D,1);
            intvec = this.interval;
            output = mat2cell(intvec(:),ones(D,1))';
            [varargout{:}] = deal(output{:});
        end
        function plotsubdivision(this,ax,varargin)
            if isempty(this.children)
                this.plotbox(ax,varargin{:});
            else
                for i = 1:numel(this.children)
                    this.children(i).plotsubdivision(ax,varargin{:});
                end
            end
        end
        function h = plotbox(this,ax,varargin)
            D = size(this.boxdimensions,1);
            d = D-sum(this.length0);
            switch d
                case 1
                    %1-dimensional box possibly in higher dimensions
                    corners = this.corners;
                    switch D
                        case {1,2}
                            h = scatter(corners(1,:),corners(2,:),varargin);
                        case 3
                            h = scatter3(corners(1,:),corners(2,:),corners(3,:),varargin);
                    end
                case 2
                    %plot the box's edges
                    corners = this.corners;
                    corners = corners(:,[1,2,4,3,1]);
                    switch D
                        case 2
                            h = plot(ax,corners(1,:),corners(2,:),varargin{:});
                        case 3
                            h = plot3(ax,corners(1,:),corners(2,:),corners(3,:),varargin{:});
                    end
                case 3
                    %plot the box's edges
                    corners = this.corners;
                    %collect the vertex indices making up the edges
                    eind = [1,2;...
                            2,4;...
                            4,3;...
                            3,1;...
                            5,6;...
                            6,8;...
                            8,7;...
                            7,5;...
                            1,5;...
                            2,6;...
                            3,7;...
                            4,8];
                    XYZ = zeros(3,36);
                    for i = 1:12
                        XYZ(:,[3*i-2,3*i-1,3*i]) = [corners(:,eind(i,:)),[NaN;NaN;NaN]];
                    end
                    h = plot3(ax,XYZ(1,:),XYZ(2,:),XYZ(3,:),varargin{:});
            end
            this.plotboxhandle = h;
        end
        function zoom(this,ax)
            plotinterval = this.boxdimensions;
            plotinterval(this.length0,:) = this.boxdimensions(this.length0,:)+0.5*[-ones(sum(this.length0),1),ones(sum(this.length0),1)];
            plotinterval = plotinterval';
            plotinterval = reshape(plotinterval(:),1,[]);
            axis(ax,plotinterval);
        %                 r1 = boundingbox(1,2)-boundingbox(1,1);
        %                 r2 = boundingbox(2,2)-boundingbox(2,1);
        %                 pbaspect([r1, r2, max(r1,r2)])
            daspect(ax,[1 1 1]);
        end
%         function fillbox(this,axeshandle,color)
%             switch size(this.boxdimensions,1)
%                 case 2
%                     X = [this.boxdimensions(1,1);this.boxdimensions(1,2);this.boxdimensions(1,2);this.boxdimensions(1,1);this.boxdimensions(1,1)];
%                     Y = [this.boxdimensions(2,1);this.boxdimensions(2,1);this.boxdimensions(2,2);this.boxdimensions(2,2);this.boxdimensions(2,1)];
%                     this.filling = fill(axeshandle,X,Y,color);
%                 case 3
%             end
%         end
        
        function delete(this)
            if ~isempty(this.children)
                while ~isempty(this.children)
                    this.children(1).delete;
                    this.children(1) = [];
                end
            end
            if ~isempty(this.parent)
                this.parent = [];
            end
        end
        

        %automatically transform into an interval for interval computations
        function result = plus(a,b)
            %a+b
            [a,b] = this.convert2interval(a,b);
            result = a+b;
        end
        function result = minus(a,b)
            %a-b
            [a,b] = this.convert2interval(a,b);
            result = a-b;
        end
        function result = uminus(a)
            %-a
            [a] = this.convert2interval(a);
            result = -a;
        end
        function result = uplus(a)
            %+a
            [a] = this.convert2interval(a);
            result = a;
        end
        function result = times(a,b)
            %a.*b
            [a,b] = this.convert2interval(a,b);
            result = a.*b;
        end
        function result = mtimes(a,b)
            %a*b
            [a,b] = this.convert2interval(a,b);
            result = a.*b;
        end
%         function result = rdivide(a,b)
%         end

    end
    methods(Static)
        function varargout = convert2interval(varargin)
            n = length(varargin);
            varargout = cell(1,n);
            for i = 1:n
                if isnumeric(varargin{1}) || isa(varargin{1},"interval")
                    varargout{i} = varargin{1};
                else
                    varargout{i} = varargin{1}.interval;
                end
            end
        end
        function [B,fig,ax] = test
            B = box([-1,1;-1,1;0,0],[false,false,true]);
            [fig,ax] = box.testsplit(B);
        end
        function [fig,ax] = testsplit(B)
            [fig,ax] = createfigure(B);
            B.split;
            B.children(1).split;
            B.children(1).children(2).split;
            B.plotsubdivision(ax,'k');
        end
    end
end

