package heurgame.event.turn;

import heurgame.event.TurnEvent;
import heurgame.event.TurnListener;
import heurgame.logging.LogBox;

import java.util.ListIterator;
import java.util.Vector;

/*
 * Created on Oct 15, 2004
 */

/**
 * @author David Kaplin
 *
 *  One pattern I found was using integers to represent each player.
 * By itself is a fine way of doing things, but there seems to be
 * too much of a temptation to change the for loop or the counter
 * in strange ways to make that representation horrible to read and
 * debug.
 * 
 * This abstraction allows a variety of different sorts of TurnIterators.
 * Some may be based on the order of Player Tokens, others might not have
 * any discernable order at all.
 */
public abstract class AbstractTurnIterator {
    /**
     * The entities that are intent on finding out the current turn <code>mylisteners</code>
     */
    protected Vector mylisteners;
    private LogBox turnLog;
    protected volatile boolean haltNotification = false;
    
    public AbstractTurnIterator(){
        mylisteners = new Vector();
        turnLog = new LogBox("AbstractTurnIterator");
    }
    
    /**
     * Adds a Turn Listener to be notified of changes
     * 
     * @param t to be added
     */
    public void addTurnListener(TurnListener t) {
        mylisteners.add(t);
        turnLog.addEntry("Added Listener ",""+t.getClass());
        haltNotification = false;
    }
    /**
     * Called when the next turn is desired
     */
    abstract public void next();
    /**
     * @return true if next can produce a valid turn event.
     */
    abstract public boolean hasMore();
    
    /**
     * Unlike other listener/callback patterns this is NOT multithreaded.
     * Order must be maintained during a single turn.
     *  
     * @param e The event to be broadcast to all listeners
     */
    protected void notifyTurnListeners(TurnEvent e){
        ListIterator listeners = mylisteners.listIterator();
        
            while ((haltNotification == false) &&listeners.hasNext()) {
                final TurnListener t = (TurnListener) listeners.next();
                t.turnChanged(e);
            }
        
    }
    /**
     * Removes a Turn Listener
     * 
     * @param t to be removed
     */
    public void removeTurnListener(TurnListener t) {
        if (mylisteners.contains(t)) {
            mylisteners.remove(t);
        } else {
            turnLog.addUrgentEntry("Could Not Remove Listener",""+t.getClass(),1);
        }
    }

    /**
     * Silences the iterator's output
     */
    public void removeAllTurnListeners() {
        haltNotification = true;
        mylisteners.clear();
        
    }

}
