package heurgame;

import heurgame.analysis.SystemAnalyzer;
import heurgame.event.*;

/**
 * @author David Kaplin
 *
 */
public interface Game {
    
    /**
     * @return Name of the Game :)
     */
    abstract public String getName();
    /**
     * @return Minimum number of players needed for the game.
     */
    abstract int getMinimumPlayers();
    
    /**
     * @return Amount of time before a warning is sent (milliseconds)
     */
    abstract public long getMaximumNormalTime();
    /**
     * @return Amount of time before player is rejected after a warning (milliseconds)
     */
    abstract public long getMaximumWarningTime();
    /**
     * Note the Initial Keyword will be sent by the Player Proxy
     * This menthod just returns the extra information.
     * 
     * @return The Initial Status of the game.
     */
    abstract public String getGreeting();
    /**
     * The SystemAnalyzer has the role of judging the entire game state.
     * 
     * @return the officiating SystemAnalyzer
     */
    abstract public SystemAnalyzer getSystemAnalyzer();
    /**
     * The TimeBroadcaster is vital in keeping players in the right bounds
     * of their allotted time.
     * @return a constructed TimeBroadcaster;
     */
    abstract public TimeBroadcaster getTimeBroadcaster();
    /**
     * Ensures that a game will construct a compatible Player Proxy.
     * 
     * @param connection	live network connection
     * @return a constructed PlayerProxy
     */
    abstract public PlayerProxy buildPlayerProxy(java.net.Socket connection);
    /**
     * Constructs an HuClient, a Human Client that sends its commands through 
     * the games user interface.  
     * 
     * This should construct a new HuClient that will then connect with the
     * game server and thus be treated like any other player as far as the
     * game's rule logic is concerned.  
     * 
     * @param name of the Human playing
     */
    abstract public void buildHumanClient(String name);
    /**
     * The interface needs a handle to the human clients.
     * Once the game starts, this method should return a valid
     * array of the connected HuClients.
     * 
     * @return array of HuClients;
     */
    abstract public HuClient[] getHumanClients();
    /**
     * Returns the current state of the game.
     * 
     * @return Implementation defined
     */
    abstract public String getState();
    /**
     * Returns the current incremental state
     * @return Implementation defined
     */
    abstract public String getIncrementalState();
    /**
     * Initiates the broadcast to send the end of turn signal.
     */
    abstract public void endTurn();
    
    /**
     * Called when a player has been added
     * 
     * @param token distinct to each player
     */
    abstract public void playerAdded(PlayerToken token);
    /**
     * Initiates the start of turn broadcast. 
     */
    abstract public void start();
    /**
     * Should supply at least one level of undo.
     * @return true if the move was undone.
     */
    abstract public boolean undoLastMove();
    /**
     * Should broadcast that a player has forcably left the game.
     * 
     * @param disqualified token representing the disqualified player
     */
    abstract public void announceDisqualification(PlayerToken disqualified);
    /**
     * Should broadcast that a player has left the game.
     * @param gone
     */
    abstract public void announcePlayerLeft(PlayerToken gone);
    /**
     * Should broadcast all Player Status information
     * to both the networked players and the User Interface
     */
    abstract public void announcePlayerStatus();
    /**
     * Reorder the players in a such a manner so that the 
     * next round's order follows the same order as this
     * set of tokens
     * 
     * @param tokens Established new order for the next round
     */
    abstract public void reorderPlayers(PlayerToken tokens[]);
    
    /**
     * Allows the game to know what referee is operating.
     * 
     * @param ref the one who is talking directly to the players
     */
    abstract public void setReferee(Referee ref);
    
    abstract public void setGameBroadcaster(GameBroadcaster b);
    abstract public void setTimeBroadcaster(TimeBroadcaster t);
    abstract public void setPlayerBroadcaster(PlayerBroadcaster p);
}
