package netmatch;

import cytoscape.CyNetwork;
import cytoscape.Cytoscape;
import cytoscape.actions.GinyUtils;
import cytoscape.util.CyFileFilter;
import cytoscape.util.FileUtil;
import cytoscape.view.CyNetworkView;
import giny.model.GraphPerspective;
import giny.model.Node;
import giny.view.NodeView;
import giny.view.EdgeView;
import giny.util.SwingWorker;
import phoebe.PGraphView;

import javax.swing.*;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.*;
import java.awt.*;
import java.awt.event.*;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.text.NumberFormat;
import java.text.DateFormat;
import java.util.*;
import java.util.List;

import netmatch.algorithm.Common;

public class netMatchResultsDialog extends JPanel {
  protected JTable table;
  JScrollPane scrollPane;
  protected netMatchResultsDialog.netMatchResultsTableModel model;
  protected final int defaultRowHeight = Common.imageSize;
  protected boolean openAsNewChild = false;
  protected GraphPerspective[] gpComplexArray;    //The list of complexes, sorted by score when !null
  CyNetwork originalInputNetwork;                 //Keep a record of the original input record for use in the
  CyNetworkView originalInputNetworkView;         //Keep a record of this too, if it exists
  HashMap hmNetworkNames;                         //Keep a record of network names we create from the table
  JButton saveButton;
  JCheckBox newWindowCheckBox;
  //JCheckBox slideShowCheckBox;

  //If imageList is present, will use those images for the complex display
  public netMatchResultsDialog() {
    setLayout(new BorderLayout());
    hmNetworkNames = new HashMap();
    JPanel panel = new JPanel();
    panel.setLayout(new BorderLayout());
    panel.setBackground(Color.WHITE);
    table = new JTable() {
      public String getToolTipText(MouseEvent e) {
        Point p = e.getPoint();
        int rowIndex = rowAtPoint(p);
        int colIndex = columnAtPoint(p);
        int realColumnIndex = convertColumnIndexToModel(colIndex);
        String tip = getColumnName(realColumnIndex);
        if(realColumnIndex == 0 || realColumnIndex == 1)
          tip += ":" + getValueAt(rowIndex, colIndex);
        /*if(realColumnIndex == 0) {
          String value = (String)getValueAt(rowIndex,colIndex);
          System.out.println("value:"+value);
          String nodes = null;
          String edges = null;
          int i = 0;
          for(StringTokenizer st = new StringTokenizer(value,",");st.hasMoreTokens();i++) {
            String s = st.nextToken();
            System.out.println("String:"+s+" "+i);
            if(i == 0)
              nodes = s;
            else if(i == 1)
              edges = s;
          }
          tip = "SubGraph has " + nodes + " nodes and " + edges + " edges.";
        }*/
        return tip;
      }
    };
    table.setRowHeight(defaultRowHeight);
    table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    table.setDefaultRenderer(String.class, new CenterAndBoldRenderer());
    table.setDefaultRenderer(StringBuffer.class, new JTextAreaRenderer());
    ListSelectionModel rowSM = table.getSelectionModel();
    rowSM.addListSelectionListener(new TableRowSelectionHandler());
    scrollPane = new JScrollPane(table);
    scrollPane.getViewport().setBackground(Color.WHITE);
    panel.add(scrollPane, BorderLayout.CENTER);
    JPanel bottomPanel = new JPanel();
    newWindowCheckBox = new JCheckBox("Create a new child network.", false) {
      public JToolTip createToolTip() {
        return new JMultiLineToolTip();
      }
    };

    /*slideShowCheckBox = new JCheckBox("SlideShow",false);
    slideShowCheckBox.addItemListener(new netMatchResultsDialog.slideShowCheckBoxAction());
    slideShowCheckBox.setEnabled(false);
    bottomPanel.add(slideShowCheckBox);*/

    newWindowCheckBox.setToolTipText("If checked, a new child network of the selected match will be created. \r\n" +
            "Otherwise, the occurrence of the query match in the network window will be highlighted.");
    newWindowCheckBox.addItemListener(new netMatchResultsDialog.newWindowCheckBoxAction());
    newWindowCheckBox.setEnabled(false);
    bottomPanel.add(newWindowCheckBox);

    saveButton = new JButton("Save");
    saveButton.setToolTipText("Save result summary to a file.");
    saveButton.setEnabled(false);
    bottomPanel.add(saveButton);
    panel.add(bottomPanel, BorderLayout.SOUTH);
    add(panel, BorderLayout.CENTER);
  }

  public void set(ArrayList complexes, CyNetwork network, Image imageList[]) {
    originalInputNetwork = network;
    originalInputNetworkView = Cytoscape.getNetworkView(network.getIdentifier());
    model = new netMatchResultsDialog.netMatchResultsTableModel(network, complexes, imageList);
    table.setModel(model);
    table.setVisible(true);
    initColumnSizes(table);
    saveButton.addActionListener(new netMatchResultsDialog.SaveAction(complexes, network));
    saveButton.setEnabled(true);
    newWindowCheckBox.setEnabled(true);
    //slideShowCheckBox.setEnabled(true);
  }

  public void clear() {
    originalInputNetwork = null;
    originalInputNetworkView = null;
    ActionListener al[] = saveButton.getActionListeners();
    for(int i = 0;i < al.length;i++)
      saveButton.removeActionListener(al[i]);
    saveButton.setEnabled(false);
    newWindowCheckBox.setEnabled(false);
    //slideShowCheckBox.setEnabled(false);
    table.setVisible(false);
  }

  /**
   * Handles the data to be displayed in the table in this dialog box
   */
  private class netMatchResultsTableModel extends AbstractTableModel {
    String[] columnNames = {"Match Number","Nodes","Image"};
    Object[][] data;    //the actual table data

    public netMatchResultsTableModel(CyNetwork network, ArrayList complexes, Image imageList[]) {
      GraphPerspective gpComplex;
      gpComplexArray = convertComplexListToNetworkList(complexes, network);
      data = new Object[gpComplexArray.length][columnNames.length];
      for(int i = 0;i < gpComplexArray.length;i++) {
        gpComplex = gpComplexArray[i];
        data[i][0] = (new Integer(i + 1)).toString();
        data[i][1] = getNodeNameList(gpComplex);
        data[i][2] = "No Image";
        if(imageList != null)
          data[i][2] = new ImageIcon(imageList[i]);
      }
    }

    public String getColumnName(int col) {
      return columnNames[col];
    }

    public int getColumnCount() {
      return columnNames.length;
    }

    public int getRowCount() {
      return data.length;
    }

    public Object getValueAt(int row, int col) {
      return data[row][col];
    }

    public Class getColumnClass(int c) {
      return getValueAt(0, c).getClass();
    }

    private GraphPerspective convertComplexToNetwork(int[] complex, CyNetwork sourceNetwork) {
      GraphPerspective gpComplex;
      gpComplex = sourceNetwork.createGraphPerspective(complex);
      return gpComplex;
    }

    private GraphPerspective[] convertComplexListToNetworkList(ArrayList complexList, CyNetwork sourceNetwork) {
      GraphPerspective gpComplexArray[] = new GraphPerspective[complexList.size()];
      for(int i = 0;i < complexList.size();i++) {
        gpComplexArray[i] = convertComplexToNetwork((int[])complexList.get(i), sourceNetwork);
      }
      return gpComplexArray;
    }

    /**
     * Utility method to get the names of all the nodes in a GraphPerspective
     *
     * @param gpInput The input graph perspective to get the names from
     * @return A concatenated set of all node names (separated by a comma)
     */
    private StringBuffer getNodeNameList(GraphPerspective gpInput) {
      Iterator i = gpInput.nodesIterator();
      StringBuffer sb = new StringBuffer();
      while(i.hasNext()) {
        Node node = (Node)i.next();
        sb.append(node.getIdentifier());
        if(i.hasNext()) {
          sb.append(", ");
        }
      }
      return (sb);
    }

    private Image convertNetworkToImage(GraphPerspective gpInput, int height, int width) {
      PGraphView view;
      //SpringEmbeddedLayouter lay;
      Image image;

      view = new PGraphView(gpInput);
      for(Iterator in = view.getNodeViewsIterator();in.hasNext();) {
        NodeView nv = (NodeView)in.next();
        String label = nv.getNode().getIdentifier();
        nv.getLabel().setText(label);
        nv.setWidth(40);
        nv.setHeight(40);
        nv.setShape(NodeView.ELLIPSE);
        nv.setUnselectedPaint(Color.red);
        nv.setBorderPaint(Color.black);
        nv.setXPosition(view.getCanvas().getLayer().getGlobalFullBounds().getWidth() * Math.random());
        nv.setYPosition((view.getCanvas().getLayer().getGlobalFullBounds().getHeight() + 100) * Math.random());
      }
      for(Iterator ie = view.getEdgeViewsIterator();ie.hasNext();) {
        EdgeView ev = (EdgeView)ie.next();
        ev.setUnselectedPaint(Color.blue);
        ev.setTargetEdgeEnd(EdgeView.BLACK_ARROW);
        ev.setTargetEdgeEndPaint(Color.CYAN);
        ev.setSourceEdgeEndPaint(Color.CYAN);
        ev.setStroke(new BasicStroke(5f));
      }
      //lay = new SpringEmbeddedLayouter(view);
      //lay.doLayout();
      image = view.getCanvas().getLayer().toImage(width, height, null);
      /*double largestSide = view.getCanvas().getLayer().getFullBounds().width;
      if(view.getCanvas().getLayer().getFullBounds().height > largestSide) {
        largestSide = view.getCanvas().getLayer().getFullBounds().height;
      }*/
      return image;
    }
  }

  /**
   * Utility method to initialize the column sizes of the table
   *
   * @param table Table to initialize sizes for
   */
  private void initColumnSizes(JTable table) {
    TableColumn column;

    for(int i = 0;i < 3;i++) {
      column = table.getColumnModel().getColumn(i);
      column.sizeWidthToFit();
    }
  }

  /**
   * Handles the Save press for this dialog (save results to a file)
   */
  private class SaveAction extends AbstractAction {
    private ArrayList complexes;
    private CyNetwork network;

    /**
     * Save action constructor
     *
     * @param complexes Complexes to save
     * @param network   Network complexes are from for information about complex components
     */
    SaveAction(ArrayList complexes, CyNetwork network) {
      super("");
      this.complexes = complexes;
      this.network = network;
    }

    public void actionPerformed(ActionEvent e) {
      File file = FileUtil.getFile("Save Graph as Interactions", FileUtil.SAVE, new CyFileFilter[] {});
      if(file != null) {
        String fileName = file.getAbsolutePath();
        saveMCODEResults(complexes, network, fileName);
      }
    }

    /**
     * Save MCODE results to a file
     *
     * @param complexes The list of complexes
     * @param network   The network source of the complexes
     * @param fileName  The file name to write to
     * @return True if the file was written, false otherwise
     */
    public boolean saveMCODEResults(ArrayList complexes, CyNetwork network, String fileName) {
      if(complexes == null || network == null || fileName == null)
        return false;
      String lineSep = System.getProperty("line.separator");
      try {
        File file = new File(fileName);
        FileWriter fout = new FileWriter(file);
        fout.write("NetMatch Plugin Results" + lineSep);
        fout.write("Date: " + DateFormat.getDateTimeInstance().format(new Date()) + lineSep + lineSep);
        fout.write("Match Number\tNodeCount\tEdgeCount\tNames" + lineSep);
        //get GraphPerspectives for all complexes, score and rank them
        //convert the ArrayList to an array of GraphPerspectives and sort it by complex score
        GraphPerspective[] gpComplexArray = convertComplexListToNetworkList(complexes, network);
        for(int i = 0;i < gpComplexArray.length;i++) {
          GraphPerspective gpComplex = gpComplexArray[i];
          fout.write((i + 1) + "\t\t");
          NumberFormat nf = NumberFormat.getInstance();
          nf.setMaximumFractionDigits(3);
          fout.write(gpComplex.getNodeCount() + "\t\t");
          fout.write(gpComplex.getEdgeCount() + "\t\t");
          fout.write(getNodeNameList(gpComplex).toString() + lineSep);
        }
        fout.close();
        return true;
      }
      catch(IOException e) {
        JOptionPane.showMessageDialog(null, e.toString(), "Error Writing to \"" + fileName + "\"", JOptionPane.ERROR_MESSAGE);
        return false;
      }
    }

    /**
     * Utility method to get the names of all the nodes in a GraphPerspective
     *
     * @param gpInput The input graph perspective to get the names from
     * @return A concatenated set of all node names (separated by a comma)
     */
    private StringBuffer getNodeNameList(GraphPerspective gpInput) {
      Iterator i = gpInput.nodesIterator();
      StringBuffer sb = new StringBuffer();
      while(i.hasNext()) {
        Node node = (Node)i.next();
        sb.append(node.getIdentifier());
        if(i.hasNext()) {
          sb.append(", ");
        }
      }
      return (sb);
    }

    private GraphPerspective convertComplexToNetwork(int[] complex, CyNetwork sourceNetwork) {
      GraphPerspective gpComplex;
      gpComplex = sourceNetwork.createGraphPerspective(complex);
      return gpComplex;
    }

    private GraphPerspective[] convertComplexListToNetworkList(ArrayList complexList, CyNetwork sourceNetwork) {
      GraphPerspective gpComplexArray[] = new GraphPerspective[complexList.size()];
      for(int i = 0;i < complexList.size();i++) {
        gpComplexArray[i] = convertComplexToNetwork((int[])complexList.get(i), sourceNetwork);
      }
      return gpComplexArray;
    }
  }


  /**
   * Handles the new window parameter choice
   */
  private class newWindowCheckBoxAction implements ItemListener {
    public void itemStateChanged(ItemEvent e) {
      openAsNewChild = e.getStateChange() != ItemEvent.DESELECTED;
    }
  }

  /**
   * Handler to selects nodes in graph or create a new network when a row is selected
   * Note: There is some fairly detailed logic in here to deal with all the cases that a user can interact
   * with this dialog box. Be careful when editing this code.
   */
  private class TableRowSelectionHandler implements ListSelectionListener {
    public void valueChanged(ListSelectionEvent e) {
      //Ignore extra messages.
      if(e.getValueIsAdjusting())
        return;
      ListSelectionModel lsm = (ListSelectionModel)e.getSource();
      final GraphPerspective gpComplex;
      NodeView nv;
      if(!lsm.isSelectionEmpty()) {
        final int selectedRow = lsm.getMinSelectionIndex();
        gpComplex = gpComplexArray[selectedRow];
        //only do this if a view has been created on this network
        if(originalInputNetworkView != null) {
          //start with no selected nodes
          GinyUtils.deselectAllNodes(originalInputNetworkView);
          //go through graph and select nodes in the complex
          List nodeList = gpComplex.nodesList();
          for(int i = 0;i < nodeList.size();i++) {
            Node n = (Node)nodeList.get(i);
            if(originalInputNetwork.containsNode(n)) {
              nv = originalInputNetworkView.getNodeView(n);
              nv.setSelected(true);
            }
          }
          if(!openAsNewChild) {
            //switch focus to the original network if not going to create a new network
            Cytoscape.getDesktop().setFocus(originalInputNetworkView.getIdentifier());
          }
        }
        else if(!openAsNewChild) {
          //Warn user that nothing will happen in this case because there is no view to select nodes with
          JOptionPane.showMessageDialog(Cytoscape.getDesktop(), "You must have a network view created to select nodes.");
        }
        if(openAsNewChild) {
          NumberFormat nf = NumberFormat.getInstance();
          nf.setMaximumFractionDigits(3);
          final String title = "Match " + (selectedRow + 1);
          //check if a network has already been created
          String id = (String)hmNetworkNames.get(new Integer(selectedRow + 1));
          if(id != null) {
            //just switch focus to the already created network
            Cytoscape.getDesktop().setFocus(id);
          }
          else {
            //create the child network and view
            final SwingWorker worker = new SwingWorker() {
              public Object construct() {
                CyNetwork newNetwork = Cytoscape.createNetwork(gpComplex.getNodeIndicesArray(), gpComplex.getEdgeIndicesArray(), title, originalInputNetwork);
                hmNetworkNames.put(new Integer(selectedRow + 1), newNetwork.getIdentifier());
                PGraphView view = (PGraphView)Cytoscape.createNetworkView(newNetwork);
                //layout new complex and fit it to window
                //randomize node positions before layout so that they don't all layout in a line
                //(so they don't fall into a local minimum for the SpringEmbedder)
                //If the SpringEmbedder implementation changes, this code may need to be removed
                NodeView nv;
                for(Iterator in = view.getNodeViewsIterator();in.hasNext();) {
                  nv = (NodeView)in.next();
                  nv.setXPosition(view.getCanvas().getLayer().getGlobalFullBounds().getWidth() * Math.random());
                  //height is small for many default drawn graphs, thus +100
                  nv.setYPosition((view.getCanvas().getLayer().getGlobalFullBounds().getHeight() + 100) * Math.random());
                }
                //SpringEmbeddedLayouter lay = new SpringEmbeddedLayouter(view);
                //lay.doLayout();
                view.fitContent();
                return null;
              }
            };
            worker.start();
          }
        }
      }
    }
  }

  /**
   * A table cell rendered that centers the item in the cell
   */
  private class CenterAndBoldRenderer extends DefaultTableCellRenderer {
    public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected,
                                                   boolean hasFocus, int row, int column) {
      Component cell = super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
      cell.setFont(new Font(this.getFont().getFontName(), Font.BOLD, 14));
      this.setHorizontalAlignment(CENTER);
      this.setVerticalAlignment(NORTH);
      return cell;
    }
  }

  /**
   * A text area renderer that creates a line wrapped, non-editable text area
   */
  private class JTextAreaRenderer extends JTextArea implements TableCellRenderer {

    /**
     * Constructor
     */
    public JTextAreaRenderer() {
      this.setLineWrap(true);
      this.setWrapStyleWord(true);
      this.setEditable(false);
      this.setAutoscrolls(true);
    }

    /**
     * Used to render a table cell.  Handles selection color and cell heigh and width.
     * Note: Be careful changing this code as there could easily be infinite loops created
     * when calculating preferred cell size as the user changes the dialog box size.
     *
     * @param table      Parent table of cell
     * @param value      Value of cell
     * @param isSelected True if cell is selected
     * @param hasFocus   True if cell has focus
     * @param row        The row of this cell
     * @param column     The column of this cell
     * @return The cell to render by the calling code
     */
    public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected,
                                                   boolean hasFocus, int row, int column) {
      StringBuffer sb = (StringBuffer)value;
      this.setText(sb.toString());
      if(isSelected) {
        this.setBackground(table.getSelectionBackground());
        this.setForeground(table.getSelectionForeground());
      }
      else {
        this.setBackground(table.getBackground());
        this.setForeground(table.getForeground());
      }
      //row height calculations
      int currentRowHeight = table.getRowHeight(row);
      this.setSize(table.getColumnModel().getColumn(column).getWidth(), currentRowHeight);
      int textAreaPreferredHeight = (int)this.getPreferredSize().getHeight();
      //JTextArea can grow and shrink here
      if(currentRowHeight < textAreaPreferredHeight) {
        //grow row height
        table.setRowHeight(row, textAreaPreferredHeight);
      }
      else if((currentRowHeight > textAreaPreferredHeight) && (currentRowHeight != defaultRowHeight)) {
        //defaultRowHeight check in if statement avoids infinite loop
        //shrink row height
        table.setRowHeight(row, defaultRowHeight);
      }
      return this;
    }
  }

  /*private class slideShowCheckBoxAction implements Runnable,ItemListener {
    slideShow s;
    Thread t = null;

    public void itemStateChanged(ItemEvent e) {
      if(e.getStateChange() == ItemEvent.SELECTED) {
        //System.out.println("selected");

        t = new Thread(this);
        t.start();
      }
      else if(e.getStateChange() == ItemEvent.DESELECTED) {
        //System.out.println("deselected");
        if(s != null) {
          //System.out.println("stop");
          s.stopp();
        }
        if(t != null) {
          t.stop();
          t = null;
        }
      }
    }

    public void run() {
      int selectedRow = 0;
      //ListSelectionModel lsm = table.getSelectionModel();
      //lsm.setSelectionInterval(0,table.getRowCount()-1);
      //lsm.
      s = new slideShow(table);
      while(true) {
        s.set(selectedRow);
        selectedRow++;
        if(selectedRow == table.getRowCount())
          selectedRow = 0;
        //System.out.println("event");
        try {
          Thread.sleep(1000);
        }
        catch(InterruptedException e) {
          System.err.println(e.toString());
        }
      }
    }
  }

  private class slideShow extends Thread {
    private int i = -1;
    private JTable table;
    //private ListSelectionModel lsm;

    public slideShow(JTable table) {
      this.table = table;
      //this.lsm = table.getSelectionModel();
      start();
    }

    public synchronized void set(int val) {
      i = val;
      notify();
    }

    public synchronized void run() {
      while(true) {
        if(i == -1) {
          try {
            wait();
          }
          catch(InterruptedException e) {
            e.printStackTrace();
          }
        }
        if(i == -1)
          break;
        //System.out.println("I:"+i);
        //lsm.setLeadSelectionIndex(i);
        //lsm.setAnchorSelectionIndex(i);
        table.setRowSelectionInterval(i,i);
        i = -1;
      }
    }

    public synchronized void stopp() {
      i = -1;
      notify();
    }
  }*/
}