package heurgame.logging;

import java.util.Vector;
import java.util.HashMap;
import java.util.ListIterator;

/**
 * @author David Kaplin
 *
 * A useable implemntation of the AbstractLog.  This log seeks out its sources
 * and then notifies all of its readers.  Since this is specific to this single
 * log entity it is thread safe.  There are no guarentees that its readers
 * and writers are thread safe.
 */
public class DefaultLog extends AbstractLog {
    private Vector readers,writers;
    private HashMap authorMap;
    private volatile boolean broadcast;
    private volatile long waitInterval;
    
    private void setup(){
        readers = new Vector();
        writers = new Vector();
        authorMap = new HashMap();
        broadcast = false;
        waitInterval = 100;
    }
    
    
    /**
     * Creates a new Log
     * Ensures the log will update itself every 100 milliseconds 
     */
    public DefaultLog(){
        setup();
    }
    /**
     * Constructs a new Log and allows it to be updated at a
     * specified rate
     * 
     * @param interval Items will be collected and reported once in this time
     */
    public DefaultLog(long interval){
        setup();
        waitInterval = interval;
    }
    /** 
     * @see heurgame.logging.AbstractLog#addLogWriter(heurgame.logging.LogWriter)
     */
    public void addLogWriter(LogWriter w) {
        String name = w.getLogName();
        authorMap.put(w,name);
        writers.add(w);
    }

    /**
     * @see heurgame.logging.AbstractLog#addLogReader(heurgame.logging.LogReader)
     */
    public void addLogReader(LogReader r) {
        readers.add(r);
    }

    /**
     * Clears all of the Readers via the LogReader interface.
     * @see heurgame.logging.AbstractLog#clear()
     */
    public void clear() {
        for(int i=0;i<readers.size();i++){
            LogReader r = (LogReader)readers.get(i);
            r.clearLog();
        }
    }

    /** 
     * Spawns a thread that reads the output from the LogWriters,
     * and writes that output to the LogReaders.
     * 
     * This is the nature of pipes.
     * 
     * @see heurgame.logging.AbstractLog#startLogging()
     */
    public void startLogging() {
        broadcast = true;
        final LogItemComparator lic = new LogItemComparator();
        Runnable update = new Runnable(){
          public void run(){
              Vector highPriority = new Vector();
              Vector normal = new Vector();
              while (broadcast){
                  try {
                      Thread.sleep(waitInterval);
                      ListIterator writeIt = writers.listIterator();
                      highPriority.clear();
                      normal.clear();
                      while(writeIt.hasNext()){
                          LogWriter w = (LogWriter)writeIt.next();
                          String owner = (String)authorMap.get(w);
                          Vector v = new Vector(w.urgentEntries());
                          ListIterator innerIt = v.listIterator();
                          while(innerIt.hasNext()){
                              LogItem i = (LogItem)innerIt.next();
                              i.setOwner(owner);
                              highPriority.add(i);
                          }
                          w.urgentEntries().removeAll(v);
                          v = new Vector(w.getEntries());
                          innerIt = v.listIterator();
                          while(innerIt.hasNext()){
                              LogItem i = (LogItem)innerIt.next();
                              i.setOwner(owner);
                              normal.add(i);
                          }
                          w.getEntries().removeAll(v);
                      }
                      //sort the high priority
                      java.util.Collections.sort(highPriority,lic);
                      java.util.Collections.sort(normal,lic);
                      ListIterator readIt = readers.listIterator();
                      ListIterator listData = highPriority.listIterator();
                      while(listData.hasNext()){
                          LogItem i = (LogItem)listData.next();
                          readIt = readers.listIterator();
                          String who = i.getWho();
                          int priority = i.priority;
                          while(readIt.hasNext()){
                              LogReader r = (LogReader)readIt.next();
                              r.readLog(who,priority,i);
                          }    
                      }
                      //highPriority.clear();
                      listData = normal.listIterator();
                      while(listData.hasNext()){
                          LogItem i = (LogItem)listData.next();
                          readIt = readers.listIterator();
                          String who = i.getWho();
                          while(readIt.hasNext()){
                              LogReader r = (LogReader)readIt.next();
                              r.readLog(who,0,i);
                          }    
                      }
                      //normal.clear();
                  }catch (InterruptedException e){
                      e.printStackTrace();
                  }
              }
          }
        };
        Thread doit = new Thread(update);
        doit.start();
    }

    /** 
     * Terminates the broadcast thread
     * 
     * @see heurgame.logging.AbstractLog#stopLogging()
     */
    public void stopLogging() {
        broadcast = false;
    }

    /**
     * @see heurgame.logging.AbstractLog#removeLogWriter(heurgame.logging.LogWriter)
     */
    public void removeLogWriter(LogWriter w) {
        writers.remove(w);
        authorMap.remove(w);
    }

    /**
     * @see heurgame.logging.AbstractLog#removeLogReader(heurgame.logging.LogReader)
     */
    public void removeLogReader(LogReader r) {
        readers.remove(r);
    }
}
