package heurgame;
/*
 * Created on Oct 12, 2004
 */
import heurgame.logging.LogBox;

import java.net.Socket;
import java.io.OutputStreamWriter;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.PrintWriter;
import java.io.IOException;
/**
 * @author David Kaplin
 * 
 * HuClients are short for "HumanClients"  each of these act on the request
 * of a physical user who is occupying the same computer as the game server.
 * Instead of giving the protocol to the user directly, the status information
 * appears in some form of user interface.  When the player is instructed it
 * it their move they should interact with a component that is a HuReceptor
 * that can properly translate the users actions into the text that needs
 * to be translated over the network.
 */
public abstract class HuClient implements Runnable{
    protected LogBox debugging = new LogBox("HuClient");
    /**
     * <code>fromPlayer</code> The part we read from the player. 
     */
    protected BufferedReader fromServer;
    /**
     * <code>toPlayer</code> The part we write to the player.
     */
    protected PrintWriter toServer;
    /**
     * <code>move</code> is a protocol specific string that corresponds to a players move.
     */
    protected volatile String move;
    /**
     * <code>guiReceptor</code> is the component that the user interacts with to play the game.
     */
    protected HuReceptor guiReceptor;
    
    /**
     * The Initial portion of the outside input contract
     * that allows the HuClient access to user input.
     * 
     * @param h some component that will notify us when
     * we desire information.
     */
    public void setReceptor(HuReceptor h){
        guiReceptor = h;
        debugging.addEntry("Receptor Set","Contract with "+h.getClass().getName()+" established.");
    }
    /**
     * Another portion of the input contract
     * Notifies the component of some new external data
     * 
     * @param ans Specific to implementation.
     */
    public void fromOutside(String ans){
        move = ans;
        debugging.addEntry("Receptor Action",guiReceptor.getClass().getName()+" said: "+ ans);
    }
    /**
     * This method should be called before loading the HeuClient into a thread.
     * The player should report their name to the server.
     * 
     * @param s					An open bound constructed sockect
     * @param name				Represents this player
     * @throws IOException
     */
    public final void connect(Socket s,String name) throws IOException{
        fromServer = new BufferedReader(new InputStreamReader(s.getInputStream()));
        toServer = new PrintWriter(new BufferedWriter(new OutputStreamWriter(s.getOutputStream())));
        toServer.println(name);
        toServer.flush();
        debugging.addEntry("Connection Established","Send name as: "+name+" to "+ s.getInetAddress());
        move = new String();
    }
    /**
     * This should be called as the thread polls
     * 
     * @return Our next message from the Server
     */
    abstract public String waitForMessage();
    /**
     * Invoked when it is our turn to move.
     * It is suggested that the player save its move information 
     * in case it sent an invalid move.
     * 
     * @param query	The state of the game as defined by the server.
     * @return our response to the server.
     */
    abstract public String handleMove(String query);
    /**
     * Now knowing the reason for an invalid move,
     * the player should respond with a hopefully valid move.
     * 
     * @param reason why the move was invalid
     * @return another move from the player.
     */
    abstract public String handleInvalidMove(String reason);
    /**
     * Initial Status usually does not involve the actions of other players.  
     * It might be something like the size of the board.
     * 
     * @param status specific to game
     */
    abstract public void handleInitialStatus(String status);
    /**
     * Usuallly the state of other players during the game.
     * 
     * @param status game specific
     */
    abstract public void handleStatus(String status);
    /**
     * Gets the latest change in status
     * @param status game specific
     */
    abstract public void handleIncrementalStatus(String status);
    /**
     * Ususally the final outcome of the game.
     * 
     * @param status exepected to be something of the lines of WIN, LOSE, or TIE 
     */abstract public void handleFinalStatus(String status);
    /**
     * Informs us that we have been rejected from the game
     * @param reason for the rejection
     */
    abstract public void handleDisqualification(String reason);
}
