package nanomunchers.bot;

import java.awt.Point;

import nanomunchers.graph.Node;

/*
 * Created on Oct 26, 2004
 */

/**
 * Reprsents the humble Nanomuncher
 * 
 * @author David Kaplin
 */
public class NanoBot {
    public final static int UP = 4;

    public final static int LEFT = 3;

    public final static int DOWN = 2;

    public final static int RIGHT = 1;

    protected int nodesMunched;

    protected boolean alive;

    protected int cycle;
    protected int positionDelta;//Extra added to instruction pointer

    protected int[] instructionOrder;

    protected Node startPosition;

    protected Node location;
    private double id;
    /**
     * Creates a new nanomuncher.
     * 
     * @param instOrder "LRUD" for example
     * @param start
     */
    public NanoBot(String instOrder, Node start) {
        id = Math.floor(Math.random()*100.0);
        location = start;
        startPosition = start;
        String[] parts = instOrder.split(" ");
        instructionOrder = new int[parts.length];
        alive = true;
        nodesMunched = 1;
        location.mark();
        cycle= 0;
        for (int i = 0; i < parts.length; i++) {
            parts[i] = parts[i].toUpperCase();
            if (parts[i].equals("U")) {
                instructionOrder[i] = UP;
            } else if (parts[i].equals("D")) {
                instructionOrder[i] = DOWN;
            } else if (parts[i].equals("L")) {
                instructionOrder[i] = LEFT;
            } else if (parts[i].equals("R")) {
                instructionOrder[i] = RIGHT;
            }
        }
    }
    /**
     * @return the next move disregarding other nanomunchers that
     * the nanomuncher wants to take.
     */
    public Node nextMove(){
        positionDelta = 0;
        Node possible = null;
        while (positionDelta < instructionOrder.length){
            int directionIndex = (cycle+positionDelta) % instructionOrder.length;
            int direction  = instructionOrder[directionIndex];
            switch (direction) {
            case UP:
                possible = getUnmarkedUp();
                break;
            case DOWN:
                possible = getUnmarkedDown();
                break;
            case LEFT:
                possible = getUnmarkedLeft();
                break;
            case RIGHT:
                possible = getUnmarkedRight();
                break;
            }
            if (possible != null){
                return possible;
            }
            positionDelta++;
        }
        return possible;
    }
    private Node getUnmarkedLeft(){
        Node[] others = location.getNeighbors();
        int destX = location.getX()-1;
        int destY = location.getY();
        for(int i=0;i<others.length;i++){
            if (others[i].isMarked()){
                continue;
            }
            if ((others[i].getPoint().equals(new Point(destX,destY)))) {
                return others[i];
            }
        }
        return null;
    }
    private Node getUnmarkedRight(){
        Node[] others = location.getNeighbors();
        int destX = location.getX()+1;
        int destY = location.getY();
        for(int i=0;i<others.length;i++){
            if (others[i].isMarked()){
                continue;
            }
            if ((others[i].getPoint().equals(new Point(destX,destY)))) {
                return others[i];
            }
        }
        return null;
    }
    private Node getUnmarkedUp(){
        Node[] others = location.getNeighbors();
        int destX = location.getX();
        int destY = location.getY()-1;
        for(int i=0;i<others.length;i++){
            if (others[i].isMarked()){
                continue;
            }
            if ((others[i].getPoint().equals(new Point(destX,destY)))) {
                return others[i];
            }
        }
        return null;        
    }
    private Node getUnmarkedDown(){
        Node[] others = location.getNeighbors();
        int destX = location.getX();
        int destY = location.getY()+1;
        for(int i=0;i<others.length;i++){
            if (others[i].isMarked()){
                continue;
            }
            if ((others[i].getPoint().equals(new Point(destX,destY)))) {
                return others[i];
            }
        }
        return null;
    }
    
    /**
     * Actually performs the move.
     */
    public void move() {
        Node home = nextMove();
        if (alive && home != null){
            //System.out.println("Bot is marking graph...");
            int direction = cycle % instructionOrder.length;
            String oldDirection = nanoDirection(instructionOrder[direction]);
            //System.out.println("***"+id+"start "+location+ " Direction is "+ oldDirection+""+direction+" cycle "+cycle);
            home.mark();
            location = home;
            nodesMunched++;
            
            cycle+= 1+positionDelta;
            //cycle++;
            direction = cycle % instructionOrder.length;
            String newDirection = nanoDirection(instructionOrder[direction]);
            
            //System.out.println("***"+id+" end "+location+" facing "+ newDirection+""+direction+" code = "+getCode()+" cycle "+cycle);
        }
        positionDelta = 0;
    }

    /**
     * @return the nanomunchers current direction (facing its next move)
     */
    public int getDirection() {
        return instructionOrder[(cycle+positionDelta)% instructionOrder.length];
    }

    /**
     * @return number of nodes consumed.
     */
    public int getNodesMunched() {
        return nodesMunched;
    }

    /**
     * @return true if the nanomuncher is still kicking :)
     */
    public boolean isAlive() {
        return alive;
    }

    /**
     * Kills the nanomuncher.
     * Usually invoked during a conflict death
     */
    public void kill() {
        alive = false;
    }

    /**
     * Renews the nanomuncher to its inital place and diet.
     */
    public void clear() {
        cycle= 0;
        location = startPosition;
        nodesMunched = 1;
        location.mark();
        alive = true;
    }

    /**
     * @return node that the nanomuncher is currently occupying
     */
    public Node getCurrentLocation() {
        return location;
    }
    /**
     * @return "LRDU" for Left Right Down Up
     * "UNKNOWN" for other combinations.
     */
    public String getCode(){
        String nanoCode = "";
        for(int i=0;i<instructionOrder.length;i++){
            nanoCode += nanoDirection(instructionOrder[i]);
        }
        return nanoCode;
    }
    
    private String nanoDirection(int n){
        switch (n){
    	case UP:
    	    return "U";
    	case DOWN:
    	    return "D";
    	case LEFT:
    	    return "L";
    	case RIGHT:
    	    return "R";
    	    
        }
        return "UKNOWN";
    }
}