#ifndef SAT_H
#define SAT_H

#include <vector>
#include <set>
#include <map>
#include <math.h>

#include "SATNode.h"

using namespace std;

#define MAX_SHIFTMODE 1
#define MAX_SHIFTMODE2 2	// MAX_SHIFTMODE*2

#define UPDATE_WEIGHT 1
#define SEARCH_WEIGHT 1


class SATNode;

class SAT
{
	class SATLayer		// structure for one layer in SAT tree
	{
	public:
		int _h;			// height in pyramid, i.e, corresponding window size, start from 1
		int _s;			// shift 
		int _mins;		// min shift
		int _minds;		// min depending shift
		vector<SATLayer*> _childLayers;		// a list of children
		vector<int>		_sIdx;				// time offset index for each children

		int _threshHighIdx;		// upper index in thresholds
		int _threshHigh[MAX_SHIFTMODE2+1];
		int _active;			// 1: active, 2: candidate for add
		int _selfIdx;

		// aggregate
		double* _aggregate;		// buffer to store the aggregates at this layer, used as recycle buffer
		int _aggLen;			// how many aggregates within one cycle
		int _curIdx;			// index for current time point

		// for dynamic change
		int	   _shiftMode;
		int	   _shiftStep;

		int    _alarmCount;

		long   _nextShiftT[MAX_SHIFTMODE2+1];
		long   _nextShiftReady[MAX_SHIFTMODE2+1];
		long   _nextAddT;

		// cross ref for the layers
		int _numActiveChildLayers;
		set<SATLayer*> _unqChildLayers;
		int _numActiveDependLayers;
		map<SATLayer*, int> _dependLayers;

	public:
		SATLayer() : _h(-1), _s(-1), _mins(-1), _minds(-1), _aggregate(NULL), _aggLen(0), _curIdx(0), _threshHighIdx(-1),  _active(0), _selfIdx(-1),
					_alarmCount(0), 
					_shiftMode(MAX_SHIFTMODE), _shiftStep(1<<MAX_SHIFTMODE),
					_nextAddT(0), 
					_numActiveChildLayers(0), _numActiveDependLayers(0)
		{
			for (int i = 0; i < MAX_SHIFTMODE2+1; ++ i)
			{
				_nextShiftT[i] = 0;
				_nextShiftReady[i] = 0;
			}
		}
		virtual ~SATLayer() { 
			delete [] _aggregate; 
		}

		bool needUpdate(long t) { return (t%_s==0); }
		inline double aggVal() {				// get the aggregate value given index
			return _aggregate[_curIdx];
		}
		inline double aggVal(int idx) {			// get the aggregate value given index
			//idx = idx << _shiftMode;
			int t = _curIdx-idx;
			if (0 <= t && t < _aggLen)
				return _aggregate[t];
			else
			if (t < 0)
				return _aggregate[t+_aggLen];
			else
				return _aggregate[t-_aggLen];
		}
		inline void setAggVal(double val) {			// set aggregate value
			_aggregate[_curIdx] = val;
		}		
		inline void setAggVal(double val, int idx) {		// set aggregate value
			//idx = idx << _shiftMode;
			int t = _curIdx-idx;
			if (0 <= t && t < _aggLen)
				_aggregate[t] = val;
			else
			if (t < 0)
				_aggregate[t+_aggLen] = val;
			else
				_aggregate[t-_aggLen] = val;
		}		
		inline void incIdx() {	// increase the index 
			//_curIdx += 1<<_shiftMode;
			_curIdx += _shiftStep;
			if (_curIdx >= _aggLen)
			{
				_curIdx -= _aggLen;
			}
		}
		inline int maxCoveredSize() const {
			return (_h-_s+1);
		}
		inline bool isActive() const {
			return (_active == 1);
		}
		inline bool childrenActive() const {
			return (_numActiveChildLayers == _unqChildLayers.size());
		}
		inline bool narrowReady(int sm, long t) const {
			return (sm >= 0 && 
					_nextShiftReady[sm] == _unqChildLayers.size() && 
					_nextShiftT[sm] <= t);
		}
		inline void widen() {
			map<SATLayer*, int>::iterator sit;
			for (sit = _dependLayers.begin(); sit != _dependLayers.end(); ++ sit)
			{
				SATLayer* layer = (*sit).first;
				int ms = layer->_mins;
				if (_s >= ms)
				{
					int ii = 0;
					while (ms < _s)
					{
						ms = ms << 1;
						++ ii;
					}
					-- (layer->_nextShiftReady[ii]);
				}
			}
			if (_shiftMode < MAX_SHIFTMODE2) {
				++ _shiftMode;
				_shiftStep = _shiftStep << 1;
				_s = _s << 1;
				_threshHighIdx = _threshHigh[_shiftMode];
			}
		}
		inline void narrow(int t) {
			if (_shiftMode > 0 && _s > 1) {
				-- _shiftMode;
				_shiftStep = _shiftStep >> 1;
				_s = _s >> 1;
				_threshHighIdx = _threshHigh[_shiftMode];
			}
			map<SATLayer*, int>::iterator sit;
			for (sit = _dependLayers.begin(); sit != _dependLayers.end(); ++ sit)
			{
				SATLayer* layer = (*sit).first;
				int ms = layer->_mins;
				if (_s >= ms)
				{
					int ii = 0;
					while (ms < _s)
					{
						ms = ms << 1;
						++ ii;
					}
					int tmp = t + (*sit).second;
					if (layer->_nextShiftT[ii] < tmp)
						layer->_nextShiftT[ii] = tmp;
					++ (layer->_nextShiftReady[ii]);
				}
			}
		}
	};

	// members
public:

	// struct
	vector<SATLayer*> _layerBank;
	vector<SATLayer*> _sat;		// SAT tree structure
	int  _numLayer;			// total # of levels
	int	 _topWinSize;		// top-layer window length in the tree
	int  _maxWinSize;		// max interesting window length

	// thresholds
	vector<pair<int,double> >	_thresholds;
	int  _threshSize;

	// buffers for detail search
	double* _aggBuf;

	int  _dataLen;
	bool _bOutput;
	int  _burstRangeT;
	int  _burstRangeH;

public:
	SAT();
	virtual ~SAT();

	void readStruct(char* satFname, char* thresFname);		// read SAT and construct the tree

	void detect(long t, double d);
	void dynamicDetect(long t, double d);
	void batchDetect(double* buffer, long len);
	void incrementalDetect(char* dataFile);
	void incrementalDynamicDetect(char* dataFile);

	void setReportBurst(bool tf) { _bOutput = tf; }

	double density() const;
	double alarmProb() const;

protected:

	int	 loadThresholds(char* thresFile);

	bool getNextData(FILE* fp, double& d);

	int  hasBurst(double agg, int lowIdx, int highIdx);
	void reportBurst(long t, int h, double agg);
	int  detailSearch(long t, int l, double agg, int hidx);
	int  detailSearch2(long t, long offset, SATLayer* curLayer, SATLayer* lowerLayer, double agg, double lowAgg, int hidx);
	void flush(long t);

	void addBurstRange(long t, int h);
	bool inBurstRange(long t, int h);

	void addExtraLayers();
};

#endif
