/**
 * @author Dennis Shasha
 * @author Anastasia Kuznetsov
 * 10/01/2005
 */



package StratPal;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.ListIterator;

/**
 * A directed acyclic graph representation of deposition questions.
 *
 */

public class QuestionSet {

	private HashMap questions;
	
	
	/** Ordered list of questions. */
	private LinkedList orderedList;
	
	/** Iterator over questions. */
	ListIterator orderedListIter;
	
	/** Question that was just asked. */
	QNode prev;
	
	/** Queue of parents */
	private LinkedList parents;
	
	/** chilf iter. */
	private ListIterator childIter;
	
	/** All questions. */
	private LinkedList all;
	
	/** Current question. */
	private ListIterator current;
	
	/** Keyword iter. */
	private ListIterator kwordIter;
	
	/** list of keywords, in order as they were answered.*/
	private LinkedList keywords;


	/** Whether or not the user chose to override default ordering of questions.*/
	private boolean overide = false;
	
	/** Node at which user chose to override default ordering. */
	private QNode overideNode;


	private QNode last;
	
	/**
	 * Default constructor creates new hashmap.
	 */
	public QuestionSet(){
		questions = new HashMap();
		orderedList =new LinkedList();
		orderedListIter = orderedList.listIterator(0);
		parents = new LinkedList();
		keywords = new LinkedList();
		kwordIter = keywords.listIterator(0);
	}
	
	/**
	 * Adds question to the DAG.
	 * @param name String name of question.
	 * @param node QNode that represents the question.
	 */
	public void add(String name, QNode node){
		questions.put(name, node);
	}
	
	/** Add in order. */
	
	public void addOrdered(String name){
		orderedList.addLast(name);
	}
	/**
	 * Returns node that has the specified question name.
	 *@param name of question.
	 * @return QNnode quesiton node.
	 */
	public QNode get(String name){
		return (QNode)questions.get(name);
	}

	/**
	 * Determines which question should be asked next depending on which answer was given.
	 * @param ans answer to the current question
	 * @return QNode- question that should be asked next. 
	 * @throws EmptyDepositionException
	 */
	public QNode setNext(String ans) throws EmptyDepositionException{
				
		if(orderedList.isEmpty()){
			throw new EmptyDepositionException();
		}
		if (ans.compareTo("top")==0){
			orderedListIter = orderedList.listIterator(0);
			prev =get((String)orderedListIter.next());
			if (prev.hasChildren())
				parents.addFirst(prev);
			prev.setCurrent(true);
			getAllIter();
			current = all.listIterator(all.indexOf(prev));
			if(prev.hasChildren()){
				parents.addFirst(prev);
			}
			return prev;
		}

		prev.setCurrent(false);	
		prev.setKeyword(ans);
		
		if(overide){
			
			if(!current.hasNext())
				return null;
			
			prev = (QNode)current.next();
			prev.setCurrent(true);
			setSkipped();
			return prev;
		}
		
		setSkipped();
		if(prev.hasGoTo()){
			String nextName = findGoTo(prev);	
			if(nextName != null){
				if(nextName != null){
					prev = get(nextName);
					prev.setCurrent(true);
					orderedListIter = orderedList.listIterator(orderedList.indexOf(nextName)+1);
					current = all.listIterator(all.indexOf(prev));
					if(prev.hasChildren()){
						parents.addFirst(prev);
					}
					return prev;
				}
			}
		}
			
		if(parents.isEmpty()){
			return findNext();
			
		}
		
		QNode parent = (QNode)parents.getFirst();
		while(parent!=null){
		childIter = parent.getChildIter();
		String nextName=null;
		
			
		while(childIter.hasNext()){
			nextName = (String)childIter.next();
			if(get(nextName).isAnswered()||get(nextName).isSkipped())
				continue;
			else if(!canBeAsked(get(nextName))){
				get(nextName).setSkipped(true);
				
			}
			else break;
		}
		if (nextName != null && canBeAsked(get(nextName))){
			prev = get(nextName);
			prev.setCurrent(true);
			if(!childIter.hasNext()){
				
				parents.removeFirst();
				
			}
			if(prev.hasChildren()){
				parents.addFirst(prev);
				childIter = prev.getChildIter();
			}
			current = all.listIterator(all.indexOf(prev));
			return prev;
		}
		else{
			parents.removeFirst();
			if(!parents.isEmpty())
				parent = (QNode)parents.getFirst();
			else
				parent =null;
		}
		}
		return findNext();
		
	
	}
	
	/**
	 * Find next question that can be asked.
	 * 
	 * @return QNode- quesiton to be asked.
	 */
	private QNode findNext(){
		while(orderedListIter.hasNext()){
			QNode next = get((String)orderedListIter.next());
			
			if (canBeAsked(next)){
				prev = next;
				next.setCurrent(true);
				if(next.hasChildren()){
					parents.addFirst(next);
				}
				current = all.listIterator(all.indexOf(next));
				return next;
			}
			next.setSkipped(true);
		}
		
		return null;
	}
	

	/**
	 * Find name of question to go to.
	 * @param q current question	
	 * @return name of question to go to.
	 */
	private String findGoTo(QNode q) {
		ListIterator iter = q.goToIter();
		while(iter.hasNext()){
			GoTo gt = (GoTo)iter.next();
			if (get(gt.to).wasAnswered(gt.kword))
				return gt.goTo;
		}
		return null;
	}

	/**
	 * Check if this question can be asked.
	 * @param question specified question
	 * @return true if it can be asked.
	 */
	private boolean canBeAsked(QNode question) {
		if(question == null)
			return false;
		
		if(question.hasConditions()){
			HashMap ifs = question.getConditions();
			Iterator iter = ifs.keySet().iterator();
			while(iter.hasNext()){
				String qname = (String)iter.next();
				String kword = (String)ifs.get(qname);
				QNode prev = get(qname);
				if(prev.wasAnswered(kword))
					return true;
			}
			return false;
		}
		
		return true;
	}

	/**
	 * Get iterator over all questions in this set, in order.
	 * @return ListIterator over all questions in order.
	 */
	public ListIterator getAllIter(){
		if (all != null){
			
			return all.listIterator();
		}
		all = new LinkedList();
		ListIterator iter = orderedList.listIterator();
		while(iter.hasNext()){
			QNode node = get((String)(iter.next()));
			insertChildren(node);
		}
		return all.listIterator();
	}

	/**
	 * Insert Children (subquestions) to this question.
	 * @param node subquestion to be added.
	 */
	private void insertChildren(QNode node) {
		all.addLast(node);
		if(node.hasChildren()){
			ListIterator iter=node.getChildIter();
			while(iter.hasNext())
				insertChildren(get((String)iter.next()));
		}
	}
	
	/**
	 * Move back one question.
	 * @param done true if deposition ended.
	 * @return QNode- question to be asked.
	 */
	public QNode back(boolean done) {
		if (prev == null)
			return null;
		String temp = prev.getAnswer();
		if(done){
			prev.setCurrent(true);
			prev.setKeyword("");
			setSkipped();
			prev.setKeyword(temp);
			return prev;
		}
		
		prev.setKeyword("");
		prev.setCurrent(false);
		
		if(parents.contains(prev.getName()) || prev.hasChildren())
			parents.removeFirst();
		
		if(prev==(QNode)all.getFirst()){
			prev.setCurrent(true);
			setSkipped();
			prev.setKeyword(temp);
			return prev;
			
		}
		if(overide){
			
			QNode previous;
			do{ 
				previous = (QNode)current.previous();
			
			}while(previous.getName().compareTo(prev.getName())==0);
			
			if(prev.getName().compareTo(overideNode.getName())!=0){
				
				prev = previous;
				kwordIter.add(prev.getAnswer());
				
				prev.setCurrent(true);
				setSkipped();
				prev.setKeyword(temp);
				return prev;
			}
			else
			{
				
				reset();
				prev.setKeyword(temp);
				return prev;
			}
	
		}
		
		while(current.hasPrevious()){
			QNode previous = (QNode)current.previous();
			
			if(previous.isAnswered()){
				prev = previous;
				if(orderedList.contains(prev.getName()))
					orderedListIter = orderedList.listIterator(orderedList.indexOf(prev.getName())+1);
				current = all.listIterator(all.indexOf(prev));
				prev.setCurrent(true);
				if(prev.hasChildren()){
					parents.addFirst(prev);
					
				}
				temp= prev.getAnswer();
				prev.setKeyword("");
				setSkipped();
				prev.setKeyword(temp);
				kwordIter.add(prev.getAnswer());
				
				return prev;
			}
			
		}
		
		return(null);
		
	}
	
	
	/**
	 * Set all questions that can't be asked as skipped.
	 *
	 */
	private void setSkipped(){
		ListIterator itr = all.listIterator();
	
		while(itr.hasNext()){
			
			QNode node = (QNode)itr.next();

			if(!shouldBeSkipped(node)){
				node.setSkipped(false);
			
			}
			else{
				node.setSkipped(true);
				System.out.println("skipping "+node.getName()+"current is "+ prev.getName());
				if(node.hasChildren()){
					skipChildren(node, itr);
					System.out.println("skipping "+node.getName()+"current is "+ prev.getName());
				}
			}
		}
	}

	/**
	 * Set all subquestions (children) as skipped.
	 * @param node question whose subquestions are being examined
	 * @param itr Iterator of over questions children
	 */
	private void skipChildren(QNode node, ListIterator itr) {
		
		node.setSkipped(true);
			
		if(node.hasChildren()){
			
			ListIterator iter=node.getChildIter();
			while(iter.hasNext()){
				itr.next();
				skipChildren(get((String)iter.next()), itr);
			}
		}
	}
	
	/**
	 * Check is this question should be skipped.
	 * @param question specified question
	 * @return true if it should be skipped.
	 */
	private boolean shouldBeSkipped(QNode question) {
		
		if(overide && 
				(all.indexOf(question.getName())>=(all.indexOf(overideNode.getName())-1))) {	
			
			return false;
		}
		
		if(question == null)
			return false;
		
		if(question.hasConditions()){
			
			HashMap ifs = question.getConditions();
			Iterator iter = ifs.keySet().iterator();
			while(iter.hasNext()){
				String qname = (String)iter.next();
				
				String kword = (String)ifs.get(qname);
				QNode prev = get(qname);
			//	if(!prev.isAnswered())
					if(!prev.isAnswered() || prev.wasAnswered(kword)){
						return false;
				}
			}
			return true;
		}
		
		return false;
	}
	
	/**
	 * Advance questions forward
	 * @return QNode next question to be asked.
	 */
	public QNode forward(){
	
		if(keywords.isEmpty()||!kwordIter.hasPrevious())
			return null;
		try {
			return(setNext((String)kwordIter.previous()));
		} catch (EmptyDepositionException e) {
			return null;
		}
	}
	
	/**
	 * Set increment level of all questions.
	 *
	 */
	public void setIncrement(){
		ListIterator itr = getAllIter();
		while(itr.hasNext()){
			QNode nd = (QNode)itr.next();
			if(nd.hasChildren())
				setIncrement(nd, 0, itr);
		}
	}
		

	/**
	 * Set increment of particular question
	 * @param node question to be indented
	 * @param i indent amount
	 * @param itr iterator pointing to this question in a list of all questions.
	 */
	private void setIncrement(QNode node, int i, ListIterator itr){
		node.incrementIndent(i);
		itr.next();
		if(node.hasChildren()){
			i+=25;
			ListIterator iter=node.getChildIter();
			while(iter.hasNext()){
				itr.next();
				setIncrement(get((String)iter.next()), i, itr);
			}
		}
	}

	/**
	 * If user chose to ask a question out of order, this method sets internal variables to overide mode.
	 * @param node question that user selected
	 * @throws EmptyDepositionException
	 */
	public void overide(QNode node) throws EmptyDepositionException  {
		last = prev;
		overide = true;
		overideNode = node;
		current = all.listIterator(all.indexOf(node));
	//	prev = (QNode)all.get(all.indexOf(node)-1);
		setNext("");
	
	}

	/**
	 * Reset deposition back to regular order if override occured and users back to original spot.
	 *
	 */
	public void reset(){
	
		overide = false;
		overideNode = null;
		prev = last;
		prev.setCurrent(true);
		setSkipped();
		current = all.listIterator(all.indexOf(prev));
	
	}
	
	/**
	 * After forwarding, reset to normal answer selection.
	 * @param f true if user is forwarding.
	 */
	public void resetForwarding(boolean f){
		if(!f){
			keywords= new LinkedList();
			kwordIter = keywords.listIterator();
		}
	}
	
	/**
	 * Print entire question set. For debug only.
	 *
	 */
	private void print(){
	
		ListIterator itr = getAllIter();
		while(itr.hasNext()){
			QNode question = (QNode) itr.next();
			System.out.println(question.getName()+"---------------------");
			if(question.hasConditions()){
				
				HashMap ifs = question.getConditions();
				Iterator iter = ifs.keySet().iterator();
				while(iter.hasNext()){
					String qname = (String)iter.next();
					
					String kword = (String)ifs.get(qname);
					System.out.print("  "+kword+" to "+qname);
				}
				System.out.println();
			}
			
			if(question.hasChildren()){
				System.out.print("  children: ");
				ListIterator child= question.getChildIter();
				while(child.hasNext()){
					System.out.print(" "+(String)child.next());
				}
				System.out.println();
			}
			
		}
		System.out.println("\n\n");
	}
}

