#include "algebra.h"

////////////////////////////////////////////
//
// Class: PathTable. Use the Defination in algebra.h
// Usage: Used for GraphGrep/GraphBuild. Storing all id-path information of a label-path pattern. 
//		  Mainly used in inserting data into BerkeleyDB and fetching data for algebra operation.
//
/////////////////////////////////////////////

/*
	Constructor classes
*/
PathTable::PathTable(const char* name,NodeType* data){
	if(data!=NULL){		//checking if the input data is NULL
		_name=string(name);
		CreateTable(name,data);		//call the real inital function
	}else{
		throw GraphGrepExcp("NULL data\n");
	}
}

PathTable::PathTable(const char* name,int rows,int cols,vector<NodeType*> data){
	_name=string(name);
	CreateTablefromResult(name,rows,cols,data);
}

PathTable::PathTable(string tabname,PathTable& intab){
	//setting data member
	_name=tabname;
	_rows=intab.get_rows();
	_cols=intab.get_cols();

	NodeType** pathdata=intab.get_alldata();	//get the data pointer of input PathTable

	//allocate mem space
	_all_pathes=(NodeType**)operator new(_rows*sizeof(NodeType*));		
	
	//copy data of each pattern
	for(int i=0;i<_rows;i++){
		NodeType* innerArr=(NodeType*)operator new (sizeof(NodeType)*_cols);

		for(int j=0;j<_cols;j++)
			innerArr[j]=pathdata[i][j];
		
		_all_pathes[i]=innerArr;
	}
	
}

//Realloc memory for all possible paths
void PathTable::resetPaths(){
	//"_tmp_counter" is an int that record the current size of the path-table
	NodeType** tmp_pathes=(NodeType**)operator new(_tmp_counter*sizeof(NodeType*));
	for(int i=0;i<_rows;i++){			
		tmp_pathes[i]= _all_pathes[i];
	}

	delete _all_pathes;
	_all_pathes=NULL;
	_all_pathes=tmp_pathes;
}

//Insert a row into current table
void PathTable::insertData(NodeType* rowdata){
	
	//check if the tmp space is enough for this row, if not, realloc space
	if(_tmp_counter < (_rows+1)){
		_tmp_counter *=2;	//rellocate twice space
		resetPaths();
	}

	//Copy each node
	NodeType* data=new NodeType[_cols];
	for(int i=0;i<_cols;i++){
		data[i]=rowdata[i];	
	}

	_all_pathes[_rows]=data;
	
	_rows++;
}

//Destructor
PathTable::~PathTable(){
	for(int i=0;i<_rows;i++){
		operator delete(_all_pathes[i]);
	}
	operator delete(_all_pathes);

}

/*
//Initiate function that takes 1. table name 2. Array of NodeType as parameters
// The data in the array are:
// pos[0-1]: 		total rows of this table
// pos[2]:    		columns of this table
// pos[3-end]: 	Node data
*/
void PathTable::CreateTable(const char* tabname,NodeType* pathdata){
	//allocate mem for table
	_rows=(pathdata[0]<<NodeTypeBitLen) + pathdata[1];
	_cols=pathdata[2];

	int rows=_rows;
	int cols=_cols;

	_all_pathes=(NodeType**)operator new(_rows*sizeof(NodeType*));
	

	for(int i=0;i<rows;i++){
		NodeType* innerArr=(NodeType*)operator new (sizeof(NodeType)*cols);

		//copy each node ..faster
		for(int j=0;j<cols;j++){
			innerArr[j]=pathdata[i*cols+j+3];
		}
				
		_all_pathes[i]=innerArr;
	}
	
}

//Create PathTable from the result of algebra operation
void PathTable::CreateTablefromResult(const char* tabname,int rows,int cols,const vector<NodeType*>& pathdata){
	//set rows,cols
	_rows=rows;
	_cols=cols;

	//allocate mem for table
	_all_pathes=(NodeType**)operator new(rows*sizeof(NodeType*));
	
	for(int i=0;i<rows;i++){
		_all_pathes[i]=(NodeType*)pathdata[i];
	}

}

//output table content to screen
void PathTable::printTable(){
	cout<<"===========\n";
	cout<<"table name:"<< _name.c_str()<<"\n";
	for(int i=0;i<_rows;i++){
		for(int j=0;j<_cols;j++)
			//cout<< nodes[2+i*cols+j]<< " ";
			cout<<((NodeType*)(_all_pathes[i]))[j]<<" ";
		cout<<"\n";
	}
	cout<<"===========\n";
}

//output table content to given file
void PathTable::print2file(FILE* fp,const char* graphname){
	//FILE* fp=fopen(fname,"a");
	assert(fp);
	
	fprintf(fp,"\nGraph index: %s\tSubgraph matches: %d\n",graphname,_rows);
	
	for(int i=0;i<_rows;i++){
		//fprintf(fp,"%s| ",graphname);
		//fprintf(fp,"%d: ",i);
		for(int j=0;j<_cols;j++)
			//cout<< nodes[2+i*cols+j]<< " ";
			//cout<<((NodeType*)(_all_pathes[i]))[j]<<" ";
			fprintf(fp,"%d ",((NodeType*)(_all_pathes[i]))[j]);
		//cout<<"\n";
		fprintf(fp,"\n");
	}	
}

////////////////////////////////////////////
//
// Class: PathTableWithDegree. Use the Defination in algebra.h
// Usage: This table has the same functionality as PathTable.
//		  It uses map data-structure to store the different degree but same lable-path data.
//		  A PathTableWithDegree contains many PathTables which with different degrees.
//
/////////////////////////////////////////////

//Destructor
PathTableWithDegree::~PathTableWithDegree(){		
	
	//free all the PathTables in this map
	typedef map<string,PathTable*> Map_pathtab;
	Map_pathtab::iterator it=(*_MapOfTables).begin(),
		it_end=(*_MapOfTables).end();
	
	while(it!=it_end){
		delete (*it).second;
		it++;
	}
	
	//free map
	delete _MapOfTables;
};

//Insert one Label-Path record into this table. 
//Parameter 1 is its degree information of each node in this path
//Parameter 1 is the data of this path
void PathTableWithDegree::insert(string& strdegree,NodeType* node_arr){
	typedef map<string,PathTable*> Map_pathtab;
	Map_pathtab::iterator it=(*_MapOfTables).find(strdegree),
						  it_end=(*_MapOfTables).end();
	
	if( it!=it_end){		//Insert into exising position
		(*it).second->insertData(node_arr);	
	}else{					//if not exist in current table, insert new slot
		PathTable *myPathTab=new PathTable(strdegree,_LP);
		myPathTab->insertData(node_arr);
		
		(*_MapOfTables).insert(Map_pathtab::value_type(strdegree,myPathTab));
		_section_count++;
	}	
	
	_record_count++;
	
}

//Output this table as an NodeType Array
// The data in the array are:
//----------------------
//starting 4 positions    -- Global Header
// pos[0]: 		Section count. Each section is a different degree.
// pos[1]:    	Length-Path of this table
// pos[2-3]: 	Total rows count in this table
//----------------------
//following each section  -- Section Header
// pos[0-1]:		rows
// pos[2-(2+LP)]:	degree array
// pos[(3+LP)- ]:	data array
		
NodeType* PathTableWithDegree::outputData(){
	typedef map<string,PathTable*> Map_pathtab;
	Map_pathtab::iterator it=(*_MapOfTables).begin(),
		it_end=(*_MapOfTables).end();
	
	
	int sec_header=2+_LP;
	
	//start header size   = 1(section_number) + 2(rows) + 1(cols)
	//section header size = sec_header* _section_count
	//total data size	  = _record_count*_LP
	
	_outputdatasize=4+ sec_header* _section_count+ (_record_count*_LP);
	
	NodeType* ret_arr=new NodeType[_outputdatasize];
	
	//Global header
	ret_arr[0]= _section_count;
	ret_arr[1]= _LP;
	ret_arr[2]= _record_count>>NodeTypeBitLen;
	ret_arr[3]= _record_count<<NodeTypeBitLen>>NodeTypeBitLen;
	
	//Test of shifting
	//cout<< "record: "<< _record_count<< " hi: "<<ret_arr[1] <<" lo: "<< ret_arr[2]<<endl;
	//int tmp= (ret_arr[1]<<NodeTypeBitLen) + ret_arr[2];
	//cout << "hi: "<<(ret_arr[1]<<NodeTypeBitLen)<<" lo: "<< ret_arr[2]<<"recover: "<<tmp<<endl;
	
	//Testing 
	//cout<< "sec_count:" <<ret_arr[0]<<" ";
	//cout<< "cols: "<< ret_arr[1]<<" ";
	//cout<< "total_rows: "<< _record_count<<endl;
	
	int count=0;
	int currentpos=4;
	
	while(it!=it_end){
		PathTable* tmpPathTab=(*it).second;
		
		NodeType* degreeArr=sep_DegreeString((*it).first,_LP);
		
		int rows=tmpPathTab->get_rows();
		int cols=tmpPathTab->get_cols();
		
		//copying rows 
		ret_arr[currentpos]= rows>>NodeTypeBitLen;
		ret_arr[currentpos+1]= rows<<NodeTypeBitLen>>NodeTypeBitLen;
		
		//copying degree array
		for(int i=0;i<_LP;i++){
			ret_arr[currentpos+2+i]= degreeArr[i];
		}
		
		delete degreeArr;
		
		//copying data
		currentpos += 2+_LP;
		for(int m=0;m<rows;m++){
			for(int n=0;n<cols;n++){
				ret_arr[currentpos+m*cols+n]=(NodeType)tmpPathTab->get_alldata()[m][n];
			}
		}
		
		currentpos +=rows*cols;
		count++;
		it++;
	}
	
	if(_section_count!=count)
		cout<< "total section:" << count<<endl;
	//Check if there's error happened in size
	if(currentpos!=_outputdatasize)
		cout<< "error in size"<<endl;
	
	return ret_arr;
}

/////////////////////////////////////////////////////
//
// Class: OPInterSect --Responsible for doing "INTERSECT" operation
//        Parameters: 1. The op string  2. The opfile pointer
//
/////////////////////////////////////////////////////
OPInterSect::OPInterSect(string inStr,OPFile* opfile){
	_in=inStr;	//record the op string 
	vector<string> params=sep_words(_in);	//seprate the paramters
	str_out_tab=params[1];	//out table
	str_in1_tab=params[2];	//join-table1
	i_joint1=atoi(((const char*)params[3].c_str()));	//joint node position for table1
	str_in2_tab=params[4];	//join-table2
	i_joint2=atoi(((const char*)params[5].c_str()));	//joint node position for table2
	_opfile=opfile;		//setting the opfile pointer
}

void OPInterSect::print(){
	cout<<_in.c_str();	
}

void OPInterSect::execute(){
	//Loading the tables from BerkeleyDB into DB	
	PathTable &tab1=*(_opfile->loadTable(_opfile->getTableName(str_in1_tab.c_str()) ));
	PathTable &tab2=*(_opfile->loadTable(_opfile->getTableName(str_in2_tab.c_str()) ));
	//setting result table's column size
	int r_cols=tab1.get_cols()+tab2.get_cols()-1;

	//testing output table
	//tab1.printTable();
	//tab2.printTable();

	//getting the row, col information of the join tables
	int t1_row=tab1.get_rows();
	int t2_row=tab2.get_rows();
	int t1_col=tab1.get_cols();
	int t2_col=tab2.get_cols();
	
	if(t1_row==0||t2_row==0)
		throw GraphGrepExcp("empty table\n"); 

	//using vector to store the pointers of result rows
	vector<NodeType*> vtmp;
	vtmp.reserve((t1_row*t2_row)/4);
	NodeType** pN1=tab1.get_alldata();	
	NodeType** pN2=tab2.get_alldata();	
	
	//PathTable ret_tab;
	for(int i=0;i<t1_row;i++){
		for(int j=0;j<t2_row;j++){
			NodeType key1=pN1[i][i_joint1];
			NodeType key2=pN2[j][i_joint2];

			//if matched intersect point, do intersection(memory copy)
			if(key1==key2){
				NodeType* innerArr=(NodeType*)operator new (sizeof(NodeType)*r_cols);
				for(int m=0;m<t1_col;m++){
					innerArr[m]=pN1[i][m];
				}
				for(int n=0;n<t2_col;n++){
					if(n!=i_joint2 && n<i_joint2)	
						innerArr[n+t1_col]=pN2[j][n];
					else if(n!=i_joint2 && n>i_joint2)
						innerArr[n+t1_col-1]=pN2[j][n];
				}

				vtmp.push_back(innerArr);
				//test output
				//for(int x=0;x<r_cols;x++)
				//	cout<< innerArr[x] <<" ";
				//cout<<"\n";
				//cout.flush();
			}
		}
	}

	if(OUTPUT_DATA_COUNT)
		cout<<"record count: "<< vtmp.size()<<"\n";

	//insert the result into table pool
	_opfile->addTable(new PathTable((const char*)str_out_tab.c_str(),vtmp.size(),r_cols,vtmp));
	//release non-used tables
	_opfile->releaseTable(_opfile->getTableName(str_in2_tab.c_str()) );
	_opfile->releaseTable(_opfile->getTableName(str_in1_tab.c_str()) );
}

/////////////////////////////////////////////////////
//
// Class: OPTestUnique
//
/////////////////////////////////////////////////////
OPTestUnique::OPTestUnique(string inStr,OPFile* opfile){
	_in=inStr;
	vector<string> params=sep_words(_in);
	str_out_tab=params[1];
	str_in_tab =params[2];

	i_pos=(int*)malloc(sizeof(int)*(params.size()-3));
	pos=params.size()-3;
	for(int i=2;i<params.size()-1;i++){
		//vpos.push_back(atoi((const char*)params[i+1].c_str()));
		i_pos[i-2]=atoi((const char*)params[i+1].c_str());
	}

	_opfile=opfile;
}

void OPTestUnique::print(){
	cout<<_in.c_str();
	
}

void OPTestUnique::execute(){	
	PathTable &in_tab=*(_opfile->loadTable(_opfile->getTableName(str_in_tab.c_str()) ));

	if(i_pos[0]==-1){	//not doing anything, speciall case
		_opfile->addTable(new PathTable(str_out_tab,in_tab));
		_opfile->releaseTable(_opfile->getTableName(str_in_tab.c_str()) );
		return;
	}

	//result table's column size
	int r_cols=in_tab.get_cols();

	//testing output table
	//in_tab.printTable();
	//tab2.printTable();

	int t_row=in_tab.get_rows();

	if(t_row==0)
		throw GraphGrepExcp("empty table\n"); 


	//using vector to store the pointers of result rows
	vector<NodeType*> vtmp;
	vtmp.reserve(t_row/4);
	NodeType** pN=in_tab.get_alldata();
	
	for(int i=0;i<t_row;i++){
		NodeType* cmparray=(NodeType*)operator new (sizeof(NodeType)*r_cols);
		memcpy(cmparray,in_tab.get_data(i),sizeof(NodeType)*r_cols);

		//start comparing position
		bool found=false;

		for(int p=0;p<pos;p++){
			int curpos=i_pos[p];
			int xx=pN[i][0];
			NodeType key=pN[i][curpos];

			for(int i=0;i<r_cols;i++){
				if(key==cmparray[i] && i!=curpos){
					found=true;
					break;
				}
			}			
		}

		if(!found){
			NodeType* innerArr=(NodeType*)operator new (sizeof(NodeType)*r_cols);
			memcpy(innerArr,in_tab.get_data(i),sizeof(NodeType)*r_cols);
			vtmp.push_back(innerArr);
		}

		operator delete(cmparray);
	}

	if(OUTPUT_DATA_COUNT)
		cout<<"record count: "<< vtmp.size()<<"\n";
	
	_opfile->addTable(new PathTable((const char*)str_out_tab.c_str(),vtmp.size(),r_cols,vtmp));
	_opfile->releaseTable(_opfile->getTableName(str_in_tab.c_str()) );
}

/////////////////////////////////////////////////////
//
// Class: OPSelect
//
/////////////////////////////////////////////////////
OPSelect::OPSelect(string inStr,OPFile* opfile){
	_in=inStr;
	vector<string> params=sep_words(_in);
	str_out_tab=params[1];
	str_in_tab =params[2];
	
	i_pos=(int*)malloc(sizeof(int)*(params.size()-3));
	pos=params.size()-3;
	for(int i=2;i<params.size()-1;i++){
		//vpos.push_back(atoi((const char*)params[i+1].c_str()));
		i_pos[i-2]=atoi((const char*)params[i+1].c_str());
	}
	_opfile=opfile;
	
}

void OPSelect::print(){
	cout<<_in.c_str();	
}

void OPSelect::execute(){
	//setclock("Select time",0);		
	PathTable &in_tab=*(_opfile->loadTable(_opfile->getTableName(str_in_tab.c_str()) ));

	//result table's column size
	int r_cols=in_tab.get_cols()-1;

	//testing output table
	//in_tab.printTable();
	//tab2.printTable();

	int t_row=in_tab.get_rows();
	if(t_row==0)
		throw GraphGrepExcp("empty table\n"); 

	//using vector to store the pointers of result rows
	vector<NodeType*> vtmp;
	vtmp.reserve(t_row);

	NodeType** pN=in_tab.get_alldata();
	int startP=i_pos[0];
	
	for(int i=0;i<t_row;i++){
		//start comparing position
		bool found=true;
		
		//start from first position
		NodeType curkey=pN[i][startP];
		for(int p=1;p<pos;p++){	//for each declared unique positions
			NodeType nextkey=pN[i][i_pos[p]];

			if(curkey!=nextkey){
				found=false;
				//test output
				//for(int m=0;m<r_cols;m++)
				//	cout<< in_tab.get_item(i,m) <<" ";
				//cout<<"\n";
				break;
			}
			
			curkey=nextkey;
		}

		if(found){
			NodeType* innerArr=(NodeType*)operator new (sizeof(NodeType)*r_cols);
			for(int m=0;m<r_cols+1;m++){
				if(m!=i_pos[1] && m<i_pos[1])
					innerArr[m]=pN[i][m];
				else if(m!=i_pos[1] && m>i_pos[1])
					innerArr[m-1]=pN[i][m];
			}
				vtmp.push_back(innerArr);

				//test output
				//for(int x=0;x<r_cols;x++)
				//	cout<< innerArr[x] <<" ";
				//cout<<"\n";
				//cout.flush();
			
		}
	}

	if(OUTPUT_DATA_COUNT)
		cout<<"record count: "<< vtmp.size()<<"\n";

	_opfile->addTable(new PathTable((const char*)str_out_tab.c_str(),vtmp.size(),r_cols,vtmp));
	_opfile->releaseTable(_opfile->getTableName(str_in_tab.c_str()) );
}

/////////////////////////////////////////////////////
//
// Class: OPCartesianProduct
//
/////////////////////////////////////////////////////
OPCartesianProduct::OPCartesianProduct(string inStr,OPFile* opfile){
	_in=inStr;
	vector<string> params=sep_words(_in);
	str_out_tab=params[1];
	str_in1_tab=params[2];
	str_in2_tab=params[3];
	_opfile=opfile;
}

void OPCartesianProduct::print(){
	cout<<_in.c_str();	
}

void OPCartesianProduct::execute(){
	//setclock("CartesianProduct time",0);		
	PathTable &tab1=*(_opfile->loadTable(_opfile->getTableName(str_in1_tab.c_str()) ));
	PathTable &tab2=*(_opfile->loadTable(_opfile->getTableName(str_in2_tab.c_str()) ));

	//result table's column size
	int r_cols=tab1.get_cols()+tab2.get_cols();

	//testing output table
	//tab1.printTable();
	//tab2.printTable();

	int t1_row=tab1.get_rows();
	int t2_row=tab2.get_rows();
	int t1_col=tab1.get_cols();
	int t2_col=tab2.get_cols();

	if(t1_row==0||t2_row==0)
		throw GraphGrepExcp("empty table\n"); 

	//using vector to store the pointers of result rows
	vector<NodeType*> vtmp;
	vtmp.reserve(t1_row*t2_row);

	NodeType** pN1=tab1.get_alldata();
	NodeType** pN2=tab2.get_alldata();

	//PathTable ret_tab;
	for(int i=0;i<t1_row;i++){
		for(int j=0;j<t2_row;j++){
			bool unique=true;
			for(int m=0;m<t1_col;m++){
				for(int n=0;n<t2_col;n++){
					NodeType key1=pN1[i][m];
					NodeType key2=pN2[j][n];
					if(key1==key2){
						unique=false;
						break;
					}
				}
				
				if(unique==false)
					break;
			}
			
			
			if(unique){
				NodeType* p=(NodeType*)operator new (sizeof(NodeType)*r_cols);
				NodeType* innerArr=p;
				memset((void*)innerArr,0,sizeof(NodeType)*t1_col+sizeof(NodeType)*t2_col);
				memcpy(p,tab1.get_data(i),sizeof(NodeType)*t1_col);
				p+=t1_col;
				memcpy(p,tab2.get_data(j),sizeof(NodeType)*t2_col);
				vtmp.push_back(innerArr);
			}
		}
	}
		
	if(OUTPUT_DATA_COUNT)
		cout<<"record count: "<< vtmp.size()<<"\n";
	
	_opfile->addTable(new PathTable((const char*)str_out_tab.c_str(),vtmp.size(),r_cols,vtmp));
	_opfile->releaseTable(_opfile->getTableName(str_in2_tab.c_str()) );
	_opfile->releaseTable(_opfile->getTableName(str_in1_tab.c_str()) );
}

/////////////////////////////////////////////////////
//
// Class: OPSpit
//
/////////////////////////////////////////////////////
OPSpit::OPSpit(string inStr,OPFile* opfile){
	_in=inStr;
	vector<string> params=sep_words(_in);
	str_in_tab =params[1];
	
	_opfile=opfile;
	
}

void OPSpit::print(){
	cout<<_in.c_str();	
}

void OPSpit::execute(){
	setclock("Spit time",0);		
	PathTable &in_tab=*(_opfile->loadTable(_opfile->getTableName(str_in_tab.c_str()) ));

	//result table's column size
	int r_cols=in_tab.get_cols();

	//testing output table	
	//in_tab.printTable();
	
	if(in_tab.get_rows()!=0){
		in_tab.print2file(_opfile->getOutFile(),_opfile->getCurrentGraphID().c_str());
		if(!QUIET_MODE)
			cout<<_opfile->getCurrentGraphID().c_str();
	}
}

