/*
  GraphBlast Algorithm
  graphbuild.cpp responsible for building the database of paths informatio
  2006(c) Modified by Diego Reforgiato Recupero
  2007(c) Modified by Dmitry Skripin
*/

#include "graphbuild.h"
#include "algebra.h"
#include "argedit.h"
#include "vf2_sub_state.h"
#include "vf_sub_state.h"
#include "vf_state.h"
#include "vf2_state.h"
#include "vf2_mono_state.h"
#include "vf_mono_state.h"
#include "match.h"


///////////////////////////////////////////////
//
//	Global variables
//
///////////////////////////////////////////////
/*

  FILTERING FROWNS + MATCHING VF [GF] : GFROWNS.
  FILTERING2 = false
  GRAPHGREP_VF = true
  ACC_VF = false

  FILTERING FROWNS + MATCHING VF [DG] : DAYGB.
  FILTERING2 = false
  GRAPHGREP_VF = true
  ACC_VF = true
  
  FILTERING GRAPHGREP + MATCHING VF [GB] : GRAPHBLAST.
  FILTERING2 = true
  GRAPHGREP_VF = true
  ACC_VF = true
  
  FILTERING GRAPHGREP + MATCHING GRAPHGREP [GG] : GRAPHGREP.
  FILTERING2 = true
  GRAPHGREP_VF = false
  ACC_VF = false

*/




listwildconnection *wildhead;
bool FILTERING2 = true;                 //CONTROLS FILTERING FROWNS OR GRAPHGREP
bool GRAPHGREP_VF = true;               //CONTROLS MATCHING GRAPHGREP OR VF
bool ACC_VF = true;                     //CONTROLS VF ACCELERATION
float totale = 0;
float tot_s;
bool QUERY_WITH_DEGREE=false;		//flag to control if querying with degree information
bool OUTPUT_DATA_COUNT=false;		//flag to control if showing the current matched record with each operation
bool QUIET_MODE=true;			//flag to control if showing the debug information
bool DEBUG_OPFILE=false;		//flag to control if showing the debug information of OPFILE

int DEFAULT_LP=4;			//declaration of the Default LP

int NUM_INTS=256;	 		//The Number of Intergers of a FingerPrint. Default is 256. Used in the Frowns Filtering process.
int INT_SIZE=sizeof(int)*8;		//The size of an int (in bits). Default is 32. Used in Frowns Filtering process.
int NUM_FP=400;				//The group size of fingerprints. Default is 4000. Used in Frowns Filtering process.
int MODE=0;                             // 0 for auto; 1 for manual
int METHOD=0;                           // 0 for gb; 1 for gg; 2 for gf; 3 for dg

const int NodeTypeBitLen=sizeof(NodeType)*8; //The size of an NodeType (in bits). The information is needed when storing data into BekerlyDB

////////////////////////////////////////////
//
// Class: GraphBuild
// Usage: Responsible for founding all the patterns and fingerprint of a graph and insert into DB
//
/////////////////////////////////////////////

class Point {
 public:
  char a[10];
  Point(char x[10]) {
    strcpy(this->a,x);
  }
};

class GraphBuild{
private:
  Db* _db;			//Pointer to the data DB
  Db* _db_fp;			//Pointer to the FP DB

  ifstream* fin;			//Input file stream pointer
  int graphCount;			//Counter that records the total graph processed
  unsigned int *tmp_fp;	        //Pointer temp u_int array which stored the set of FPs
  GraphADT* readNextGraph(int count);	//function reads graph from input dbfile and returns a graph pointer of the current graph
  
public:
  ~GraphBuild(){			//Destructor
    delete _db;
    if(fin)
      delete fin;				
  }
  GraphBuild(char* dbfile);	//Constructor
  int AutomaticLearning();
  void run();			//Starting GraphBuild
  void start_DB_Action(Map_PathTableWithDegree&);	//Starting storing data into BerkeleyDB(with degree information)
  void start_DB_Action(Map_PathTable&, int flag);	//Starting storing data into BerkeleyDBwithout degree information
  
  void add2FP_DB(unsigned int* fp,int graphIndex);//Inserting FP group data into db
  void add_FP(unsigned int*,int );						//Adding a FP to the fp group 
  
};

////////////////////////////////////////////
//
//	Implementation of GraphBuild Class
//
/////////////////////////////////////////////

GraphBuild::GraphBuild(char* dbfile):graphCount(0){	//Construtor 
  FILE *tmp;

  rmdir_recursive("graphblastdata");
  mkdir("graphblastdata",0777);

  tmp = fopen(".//graphblastdata//Data","w");
  fprintf(tmp,"%s",dbfile);
  fclose(tmp);
  
  fin=NULL;
  if(dbfile!=""){
    ifstream in(dbfile);
    if(!in){
      cout<<dbfile <<" doest not exist\n";
      exit(0);
    }
    
    fin=new ifstream(dbfile);	
  }
}

// try to understand the input dimensions and selecting the most appropriate matching method
int GraphBuild::AutomaticLearning() {
  int glob_3d=0; // number of 3d mesh graphs
  int num_graph=0;
  int num_labels=0;
  while(!fin->eof()){
    char line[51];
    fin->getline(line,50);
    if(line[0]=='#'){
      try{
	fin->getline(line,50); // number of node
	int numNode=atoi(line);
	vector<string> lab;

	for(int i=0;i<numNode;i++) {	//read node
	  fin->getline(line,50);
	  bool flag=false;
	  for(int k=0;k<lab.size();k++)
	    if(lab[k]==line) {
	      flag=true;
	      break;
	    }
	  if(flag==false) {
	    string linestr(line);
	    lab.push_back(linestr);
	  }
	}
	num_labels = lab.size();
	fin->getline(line,50); // number of edges
	int numEdge=atoi(line);
	int *T=NULL;
	T= (int *) calloc(sizeof(int),numNode);
	assert(T);
	for(int j=0;j<numEdge;j++) {

	  fin->getline(line,50);
	  string tmp(line);
	  vector<string> nodes=sep_words(tmp);

	  T[Edge(nodes[0],nodes[1]).v]++;
	  T[Edge(nodes[0],nodes[1]).w]++;
	}
	int _3d=0;
	int _3d1=0;
	for(int i=0;i<numNode;i++) {
	  if(T[i]>4)
	    _3d++;
	  if(T[i]>5)
	    _3d1++;
	}
	if(_3d>(int)(numNode/2) && _3d1>(int)(numNode/10))
	  glob_3d++;
	free(T);
	num_graph++;
      }
      catch(ReadGraphExcp ex) {
      }
    }
  }
  fin->clear();
  fin->seekg(0);
  if((num_graph==1 && num_labels>40) || (glob_3d > (int)(num_graph*2/3))) {
    return 1;
  }
  return 0;
}

// main method of GraphBuild class
void GraphBuild::run(){
  //open FP database
  
  time_t beg,end;
  
  if(FILTERING2 == false) { // FROWNS FILTERING
    char FP_DBFile[]="fingerprint.db";
    
    (void)remove(FP_DBFile);
    _db_fp=new Db(0,0);
    
    _db_fp->set_errpfx("GraphBuild: ");
    _db_fp->set_pagesize(1024);		// Page size: 1K. 
    
    _db_fp->set_cachesize(0, 32 * NUM_INTS*NUM_FP, 0);
    _db_fp->open(NULL, FP_DBFile, NULL, DB_BTREE, DB_CREATE, 0664);
  }
  
  //open Main DataBase
  char DBFile[]="graphdata.db";
  (void)remove(DBFile);
  _db=new Db(0,0);
  
  _db->set_error_stream(&cerr);
  _db->set_errpfx("GraphBuild: ");
  _db->set_pagesize(1024);		// Page size: 1K. 
  
  _db->set_cachesize(0, 32 * 1024, 0);
  _db->open(NULL, DBFile, NULL, DB_BTREE, DB_CREATE, 0664);
  
  if(FILTERING2==true) { // GRAPHGREP FILTERING
    char DBHASH[]="dbHASH.db";
    (void)remove(DBHASH);
    _dbHASH = new Db(0,0);
    _dbHASH->set_error_stream(&cerr);
    _dbHASH->set_errpfx("HASH: ");
    _dbHASH->set_pagesize(1024);		// Page size: 1K. 	
    _dbHASH->set_cachesize(0, 32 * 1024, 0);
    _dbHASH->open(NULL, DBHASH, NULL, DB_BTREE, DB_CREATE, 0664);
  }
  
  int count=0;
  
  char f_index[]="index.dat";
  FILE *fp_index=fopen(f_index,"w");
  GraphADT* g;
  
  while((g=readNextGraph(count))!=NULL){	//Read each graph from the input file
    
    cout << "\nReading Graph "<<count<<"...";
    
    //Process the given graph
    vDFS_GraphBuild<GraphADT> myDFS(*g,DEFAULT_LP);

    fprintf(fp_index,"gn%d: %s\n",count,g->getGraphName().c_str());
    
    setclock("#DFS time",1);
    
    //Get the data map of the graph
    if(QUERY_WITH_DEGREE){
      Map_PathTableWithDegree *myPathMap=myDFS.getMapDegreePathTable();
      setclock("Inheriting time into DB",0);
      start_DB_Action(*myPathMap);
      setclock("Inheriting time into DB",1);
      
    }else{
      beg = clock();
      Map_PathTable *myPathMap=myDFS.getMapPathTable();
      setclock("Inheriting time into DB",0);	
      start_DB_Action(*myPathMap,0);
      setclock("Inheriting time into DB",1);
      end = clock();
      totale += (float)(end-beg)/CLOCKS_PER_SEC;      
    }
    
    //Get the fingerprint of the graph
    if(FILTERING2==false) { // FROWNS FILTERING
      unsigned int* fp=myDFS.getFingerPrint(); // get the fingerprint
      
      int itmp=0;
      for(int i=0;i<NUM_INTS;i++){
	if(fp[i]!=0){
	  itmp++;
	}
      }
      if(!QUIET_MODE)
	printf("\nNumber of total pathes %d",itmp);
      
      
      add_FP(fp,count);
    }
    
    delete g;
    count++;
    if(mem>1000000000) { // database COMMIT when memory reachs this quantity
      beg = clock();
      if(FILTERING2==true) { // only during GRAPHGREP FILTERING
	cout << "\nWRITTEN mem:"<<mem;
	mem=0;
	for(int i=0;i<HASHP;i++) {
	  if(vet_map[i].get_cont_graph() > 0) {
	    int siz = 0;
	    int * st = NULL;
	    st = vet_map[i].Compose(i,siz);
	    char i_s[10];
	    strcpy(i_s,"");
	    sprintf(i_s,"%d",i);
	    
	    //cout << "i_s:" << i_s;
	    
	    Dbt key((char *)i_s,sizeof(char)*(strlen(i_s)+1));
	    Dbt data(st,siz*sizeof(int));
	    
	    _dbHASH->put(0, &key, &data, DB_NOOVERWRITE);
	    if(st!=NULL)
	      free(st);
	    vet_map[i].Free1();
	  }
	}
      }
      end = clock();
      time_to_remove += (float)(end-beg)/CLOCKS_PER_SEC;
    }
    cout <<"done";
  }
  cout << endl<<endl;
  beg = clock();
  if(FILTERING2==true) { // COMMIT when all graphs have been read
    for(int i=0;i<HASHP;i++) {
      if(vet_map[i].get_cont_graph() > 0) {
	int siz = 0;
	int * st = NULL;
	st = vet_map[i].Compose(i,siz);
	char i_s[10];
	strcpy(i_s,"");
	sprintf(i_s,"%d",i);
	
	//cout << "i_s:" << i_s;
	
	Dbt key((char *)i_s,sizeof(char)*(strlen(i_s)+1));
	Dbt data(st,siz*sizeof(int));
	
	_dbHASH->put(0, &key, &data, DB_NOOVERWRITE);
	if(st != NULL)
	  free(st);
	vet_map[i].Free1();
      }
    }
  }
  end = clock();
  time_to_remove += (float)(end-beg)/CLOCKS_PER_SEC;
  
  fclose(fp_index);
  cout<<"Number of graphs in DB: "<<count<<endl;   //output the total graphs processed

  if(FILTERING2==false)
    add2FP_DB(tmp_fp,count);	//insert the current FP group into DB
  
  
  fin->close();
  
  _db->close(0);
  if(FILTERING2==false)
    _db_fp->close(0);
  if(FILTERING2==true)
    _dbHASH->close(0);
}

// internal method to add the fingerprint
void GraphBuild::add_FP(unsigned int* fp,int graphIndex) {
  //counting the current position in the group of fp
  int base=graphIndex/NUM_FP;
  int remainder=graphIndex % NUM_FP;
  
  if( remainder != 0){	//insert FP in the right slot of the group
    unsigned int* pi=(tmp_fp+remainder*NUM_INTS);
    memcpy(pi,fp, NUM_INTS*sizeof(unsigned int));
  }else if(graphIndex==0){	//start a new group, setting the memory and copy the first FP
    tmp_fp=new unsigned int[NUM_FP*NUM_INTS];
    memset(tmp_fp,0,sizeof(unsigned int)*NUM_FP*NUM_INTS);
    memcpy(tmp_fp,fp, NUM_INTS*sizeof(unsigned int));
  }else{	//Already exceed the group size, insert into BerkeleyDB and start a new group
    add2FP_DB(tmp_fp,graphIndex);
    tmp_fp=new unsigned int[NUM_FP*NUM_INTS];
    
    //The new begin 
    memcpy(tmp_fp,fp, NUM_INTS*sizeof(unsigned int));
  }
}

//Insert 1 FP group into BerkeleyDB
void GraphBuild::add2FP_DB(unsigned int* fp,int graphIndex) {
  int base= graphIndex/NUM_FP;
  int remainder= graphIndex % NUM_FP;
  
  char mykey[8];
  //itoa(graphIndex,mykey,10);
  sprintf(mykey,"%d",graphIndex);
  
  Dbt key(mykey,strlen(mykey)+1);
  
  int graphs=(remainder==0 && base!=0)?NUM_FP:(remainder+1);
  Dbt data(fp,sizeof(unsigned int)*NUM_INTS*graphs );
  
  int ret = _db_fp->put(0, &key, &data, DB_NOOVERWRITE);
  
  if (ret == DB_KEYEXIST) {
    cout << "Key " << mykey << " already exists.\n";
  }	
  
  delete tmp_fp;
  
  //count++;
  //cout<<"total table count"<<count<<endl;
}

//Insert the patterns of a graph into BerkeleyDB(with Degree information)
void GraphBuild::start_DB_Action(Map_PathTableWithDegree& myMapPath) {
  Map_PathTableWithDegree::iterator it=myMapPath.begin(),
    it_end=myMapPath.end();
  
  int count=0;
  while(it!=it_end){
    char *mykey=new char[(*it).first.size()+1];
    strcpy(mykey,(*it).first.c_str());
    
    Dbt key(mykey, strlen(mykey)+1);
    
    NodeType* int_arr=(*it).second->outputData();
    int size=(*it).second->getOutputdatasize();
    
    Dbt data(int_arr,size*sizeof(NodeType));
    
    int ret = _db->put(0, &key, &data, DB_NOOVERWRITE);
    
    if (ret == DB_KEYEXIST) {
      cout << "Key " << mykey << " already exists.\n";
    }	
    
    delete mykey;
    delete int_arr;
    
    ++it;
    count++;
  }
  
  //cout<<"total table count"<<count<<endl;
}

//Insert the patterns of a graph into BerkeleyDB(without Degree information)
void GraphBuild::start_DB_Action(Map_PathTable& myMapPath, int flag) {
  Map_PathTable::iterator it=myMapPath.begin(),
    it_end=myMapPath.end();
  
  int count=0;
  while(it!=it_end){
    char *mykey=new char[(*it).first.size()+1];
    strcpy(mykey,(*it).first.c_str());
    Dbt key(mykey, strlen(mykey)+1);
    
    int rows=(*it).second->get_rows();
    int cols=(*it).second->get_cols();
    
    
    // The data in the array are:
    //----------------------
    //starting 4 positions    -- Global Header
    // pos[0]: 		Section count. Each section is a different degree.
    // pos[1]:    	Length-Path of this table
    // pos[2-3]: 	Total rows count in this table
    //----------------------
    //following each section  -- Section Header
    // pos[0-1]:	rows
    // pos[2-(2+LP)]:	degree array
    // pos[(3+LP)- ]:	data array
    
    int sec_header=2+cols;
    //start header size   = 1(section_number) + 1(cols)+ 2(rows)
    //section header size = sec_header* _section_count
    //total data size	  = _record_count*_LP
    
    int size=4+ + sec_header*1 + rows*cols;
    NodeType* int_arr=new NodeType[size];
    
    //global header
    int_arr[0]= 1;
    int_arr[1]= cols;
    int_arr[2]= rows>>NodeTypeBitLen;
    int_arr[3]= rows<<NodeTypeBitLen>>NodeTypeBitLen;
    
    //section header
    int_arr[4]= int_arr[2];
    int_arr[5]= int_arr[3];
    
    //copying degree array
    for(int p=0;p<cols;p++){
      int_arr[6+p]= 0;
    }
    
    for(int i=0;i<rows;i++)
      for(int j=0;j<cols;j++)
	int_arr[6+cols+i*cols+j]=(NodeType)(*it).second->get_item(i,j);
    
		
    Dbt data(int_arr,size*sizeof(NodeType));
    
    int ret;
    if(flag==0)
      ret = _db->put(0, &key, &data, DB_NOOVERWRITE);
    
    if (ret == DB_KEYEXIST) {
      cout << "Key " << mykey << " already exists.\n";
    }
    
    delete mykey;
    delete int_arr;
    
    ++it;
    count++;
  }

  //cout<<"total table count"<<count<<endl;
}

//function reads graph from input dbfile and returns a graph pointer of the current graph
GraphADT* GraphBuild::readNextGraph(int count){
  ifstream& f=*fin;
  GraphADT* retG=NULL;
  char line[51];
  string name;
  int *T;
  char *wri=NULL;
  while(!f.eof()){
    f.getline(line,50);
    if(line[0]=='#'){
      try{
	char* p=line+1;
	name=string(p);	//set name of this graph
	
	f.getline(line,50);
	int numNode=atoi(line);
	retG=new GraphADT(numNode,false);
	retG->setGraphName(name);
	
	//setting table name
	char tabname[MAX_TABLENAME];
	sprintf(tabname,"gn%d",graphCount++);
	retG->setName(tabname);
	for(int i=0;i<numNode;i++){	//read node
	  f.getline(line,50);
	  (*retG).addNode(i,line);
	  /* Save in wri the nodes of the graph just read */
	  if(ACC_VF == true) {
	    int len = 0;
	    if(wri != NULL)
	      len = strlen(wri);
	    wri = (char *)realloc(wri,len + sizeof(char)*(strlen(line)+2));
	    if(i==0)
	      strcpy(wri,line);
	    else
	      strcat(wri,line);
	    strcat(wri,"\n");
	  }
	  /* */
	}
	
	f.getline(line,50);
	int numEdge=atoi(line);
	T= (int *) calloc(sizeof(int),numNode*numNode);
	assert(T);
	
	for(int j=0;j<numEdge;j++){
	  f.getline(line,50);
	  string tmp(line);
	  vector<string> nodes=sep_words(tmp);
	  if(nodes.size()<2)
	    throw ReadGraphExcp(graphCount);
	  
	  (*retG).insert(Edge(nodes[0],nodes[1]));
	  T[numNode*Edge(nodes[0],nodes[1]).v+Edge(nodes[0],nodes[1]).w]=1;
	  T[numNode*Edge(nodes[0],nodes[1]).w+Edge(nodes[0],nodes[1]).v]=1;
	}
	
	FILE *gn=NULL;
	char str[100];
	sprintf(str,".//graphblastdata//gn%d.sp",count);
	gn = fopen(str,"w+"); 
	assert(gn);
	fprintf(gn,"%d",numNode);
	fwrite(T,sizeof(int)*numNode*numNode,1,gn); 
	fclose(gn); 
	free(T);
	
	/* Writing in gn%d.node the nodes of each graph */
	if(ACC_VF == true) {
	  sprintf(str,".//graphblastdata//gn%d.node",count);
	  gn = fopen(str,"w+");
	  assert(gn);
	  fprintf(gn,"%d\n",numNode); 
	  fprintf(gn,"%s",wri);
	  fclose(gn);
	  free(wri);
	}
	/* */
	
	return retG;
      }catch(ReadGraphExcp ex){
	graphCount--;
	delete retG;
	
	cout<< "error reading graph:" <<ex.getID()<<endl;
      }
    }
  }
  return retG;
}

////////////////////////////////////////////
//
//	Main Function
//
/////////////////////////////////////////////
int main(int argc, char** argv) {
  
  try {
    setclock("Total running time",0);			
    
    bool HasDBFile=false;
    GraphBuild* app=NULL;
    
    //parse the input parameters
    for (int n = 1; n <argc; n++) {
      if (strcmp(argv[n],"-f")==0) {
	app=new GraphBuild(argv[++n]);
	HasDBFile=true;
      }else if(strcmp(argv[n],"-degree")==0) {
	QUERY_WITH_DEGREE=true;
      }else if(strcmp(argv[n],"-v")==0) {
	QUIET_MODE=false;
      }else if(strcmp(argv[n],"-lp")==0) {
	DEFAULT_LP=atoi(argv[++n]);
      }else if(strcmp(argv[n],"-fp")==0) {
	NUM_INTS=atoi(argv[++n]);
      }
      else if(strcmp(argv[n],"-mode")==0) {
	char *st = argv[++n];
	if(strcmp(st,"man")==0)
	  MODE = 1;
	else
	  MODE = 0;
      }
      else if(strcmp(argv[n],"-select")==0) {
	char *st = argv[++n];
	if(strcmp(st,"gb")==0)
	  METHOD = 0;
	if(strcmp(st,"gg")==0)
	  METHOD = 1;	
	if(strcmp(st,"gf")==0)
	  METHOD = 2;	
	if(strcmp(st,"dg")==0)
	  METHOD = 3;
      }
      else {
	cout<< "Usage: GraphBuild -f graphDBfile [-degree -lp lengthpath -fp length of FP -mode mode_type -select method]\n"<<
	  "\t-degree            query with degree\n"<<
	  "\t-lp lengthpath     input different LP, default is 4\n"<<
	  "\t-fp length of FP   input different length of FP, default is 256"<<
	  "\t-mode mode_type    auto for automatic, man for manual, default is auto"<<
	  "\t-select method     gb for graphblast, default"<<
	  "\t                   gg for graphgrep"<<
	  "\t                   gf for gfrowns"<<
	  "\t                   dg for daygb"<<
	  "\t                   Active only if mode is manual"<<endl;	
	exit(2);
      }
    }  // end for(....)
    
    if(app!=NULL){	//starting GraphBuild
      int value = 0;
      FILE *tmp_auto; 
      remove(".//method.dat");
      tmp_auto = fopen(".//method.dat","w");
      switch(MODE) {
      case 0: //automatic mode
	cout << "\nAutomatic Learning...";
	value = app->AutomaticLearning();
	if(value==0)
	  fprintf(tmp_auto,"gb");
	else
	  fprintf(tmp_auto,"gg");
	cout << "done";
	if(value==0) {
	  FILTERING2 = true;
	  GRAPHGREP_VF = true;
	  ACC_VF = true;
	}
	else {
	  FILTERING2 = true;
	  GRAPHGREP_VF = false;
	  ACC_VF = false;
	}
	break;
      case 1: // manual mode
	switch(METHOD) {
	case 0:
	  FILTERING2 = true;
	  GRAPHGREP_VF = true;
	  ACC_VF = true;
	  fprintf(tmp_auto,"gb");
	  break;
	case 1:
	  FILTERING2 = true;
	  GRAPHGREP_VF = false;
	  ACC_VF = false;
	  fprintf(tmp_auto,"gg");
	  break;
	case 2:
	  FILTERING2 = false;
	  GRAPHGREP_VF = true;
	  ACC_VF = false;
	  fprintf(tmp_auto,"gf");
	  break;
	case 3:
	  FILTERING2 = false;
	  GRAPHGREP_VF = true;
	  ACC_VF = true;
	  fprintf(tmp_auto,"dg");
	  break;
	}
      }
      fclose(tmp_auto);
      app->run();
      setclock("Total running time",1);
      printtime("Inheriting time into DB");
      printtime("Total running time");
      cout << endl;
      //      cout << "Total VFV to subtract " << totale+time_to_remove << endl;
      delete app;
      
      //char ch;
      //cin>> ch;
    }else{
      cout<< "Usage: GraphBuild -f graphDBfile [-degree -lp lengthpath -fp length of FP -mode mode_type -select method]\n"<<
	  "\t-degree            query with degree\n"<<
	  "\t-lp lengthpath     input different LP, default is 4\n"<<
	  "\t-fp length of FP   input different length of FP, default is 256\n"<<
	  "\t-mode mode_type    auto for automatic, man for manual, default is auto\n"<<
	  "\t-select method     gb for graphblast, default\n"<<
	  "\t                   gg for graphgrep\n"<<
	  "\t                   gf for gfrowns\n"<<
	  "\t                   dg for daygb\n"<<
	  "\t                   Active only if mode is manual"<<endl;
    }
    
    return (EXIT_SUCCESS);
  }
  catch (DbException &dbe) {
    cerr << "GraphBuild: " << dbe.what() << "\n";
    return (EXIT_FAILURE);
  }
}
