/***************************************************************************
 *   Project: GraphBlast                                                   *
 *   File: Hash Table for indices                                          *
 *   Date: 13-May-2005                                                     *
 ***************************************************************************/

#ifndef HTABLE_H
#define HTABLE_H

#include "common.h"

#define MAX_V_NUM 32767

/////////////////////////////////////////////////////////////////////////
//  HList                                                              //
/////////////////////////////////////////////////////////////////////////
typedef struct HList
{
  string pattern;
  string key_pattern;
  int acc_index;
  HList *next;
};

/////////////////////////////////////////////////////////////////////////
//  HTable - hash table with "this key pattern is alredy getted" acc.  //
/////////////////////////////////////////////////////////////////////////
class HTable
{
   int query_number;
   int hsize;
   vector<HList*> table;
   vector<short int> acc_table;
   map<string,int> key_to_int;

public:
   // constructor (int: hash size)
   explicit HTable(int size):hsize(size),table(size),query_number(1)
   {
       for(int i; i<table.size(); i++)
          table[i] = NULL;
   }
   
   // destructor
   ~HTable()
   {
        free_table();
   }
   
   // initialize new query
   void InitNewQuery()
   {
       if(++query_number == MAX_V_NUM)
       {
           // reset acceleration array
           acc_table.assign(acc_table.size(),0);
           query_number = 1;
       }
   }
   
   // load table from file.
   // file format: (int: pattern_size, char[]: pattern, int: key_pattern_size, char[]: key_pattern)
   void Load(char* filename)
   {
       FILE *fidx;
       fidx = fopen(filename,"rb");
       char *pattern, *kpattern;
       int rig_all;
//#define MH_DEBUG     
       if(fidx == NULL) cout << "Can't find indices file" << endl;

       FILE *fni= fopen(".//graphblastdata//num_ind.dat","rb");
       fread(&rig_all,sizeof(int),1,fni);
       fclose(fni);


#ifdef MH_DEBUG
       ofstream os("blast_table.txt");
#endif
       
       acc_table.clear();
       key_to_int.clear();
       
       //while(!feof(fidx))
       for(int c=0;c<rig_all;c++)
       {
          // read pairs (pattern, key_pattern) from file                
          int length;  
          fread(&length,sizeof(int),1,fidx);
          pattern = new char[length+1];
          fread(pattern,sizeof(char),length,fidx);
          pattern[length]='\0';
          fread(&length,sizeof(int),1,fidx);
          kpattern = new char[length+1];
          fread(kpattern,sizeof(char),length,fidx);
          kpattern[length]='\0';
#ifdef MH_DEBUG
          os << pattern << "==" << kpattern << endl;
#endif
          
          // insert to table
          insert(string(pattern),string(kpattern));
          
          // free buffers
          delete[] pattern;
          delete[] kpattern;
       }
       fclose(fidx);
#ifdef MH_DEBUG
       os.close();
#endif
       key_to_int.clear();
   }
   
   string get(string& pattern)
   {
     HList* hl = table[hash(pattern)];
     if(hl==NULL) 
        return string("");
     
     do{
       if(hl->pattern == pattern) break;
     }while((hl=hl->next)!=NULL); 
     
     if(hl==NULL) 
        return string("");
     
     if(acc_table[hl->acc_index] == query_number)
        return string(""); // key pattern was already taken in this query
     else{
        acc_table[hl->acc_index] = query_number;
        return hl->key_pattern;
     }

   }

private:
   // hash function
   int hash(string pattern)
   {
      int h = 0;
      for(int j=0;j<pattern.length();j++) {
    	h = (64*h + pattern[j])%hsize;
      }
      return h;
   }
   
   // isnsert to table
   void insert(string pattern, string key_pattern)
   {
        // insert to table
        int h = hash(pattern);
        HList* new_hl;
        
        new_hl = new HList;
        if(table[h] == NULL){
            table[h] = new_hl; 
        }
        else{
            HList* hl = table[h];
            while(hl->next != NULL) hl=hl->next;
            hl->next = new_hl;
        }
        new_hl->pattern = pattern;
        new_hl->key_pattern = key_pattern;
        new_hl->next = NULL;
        
        // acc. table
        map<string,int>::iterator it = key_to_int.find(key_pattern);
        if(it != key_to_int.end()) {
            new_hl->acc_index = it->second;
        }
        else{
            acc_table.push_back(0);
            new_hl->acc_index = acc_table.size()-1;
            key_to_int.insert(make_pair(key_pattern,acc_table.size()-1));
        }
        
   }
   
   // free table
   void free_table()
   {
       for(int i=0; i<hsize; i++)
       {
           HList* hln;
           HList* hl = table[i];
           while(hl != NULL)
           {
               hln = hl->next;
               delete hl;
               hl = hln;
           }
       }
   }
  
};

#endif
