function [Anew,bnew,flag] = preproc(A,b,rkthresh)
% PREPROC   checks for consistency and redundancy of the system Ax = b
%
% [Anew,bnew,flag] = preproc(A,b,rkthresh)
%
% Here A is an mxn matrix, and b is an m-vector.
% Uses QR factorization and a value rkthresh to estimate the rank
% (e.g. rkthresh = 1.0e-8).
% On exit,
% if flag = 0, A was not rank deficient and Anew = A, bnew = b.
% if flag = 1, A was rank deficient, but the system was consistent;
%              the rows of Anew form a linearly independent susbset of
%              the rows of A, and bnew is the corresponding entries of b.
% if flag = 2, A was rank deficient and the system was also inconsistent.
%              No more processing is done.  Anew = A and bnew = b.

% SDPPACK Version 0.8 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton
% Last modified: 3/24/97

 flag = 0;
 if issparse(A)
    sparseblocks = 1;
 else
    sparseblocks = 0;
 end;
 A = full(A);
 [m,n] = size(A);
 [Q,R,colperm] = qr(A');  % recall: A'*colperm = Q*R
 Rdim = min(size(R));
 mrank = 0;
 idx = 1;
 while abs(R(idx,idx)) > rkthresh,
    mrank = mrank + 1;
    idx = idx + 1;
    if idx > Rdim,
       break;
    end;
 end;
 if mrank < m,
    flag = 1;   % rank deficient
    fprintf('preproc: A is rank deficient\n');
    fprintf('m = %3.0f  rank = %3.0f  tol = %3.1e\n',m,mrank,rkthresh);
    R1 = R(1:mrank,1:mrank);
    bt = colperm'*b;
    b1 = bt(1:mrank);
    b2 = bt(mrank+1:m);
    v1 = R1'\b1;
    S1 = R(1:mrank,mrank+1:m);
    t = norm(S1'*v1 - b2);
    if t > 10*rkthresh,
       flag = 2;   % rank deficient and inconsistent
       fprintf('preproc: System is inconsistent: Norm of residual = %3.11e\n',t);
    end;
 else,
    fprintf('preproc: System has maximal rank %d and is consistent\n',mrank);
 end;
%
% now eliminate the redundant rows of A and replace b by b1 and update m
%
 if flag == 1,
    m = mrank;
    Atmp = colperm'*A;
    A = Atmp(1:m,:);
    b = b1;
 end;
 if sparseblocks,
    Anew = sparse(A);
 else
    Anew = A;
 end;
 bnew = b;
%
% END function
