function [xhat,yhat] = getEigVecNormalized(x,y,normalization,...
    normalization_x_indx,normalization_y_indx,xhat0,x0,y0)
%
% normalize the right and left eigenvectors x and y (of the
% erturbed matrix) by normalization 1, 2, 3 or 4 as described in
% Greenbaum, Li, Overton Sec 3.4, or normalization 0, which refers to
% the analytic family x(t), y(t) defined in eq. (25), Sec 3.3. 
%
% normalizations 1 and 2 require normalization_x_indx, while only
% normalization 1 reqires normalization_y_indx
%
% normalization 3 requires xhat0 since there is a choice of sign in the
% normalization, and we need to choose the one that makes xhat close to xhat0
%
% normalization 0 requires x0 and y0 as they are part of the definition of 
% the analytic functions x(t), y(t)
%

%
if normalization == 1 % simplest possible
    % normalization 1:  x_j = 1, y_k = 1, where j,k are defined by the 
    % 4th and 5th input args
    j = normalization_x_indx;
    k = normalization_y_indx;
    xhat = x/x(j); % so x(j)=1
    yhat = y/y(k); % so y(k)=1 (but y'*x is no longer one)
elseif normalization == 2
    % normalization 2:  x_j = 1, y'*x = 1
    j = normalization_x_indx;
    scalfac = x(j);
    xhat = x/scalfac; % so x(1) = 1
    yhat = y*(scalfac'); % so y'*x is still 1
elseif normalization == 3
    % normalization 3: x.'*x = 1 (x^T x, NOT the norm), y'*x = 1 
    scalfac = sqrt(x.'*x); % NOT the norm, could be zero
    xhat = x/scalfac; % so x.'*x = 1
    yhat = y*(scalfac'); % so y'*x is still 1
    % there are two solutions: the other is xhat = -x/scalfac, yhat = -y/scalfac
    % we need to use the one that ensures xhat is close to x0
    if nargin > 5 % xhat0 is not passed when this is called the first time with x = x0
        [~,j] = max(abs(real(xhat0))); % not both the real and imag parts can be small
        if abs(real(xhat0(j))) > abs(imag(xhat0(j)))
            if sign(real(xhat(j))/real(xhat0(j))) ~= 1
                xhat = -xhat;
                yhat = -yhat;
            end
        else
            if sign(imag(xhat(j))/imag(xhat0(j))) ~= 1
                xhat = -xhat;
                yhat = -yhat;
            end
        end
    end
elseif normalization == 4; % NOT UNIQUE
    % normalization 4: x'*x = ||x||^2 = 1, y'*x = 1 
    scalfac = norm(x);
    xhat = x/scalfac; % so ||x|| = 1
    yhat = y*(scalfac'); % so y'*x is still 1, but could multiply scalfac
    % by any unimodular quantity exp(i*theta) and still satisfy constraints
elseif normalization == 0 % use the analytic formulas for x, y', based on
    % the eigenprojector and the base eigenvectors x0, y0', as described
    % at the end of Section 3.3 of our paper
    %
    P = x*y';  % the eigenprojector: note that y'*x = 1
    y0Px0 = y0'*P*x0;
    s = sqrt(y0Px0);
    xhat = (1/s)*P*x0;
    yhat = (1/s')*P'*y0;
    % this should be equivalent:
    scale = sqrt((y'*x0)/(y0'*x));
    xhat2 = scale*x;
    yhat2 = y/scale'; % reciprocal of conjugate of scale
else
    error('invalid normalization')
end